﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/dd.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Device.h>

#include <nnt.h>
#include <nnt/gtest/gtest.h>

#include "../Parameter.h"
#include "../../Common/UsbDsInterface.h"
#include "../../Common/UsbTestUtil.h"

using namespace nn;
using namespace nn::usb;
using namespace nn::usb::test;

namespace nnt {
namespace usb {
namespace ds {

#define ADVANCE(i) (((i) + 1) % DsLimitRingSize)

class Ring {
public:
    Ring() : head(0), tail(0)
    {
        // do nothing
    }

    bool IsEmpty()
    {
        return head == tail;
    }

    bool IsFull()
    {
        return ADVANCE(head) == tail;
    }

    bool Put(uint32_t urbId)
    {
        if (IsFull())
        {
            return false;
        }

        id[head] = urbId;

        head = ADVANCE(head);

        return true;
    }

    bool Get(uint32_t *pUrbId)
    {
        if (IsEmpty())
        {
            return false;
        }

        *pUrbId = id[tail];

        tail = ADVANCE(tail);

        return true;
    }

    uint8_t* PeekHead()
    {
        return buffer[head];
    }

    uint8_t* PeekTail()
    {
        return buffer[tail];
    }

private:
    uint32_t id[DsLimitRingSize];
    NN_USB_DMA_ALIGN uint8_t buffer[DsLimitRingSize][DataSize + 4096];
    uint32_t head;
    uint32_t tail;
};

uint32_t seed = 1;

NN_USB_DMA_ALIGN uint8_t readBuffer[DataSize + 4096];
NN_USB_DMA_ALIGN uint8_t writeBuffer[DataSize + 4096];
NN_USB_DMA_ALIGN uint8_t syncBuffer[DataSize + 4096];

Ring ring;
UrbReport report;

class DsPerformanceTest : public ::testing::TestWithParam<uint32_t>{};

INSTANTIATE_TEST_CASE_P(
    PerformanceTest, DsPerformanceTest, ::testing::ValuesIn(BlockSize)
);

TEST_P(DsPerformanceTest, BulkWriteSync)
{
    uint32_t totalWrote = 0;
    uint32_t blockSize  = GetParam();
    DsEndpoint& ep = UsbDsInterface::m_EndPoints[0][0];

    memset(syncBuffer, 0, sizeof(syncBuffer));
    memset(writeBuffer, 0, sizeof(writeBuffer));

    // prepare data
    MakeGaloisPattern(writeBuffer, DataSize, seed++);

    while (totalWrote < DataSize)
    {
        uint32_t bytesWrote = 0;
        uint32_t remain = DataSize - totalWrote;
        uint32_t size = remain > blockSize ? blockSize : remain;

        memcpy(syncBuffer, writeBuffer + totalWrote, size);

        NNT_ASSERT_RESULT_SUCCESS(
            ep.PostBuffer(&bytesWrote, syncBuffer, size)
        );

        ASSERT_EQ(bytesWrote, size);

        totalWrote += size;
    }

    for(int i=0; i<4096; i++)
    {
        ASSERT_EQ(syncBuffer[DataSize + i], 0);
        ASSERT_EQ(readBuffer[DataSize + i], 0);
    }

    ASSERT_EQ(totalWrote, DataSize);
}

TEST_P(DsPerformanceTest, BulkReadSync)
{
    uint32_t totalRead = 0;
    uint32_t blockSize = GetParam();
    DsEndpoint& ep = UsbDsInterface::m_EndPoints[0][1];
    memset(syncBuffer, 0, sizeof(syncBuffer));
    memset(readBuffer, 0, sizeof(readBuffer));

    while (totalRead < DataSize)
    {
        uint32_t bytesRead = 0;
        uint32_t remain = DataSize - totalRead;
        uint32_t size = remain > blockSize ? blockSize : remain;

        NNT_ASSERT_RESULT_SUCCESS(
            ep.PostBuffer(&bytesRead, syncBuffer, size)
        );

        ASSERT_EQ(bytesRead, size);

        memcpy(readBuffer + totalRead, syncBuffer, size);
        totalRead += size;
    }

    for(int i=0; i<4096; i++)
    {
        ASSERT_EQ(syncBuffer[DataSize + i], 0);
        ASSERT_EQ(readBuffer[DataSize + i], 0);
    }

    ASSERT_EQ(totalRead, DataSize);

    // check data
    ASSERT_TRUE(CheckGaloisPattern(readBuffer, DataSize, seed++));
}

#ifdef ENABLE_ASYNC_TEST

TEST_P(DsPerformanceTest, BulkWriteAsync)
{
    uint32_t totalWrote = 0;
    uint32_t totalVerified = 0;
    uint32_t blockSize  = GetParam();
    DsEndpoint& ep = UsbDsInterface::m_EndPoints[0][0];
    nn::os::SystemEventType *pCompletionEvent = ep.GetCompletionEvent();

    // statistic
    uint32_t submitCount = 0;
    uint32_t queryCount  = 0;

    // prepare data
    MakeGaloisPattern(writeBuffer, DataSize, seed++);

    while (totalVerified < DataSize)
    {
        // keep posting until full or done
        while (!ring.IsFull() && totalWrote < DataSize)
        {
            uint32_t urbId;
            uint32_t remain = DataSize - totalWrote;
            uint32_t size = remain > blockSize ? blockSize : remain;
            uint8_t* buffer = ring.PeekHead();

            memcpy(buffer, writeBuffer + totalWrote, size);

            NNT_ASSERT_RESULT_SUCCESS(
                ep.PostBufferAsync(&urbId, buffer, size)
            );

            totalWrote += size;

            ASSERT_TRUE(ring.Put(urbId));

            submitCount++;
        }

//        NN_LOG("[PERF:I:A] =========== WaitSystemEvent ============\n");
        // wait for completion
        nn::os::WaitSystemEvent(pCompletionEvent);
        nn::os::ClearSystemEvent(pCompletionEvent);
//        NN_LOG("[PERF:I:A] =========== pCompletionEvent DONE ============\n");

        // get the report ...
        NNT_ASSERT_RESULT_SUCCESS(ep.GetUrbReport(&report));
        queryCount++;

        // ... and then check the status
        for (unsigned i = 0; i < report.count; i++)
        {
            ASSERT_NE(report.report[i].status, UrbStatus_Invalid);
            ASSERT_NE(report.report[i].status, UrbStatus_Cancelled);
            ASSERT_NE(report.report[i].status, UrbStatus_Failed);

            if (report.report[i].status == UrbStatus_Finished)
            {
                uint32_t id;

                ASSERT_TRUE(ring.Get(&id));

                ASSERT_EQ(report.report[i].id,            id);
                ASSERT_EQ(report.report[i].requestedSize, report.report[i].transferredSize);

                totalVerified += report.report[i].transferredSize;
                continue;
            }

            ASSERT_GE(i, 0);
            break;
        }
    }

    NN_LOG("[PERF:I:A] =========== Statictic ============\n");
    NN_LOG("[PERF:I:A] SubmitCount: %d\n", submitCount);
    NN_LOG("[PERF:I:A] QueryCount : %d\n", queryCount);
    NN_LOG("[PERF:I:A] Ratio      : %f\n", (float)queryCount / (float)submitCount);
    NN_LOG("[PERF:I:A] =========== Statictic ============\n");
}


TEST_P(DsPerformanceTest, BulkReadAsync)
{
    uint32_t totalRead = 0;
    uint32_t totalVerified = 0;
    uint32_t blockSize  = GetParam();
    DsEndpoint& ep = UsbDsInterface::m_EndPoints[0][1];
    nn::os::SystemEventType *pCompletionEvent = ep.GetCompletionEvent();

    // statistic
    uint32_t submitCount = 0;
    uint32_t queryCount  = 0;

    while (totalVerified < DataSize)
    {
        // keep posting until full or done
        while (!ring.IsFull() && totalRead < DataSize)
        {
            uint32_t urbId;
            uint32_t remain = DataSize - totalRead;
            uint32_t size = remain > blockSize ? blockSize : remain;
            uint8_t* buffer = ring.PeekHead();

            NNT_ASSERT_RESULT_SUCCESS(
                ep.PostBufferAsync(&urbId, buffer, size)
            );

            totalRead += size;

            ASSERT_TRUE(ring.Put(urbId));

            submitCount++;
        }

//        NN_LOG("[PERF:O:A] =========== WaitSystemEvent ============\n");
        // wait for completion
        nn::os::WaitSystemEvent(pCompletionEvent);
        nn::os::ClearSystemEvent(pCompletionEvent);
//        NN_LOG("[PERF:O:A] =========== pCompletionEvent DONE ============\n");

        // get the report ...
        NNT_ASSERT_RESULT_SUCCESS(ep.GetUrbReport(&report));
        queryCount++;

        // ... and then check the status
        for (unsigned i = 0; i < report.count; i++)
        {
            ASSERT_NE(report.report[i].status, UrbStatus_Invalid);
            ASSERT_NE(report.report[i].status, UrbStatus_Cancelled);
            ASSERT_NE(report.report[i].status, UrbStatus_Failed);

            if (report.report[i].status == UrbStatus_Finished)
            {
                uint32_t id;
                uint8_t* buffer = ring.PeekTail();

                ASSERT_TRUE(ring.Get(&id));

                ASSERT_EQ(report.report[i].id,            id);
                ASSERT_EQ(report.report[i].requestedSize, report.report[i].transferredSize);

                memcpy(readBuffer + totalVerified, buffer, report.report[i].transferredSize);

                totalVerified += report.report[i].transferredSize;
                continue;
            }

            ASSERT_GE(i, 0);
            break;
        }
    }

    // check data
    ASSERT_TRUE(CheckGaloisPattern(readBuffer, DataSize, seed++));

    NN_LOG("[PERF:O:A] =========== Statictic ============\n");
    NN_LOG("[PERF:O:A] SubmitCount: %d\n", submitCount);
    NN_LOG("[PERF:O:A] QueryCount : %d\n", queryCount);
    NN_LOG("[PERF:O:A] Ratio      : %f\n", (float)queryCount / (float)submitCount);
    NN_LOG("[PERF:O:A] =========== Statictic ============\n");
}
#endif

} // end of namespace ds
} // end of namespace usb
} // end of namespace nnt

