﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/usb0/ds/usb_DsClientInterface.h>
#include <nn/sm/sm_UserApi.h>

using namespace nn;
using namespace nn::svc;
using namespace nn::usb0;
using namespace nn::usb0::ds;


// Instantiate the maximum number of clients
DsClientInterface usbDevClient[DsClientInterface::m_maxClients];

NN_ALIGNAS(4096) uint8_t testData[4 * 1024 * 1024];
NN_ALIGNAS(4096) uint8_t readData[4 * 1024 * 1024];
NN_ALIGNAS(4096) uint8_t writeData[4 * 1024 * 1024];

const size_t StackSize = 64 * 1024;
NN_ALIGNAS(4096) uint8_t readThreadStack[StackSize];
NN_ALIGNAS(4096) uint8_t writeThreadStack[StackSize];

size_t g_Size;

void ReadThread(void* argument)
{
    //nn::os::SleepThread(nn::TimeSpan::FromSeconds(5));

    const size_t* pSize = reinterpret_cast<size_t*>(argument);
    const size_t size = *pSize;

    NN_LOG("Started reading %d bytes.\n", size);
    Result result = usbDevClient[0].PostBuffer(0x01, readData, size, NULL); // receive data from host

    if( result.IsSuccess() )
    {
        NN_LOG("Succeed to read %d bytes.\n", size);
    }
    else
    {
        NN_LOG("Failed to read %d bytes.\n", size);
    }
}

void WriteThread(void* argument)
{
    const size_t* pSize = reinterpret_cast<size_t*>(argument);
    const size_t size = *pSize;

    NN_LOG("Start to write %d bytes.\n", size);
    Result result = usbDevClient[0].PostBuffer(0x81, writeData, size, NULL); // receive data from host

    if( result.IsSuccess() )
    {
        NN_LOG("Succeed to write %d bytes.\n", size);
    }
    else
    {
        NN_LOG("Failed to write %d bytes.\n", size);
    }
}

void ConcurrentReadWrite(size_t size)
{
    nn::os::ThreadType readThread;
    nn::os::ThreadType writeThread;

    g_Size = size;

    nn::os::CreateThread(&readThread, ReadThread, &g_Size, readThreadStack, StackSize, nn::os::DefaultThreadPriority);
    nn::os::CreateThread(&writeThread, WriteThread, &g_Size, writeThreadStack, StackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&readThread);
    nn::os::StartThread(&writeThread);

    nn::os::WaitThread(&readThread);
    nn::os::WaitThread(&writeThread);

    nn::os::DestroyThread(&readThread);
    nn::os::DestroyThread(&writeThread);
}

extern "C" void nninitStartup()
{
}

extern "C" int nnMain()
{
    Result result;

    NN_LOG("+++ START: Initializing ???.\n");
    result = nn::sm::Initialize();
    NN_ABORT_UNLESS(result.IsSuccess());

    // Initialize client interfaces
    NN_LOG("+++ START: Initializing client interfaces.\n");
    result = usbDevClient[0].Initialize();
    NN_ABORT_UNLESS(result.IsSuccess());

    NN_LOG("+++ START: adding string...\n");
    uint8_t str_interface1[USB_MAX_INTERFACE_DESCRIPTION_SIZE]  = {22, UsbStringDescriptorType, 'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0, 'a', 0, 'c', 0, 'e', 0, '1', 0};
    uint8_t str_index_interface1;

    result = usbDevClient[0].AddString((UsbStringDescriptor*)str_interface1, &str_index_interface1);
    NN_ABORT_UNLESS(result.IsSuccess());

    NN_LOG("+++ START: adding interface...\n");
    UsbEndpointDescriptor epDescs1[2] =
    {
        // Audio endpoint descriptors differ in size.
        // use UsbEndpointDescriptorSize or USB_DT_ENDPOINT_AUDIO_SIZE for bLength
        {
            UsbEndpointDescriptorSize,                           // bLength;
            UsbEndpointDescriptorType,                           // bDescriptorType;
            NN_USB_ENDPOINT_ADDRESS(1, UsbEndpointDirection_ToHost),   // bEndpointAddress;
            UsbEndpointAttributeMask_XferTypeBulk,               // bmAttributes;
            512,                                                 // wMaxPacketSize;
            0                                                    // bInterval;
        },

        {
            UsbEndpointDescriptorSize,                           // bLength;
            UsbEndpointDescriptorType,                           // bDescriptorType;
            NN_USB_ENDPOINT_ADDRESS(1, UsbEndpointDirection_ToDevice), // bEndpointAddress;
            UsbEndpointAttributeMask_XferTypeBulk,               // bmAttributes;
            512,                                                 // wMaxPacketSize;
            0                                                    // bInterval;
        }
    };

    UsbInterfaceDescriptor ifDesc1 =
    {
        UsbInterfaceDescriptorSize,                              // bLength;
        UsbInterfaceDescriptorType,                              // bDescriptorType;
        0,                                                       // bInterfaceNumber;
        0,                                                       // bAlternateSetting;
        2,                                                       // bNumEndpoints;
        0xff,                                                    // bInterfaceClass;
        0xff,                                                    // bInterfaceSubClass;
        0xff,                                                    // bInterfaceProtocol;
        str_index_interface1                                     // iInterface;
    };

    result = usbDevClient[0].RegisterInterface(&ifDesc1, epDescs1, 2);
    NN_ABORT_UNLESS(result.IsSuccess());

    NN_LOG("+++ START: bulk loop back...\n");

    // host has to send and receive these buffers for this to continue
    ConcurrentReadWrite(1);
    ConcurrentReadWrite(2);
    ConcurrentReadWrite(3);
    ConcurrentReadWrite(511);
    ConcurrentReadWrite(512);
    ConcurrentReadWrite(513);
    ConcurrentReadWrite(4095);
    ConcurrentReadWrite(4096);
    ConcurrentReadWrite(4097);
    ConcurrentReadWrite(20 * 1024);
    ConcurrentReadWrite(15 * 20 * 1024 - 1);
    ConcurrentReadWrite(15 * 20 * 1024);
    ConcurrentReadWrite(15 * 20 * 1024 + 1);
    ConcurrentReadWrite(2 * 15 * 20 * 1024 + 1);
    ConcurrentReadWrite(3 * 15 * 20 * 1024 + 1);
    ConcurrentReadWrite(4 * 15 * 20 * 1024 - 1);
    ConcurrentReadWrite(4 * 15 * 20 * 1024);
    ConcurrentReadWrite(4 * 1024 * 1024);
    ConcurrentReadWrite(256 * 1024);

    NN_LOG("+++ START: UnregisterInterface...\n");
    result = usbDevClient[0].UnRegisterInterface(&ifDesc1);
    NN_ABORT_UNLESS(result.IsSuccess());

    NN_LOG("+++ START: DelString...\n");
    result = usbDevClient[0].DelString(str_index_interface1);
    NN_ABORT_UNLESS(result.IsSuccess());

    NN_LOG("+++ START: Finalizing client interfaces.\n");
    result = usbDevClient[0].Finalize();
    NN_ABORT_UNLESS(result.IsSuccess());

    return 0;
}


