﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>
#include <nn/nn_Result.h>
#include <nn/usb/usb_Device.h>
#include <nnt/gtest/gtest.h>

#include "../../Common/UsbDsInterface.h"
#include "../../Common/UsbTestUtil.h"

using namespace nn;
using namespace nn::usb;
using namespace nn::usb::test;

NN_USB_DMA_ALIGN uint8_t readData[4 * 1024 * 1024];
NN_USB_DMA_ALIGN uint8_t writeData[4 * 1024 * 1024];

const size_t StackSize = 64 * 1024;
NN_ALIGNAS(4096) uint8_t readThreadStack[StackSize];
NN_ALIGNAS(4096) uint8_t writeThreadStack[StackSize];

size_t g_Size;

uint32_t read_transferCycle = 1; //used to change data pattern every transfer

void ReadThread(void * argument)
{
    const size_t* pSize = reinterpret_cast<size_t*>(argument);
    const size_t size =  *pSize;
    uint32_t bytesRead = 0;

    //prepare data
    MakeGaloisPattern(readData, (uint32_t)size, read_transferCycle++);

    NN_LOG("Start reading %d bytes. \n", size);
    //                                       (uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes)
    Result result = UsbDsInterface::m_EndPoints[0][0].PostBuffer(&bytesRead, readData, size);
    if (result.IsSuccess())
    {
        NN_LOG("Succeed to read %d bytes\n", bytesRead);
    }
    else
    {
        NN_LOG("Failed to read %d bytes\n", bytesRead);
        NN_LOG("Module: %d Description: %d\n", result.GetModule(), result.GetDescription());
    }
}

uint32_t write_transferCycle = 1; //used to change data pattern every transfer

void WriteThread(void* argument)
{
    const size_t* pSize = reinterpret_cast<size_t*>(argument);
    const size_t size =  *pSize;
    uint32_t bytesWrote = 0;

    NN_LOG("Start to write %d bytes. \n", size);
    //                                       (uint32_t *pOutBytesTransferred, void *buffer, uint32_t bytes)
    Result result = UsbDsInterface::m_EndPoints[0][1].PostBuffer(&bytesWrote, writeData, size);
    if (result.IsSuccess())
    {
        uint8_t* pWriteData = &writeData[0];
        NN_LOG("Succeed to write %d bytes\n", bytesWrote);
        if (size < 4 * 1024 * 1024)
        {
            NN_LOG("WriteData[%02x", *pWriteData);
            pWriteData++;
            for (int i = 0; i < 16 && i < size; i++)
            {
                NN_LOG(", %02x", *pWriteData);
                pWriteData++;
            }
            NN_LOG("]\n");
        }
    }
    else
    {
        NN_LOG("Failed to write %d bytes\n", bytesWrote);
        NN_LOG("Module: %d Description: %d\n", result.GetModule(), result.GetDescription());
    }
}

void ConcurrentReadWrite(size_t size)
{
    nn::os::ThreadType readThread;
    nn::os::ThreadType writeThread;

    g_Size = size;

    nn::os::CreateThread(&readThread, ReadThread, &g_Size, readThreadStack, StackSize, nn::os::DefaultThreadPriority);
    nn::os::CreateThread(&writeThread, WriteThread, &g_Size, writeThreadStack, StackSize, nn::os::DefaultThreadPriority);

    nn::os::StartThread(&readThread);
    nn::os::StartThread(&writeThread);

    nn::os::WaitThread(&readThread);
    nn::os::WaitThread(&writeThread);

    nn::os::DestroyThread(&readThread);
    nn::os::DestroyThread(&writeThread);
}

class UsbDeviceStackTest : public ::testing::TestWithParam<int>{};

/*
    SIGLONTD-8148 - Test is DISABLED as of 8/31
    Windows tries to send 32 bytes then 0 when SDEV is connected to PC
    Might be Target Manager doing this automatically
    Possible Solution: Create unique VID/PID GUID for device stack test
*/
INSTANTIATE_TEST_CASE_P(ConcurrentReadWriteTests, UsbDeviceStackTest, testing::ValuesIn(testValues));

uint32_t transferCycle = 1;

TEST_P(UsbDeviceStackTest, ConcurrentReadWrite)
{
    ConcurrentReadWrite(GetParam());
    NN_LOG("transferCycle: %d\n", transferCycle);
    EXPECT_TRUE(CheckGaloisPattern(writeData, (uint32_t)GetParam(), transferCycle++));
}
