﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "UsbTestPcUtil.h"

namespace nnt {
namespace usb {

const GUID UsbDsTestEnvironment::TARGET_GUID = {
        0x61A954B0, 0xB7F1, 0x4333,{ 0xAF, 0xBD, 0x6A, 0x41, 0x76, 0x9C, 0xE2, 0x66 }
};

bool UsbDevice::Initialize(const GUID *pGuid)
{
    HDEVINFO deviceInfoHandle = SetupDiGetClassDevs(
        pGuid, NULL, NULL, DIGCF_PRESENT | DIGCF_DEVICEINTERFACE
    );

    if (!deviceInfoHandle)
    {
        NN_LOG("[Host PC] device information was not found (%d)\n", GetLastError());
        return false;
    }

    SP_DEVICE_INTERFACE_DATA interfaceData = { 0 };
    interfaceData.cbSize = sizeof(SP_DEVICE_INTERFACE_DATA);

    DWORD dwIndex = 0;

    while (NN_STATIC_CONDITION(true))
    {
        if (SetupDiEnumDeviceInterfaces(deviceInfoHandle, NULL, pGuid,
                                        dwIndex, &interfaceData))
        {
            break;
        }
        else
        {
            if (GetLastError() == ERROR_NO_MORE_ITEMS)
            {
                return false;
            }
        }

        dwIndex++;
    }

    ULONG interfaceDetailDataLength = 256;

    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData, NULL,
                                         0, &interfaceDetailDataLength, NULL))
    {
        if (GetLastError() != ERROR_INSUFFICIENT_BUFFER)
        {
            return false;
        }
    }

    ::std::unique_ptr<char[]> buffer(new char[interfaceDetailDataLength]);
    auto pInterfaceDetailData = reinterpret_cast<PSP_DEVICE_INTERFACE_DETAIL_DATA>(buffer.get());
    pInterfaceDetailData->cbSize = sizeof(SP_DEVICE_INTERFACE_DETAIL_DATA);

    if (!SetupDiGetDeviceInterfaceDetail(deviceInfoHandle, &interfaceData,
                                         pInterfaceDetailData, interfaceDetailDataLength,
                                         &interfaceDetailDataLength, NULL))
    {
        return false;
    }

    deviceHandle = CreateFile(pInterfaceDetailData->DevicePath,
                              GENERIC_WRITE | GENERIC_READ,
                              FILE_SHARE_WRITE | FILE_SHARE_READ,
                              NULL,
                              OPEN_EXISTING,
                              FILE_ATTRIBUTE_NORMAL | FILE_FLAG_OVERLAPPED,
                              NULL);

    if (deviceHandle == INVALID_HANDLE_VALUE)
    {
        return false;
    }

    if (!WinUsb_Initialize(deviceHandle, &winUsbHandle))
    {
        CloseHandle(deviceHandle);
        return false;
    }

    USB_INTERFACE_DESCRIPTOR interfaceDescriptor;
    WinUsb_QueryInterfaceSettings(winUsbHandle, 0, &interfaceDescriptor);
    for (int i = 0; i < interfaceDescriptor.bNumEndpoints; i++)
    {
        WINUSB_PIPE_INFORMATION pipeInfo;
        while (!WinUsb_QueryPipe(winUsbHandle, 0, static_cast<uint8_t>(i), &pipeInfo))
        {
            Sleep(10);
        }
        if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_IN(pipeInfo.PipeId))
        {
            pipeBulkIn = pipeInfo.PipeId;
        }
        else if (pipeInfo.PipeType == UsbdPipeTypeBulk && USB_ENDPOINT_DIRECTION_OUT(pipeInfo.PipeId))
        {
            pipeBulkOut = pipeInfo.PipeId;
        }
    }

    if (!pipeBulkIn)
    {
        NN_LOG("[Host PC] BULK IN pipe was not found\n");
        return false;
    }

    if (!pipeBulkOut)
    {
        NN_LOG("[Host PC] BULK OUT pipe was not found\n");
        return false;
    }

    interfaceIndex = interfaceDescriptor.bInterfaceNumber;

    return true;
}

void UsbDevice::Finalize()
{
    WinUsb_Free(winUsbHandle);
    CloseHandle(deviceHandle);
}

void UsbDevice::ReceiveDataSync(unsigned char *buffer, uint32_t size, uint32_t blockSize)
{
/*
    NN_UNUSED(blockSize);

    ULONG bytesTransferred = 0;

    ASSERT_EQ(
                WinUsb_WritePipe(
                    winUsbHandle,
                    pipeBulkIn,
                    buffer,
                    size,
                    &bytesTransferred,
                    NULL
                    ),
                TRUE
                );

    ASSERT_EQ(bytesTransferred, size);
*/

    uint32_t bytesRecv = 0;

    while (bytesRecv < size)
    {
        ULONG bytesTransferred = 0;
        uint32_t remain = size - bytesRecv;
        uint32_t trunkSize = remain > blockSize ? blockSize : remain;

        ASSERT_EQ(
            WinUsb_ReadPipe(winUsbHandle,
                            pipeBulkIn,
                            buffer + bytesRecv,
                            trunkSize,
                            &bytesTransferred,
                            NULL),
            TRUE
        );

        // not an error to receive less than you want
        // ASSERT_EQ(bytesTransferred, trunkSize);

        bytesRecv += bytesTransferred;
    }
}

void UsbDevice::SendDataSync(uint8_t *buffer, uint32_t size, uint32_t blockSize)
{
/*
    NN_UNUSED(blockSize);

    ULONG bytesTransferred = 0;

    ASSERT_EQ(
                WinUsb_WritePipe(
                    winUsbHandle,
                    pipeBulkOut,
                    buffer,
                    size,
                    &bytesTransferred,
                    NULL
                    ),
                TRUE
                );

    ASSERT_EQ(bytesTransferred, size);
 */

    uint32_t bytesSend = 0;

    while (bytesSend < size)
    {
        ULONG bytesTransferred = 0;
        uint32_t remain = size - bytesSend;
        uint32_t trunkSize = remain > blockSize ? blockSize : remain;
/*
        ASSERT_EQ(
            // for some reason, if wMaxPacketSize of the EP is 1024, this call will
            // block forever and you won't even be able to kill the process...
            WinUsb_WritePipe(winUsbHandle,
                             pipeBulkOut,
                             buffer + bytesSend,
                             trunkSize,
                             &bytesTransferred,
                             NULL),
            TRUE
        );
*/

        if (
            WinUsb_WritePipe(winUsbHandle,
                             pipeBulkOut,
                             buffer + bytesSend,
                             trunkSize,
                             &bytesTransferred,
                             NULL) == false)
        {
            NN_LOG("GetLastError() retured %08x size %d trunkSize %d\n", GetLastError(), size, trunkSize);
        }
else
{
            NN_LOG("trunkSize %d\n", trunkSize);
}



        // all bytes must be send
        ASSERT_EQ(bytesTransferred, trunkSize);

        bytesSend += bytesTransferred;
    }
}

void UsbDevice::ReceiveDataAsync(uint8_t *buffer, uint32_t size, uint32_t blockSize)
{
    uint32_t bytesSubmitted = 0;
    uint32_t bytesReceived  = 0;

    ASSERT_TRUE(m_Ring.IsEmpty());

    ASSERT_LE(blockSize, m_Ring.MaxBytesPerTransaction);

    // fill the pipeline
    while (!m_Ring.IsFull() && bytesSubmitted < size)
    {
        uint32_t remain = size - bytesSubmitted;
        uint32_t chunk  = remain > blockSize ? blockSize : remain;

        WinUsb_ReadPipe(winUsbHandle,
                        pipeBulkIn,
                        m_Ring.GetHeadBuffer(),
                        chunk,
                        NULL,
                        m_Ring.GetHeadOverlap());

        m_Ring.Enqueue(chunk);

        bytesSubmitted += chunk;
    }

    while (bytesReceived < size)
    {
        ULONG bytesTransferred = 0;
        OVERLAPPED* pOverlap = m_Ring.GetTailOverlap();
        uint32_t expected = m_Ring.GetTailBytes();

        WaitForSingleObject(pOverlap->hEvent, INFINITE);

        WinUsb_GetOverlappedResult(winUsbHandle,
                                   pOverlap,
                                   &bytesTransferred,
                                   FALSE);


        // not an error to receive less than you want
        // ASSERT_EQ(bytesTransferred, expected);

        memcpy(buffer + bytesReceived, m_Ring.GetTailBuffer(), bytesTransferred);
        bytesReceived += bytesTransferred;

        // compensation
        bytesSubmitted -= expected - bytesTransferred;

        m_Ring.Dequeue();

        // refill the pipeline
        if (bytesSubmitted < size)
        {
            uint32_t remain = size - bytesSubmitted;
            uint32_t chunk  = remain > blockSize ? blockSize : remain;

            WinUsb_ReadPipe(winUsbHandle,
                            pipeBulkIn,
                            m_Ring.GetHeadBuffer(),
                            chunk,
                            NULL,
                            m_Ring.GetHeadOverlap());

            m_Ring.Enqueue(chunk);

            bytesSubmitted += chunk;
        }
    }

    ASSERT_TRUE(m_Ring.IsEmpty());
}

void UsbDevice::SendDataAsync(uint8_t *buffer, uint32_t size, uint32_t blockSize)
{
    uint32_t bytesSubmitted = 0;
    uint32_t bytesSend  = 0;

    ASSERT_TRUE(m_Ring.IsEmpty());

    // fill the pipeline
    while (!m_Ring.IsFull() && bytesSubmitted < size)
    {
        uint32_t remain = size - bytesSubmitted;
        uint32_t chunk = remain > blockSize ? blockSize : remain;

        WinUsb_WritePipe(winUsbHandle,
                         pipeBulkOut,
                         buffer + bytesSubmitted,
                         chunk,
                         NULL,
                         m_Ring.GetHeadOverlap());

        m_Ring.Enqueue(chunk);

        bytesSubmitted += chunk;
    }

    while (bytesSend < size)
    {
        ULONG bytesTransferred = 0;
        OVERLAPPED* pOverlap = m_Ring.GetTailOverlap();
        uint32_t expected = m_Ring.GetTailBytes();

        WaitForSingleObject(pOverlap->hEvent, INFINITE);

        WinUsb_GetOverlappedResult(winUsbHandle,
                                   pOverlap,
                                   &bytesTransferred,
                                   FALSE);

        // all bytes must be send
        ASSERT_EQ(bytesTransferred, expected);

        bytesSend += bytesTransferred;

        m_Ring.Dequeue();

        // refill the pipeline
        if (bytesSubmitted < size)
        {
            uint32_t remain = size - bytesSubmitted;
            uint32_t chunk = remain > blockSize ? blockSize : remain;

            WinUsb_WritePipe(winUsbHandle,
                             pipeBulkOut,
                             buffer + bytesSubmitted,
                             chunk,
                             NULL,
                             m_Ring.GetHeadOverlap());

            m_Ring.Enqueue(chunk);

            bytesSubmitted += chunk;
        }
    }

    ASSERT_TRUE(m_Ring.IsEmpty());
}

} // end of namespace usb
} // end of namespace nnt
