﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_Log.h>
#include <nn/nn_Result.h>

#include <nn/os.h>
#include <nn/usb/pd/usb_PdResult.h>
#include <nn/usb/pd/usb_PdCradle.h>
#include <nn/usb/usb_Host.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneTestUsbHubReset.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        // API を呼ぶ間隔 (frame)
        const int64_t UpdateInterval = 20;

        // QueryAllInterfaces 用
        const size_t MaxIfCount = 32;

        nn::usb::InterfaceQueryOutput g_IfBuffer[MaxIfCount];
        int32_t g_IfCount;

        nn::TimeSpan g_ApiExecutionTime;
        nn::Result g_ApiResult;

        nn::os::Tick g_SceneStart;

        nn::usb::pd::CradleSession g_Session;

        void Print() NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            ss << "API Result: " << ToString(g_ApiResult) << std::endl;
            ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;
            ss << std::endl;

            ss << "USB IF Count: " << g_IfCount << std::endl;
            ss << std::endl;

            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;

            ss << "Press A key to reset USB hub." << std::endl;
            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }

        void UpdateIfCount(nn::usb::Host *pHost) NN_NOEXCEPT
        {
            auto filter = nn::usb::InvalidDeviceFilter;
            NN_ABORT_UNLESS_RESULT_SUCCESS(pHost->QueryAllInterfaces(&g_IfCount, g_IfBuffer, sizeof(g_IfBuffer), &filter));
        }

        void ResetUsbHub() NN_NOEXCEPT
        {
            auto start = nn::os::GetSystemTick();

            nn::usb::pd::Vdo vdo;
            g_ApiResult = nn::usb::pd::SetCradleVdo(&g_Session, &vdo, nn::usb::pd::CradleVdmCommand_UsbHubReset);
            if (g_ApiResult.IsFailure())
            {
                NN_LOG("[error] CradleVdmCommand_UsbHubReset module:%d description:%d\n", g_ApiResult.GetModule(), g_ApiResult.GetDescription());
            }

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();
        }
    } // namespace

    void SceneTestUsbHubReset(SceneResult *pOutResult, nn::usb::Host *pHost)
    {
        g_SceneStart = nn::os::GetSystemTick();

        nn::usb::pd::OpenCradleSession( &g_Session );

        for(int64_t frame = 0;; frame++)
        {
            input::Update();
            Print();

            if (frame % UpdateInterval == 0)
            {
                UpdateIfCount(pHost);
            }

            // 決定ボタンで USB ハブリセット
            if (input::IsOkPushed())
            {
                ResetUsbHub();
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        nn::usb::pd::CloseCradleSession(&g_Session);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_Top;
    }
}}}
