﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <sstream>
#include <iomanip>

#include <nn/nn_Assert.h>
#include <nn/nn_Result.h>

#include <nn/init.h>
#include <nn/omm/omm_Api.h>
#include <nn/os.h>
#include <nn/psm/psm.h>
#include <nn/psm/psm_System.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/usb/pd/usb_Pd.h>
#include <nn/vi.private.h>
#include <nn/vi/vi_DisplayEvents.h>

#include "Draw.h"
#include "Input.h"
#include <nnt/usbPdUtil/testUsbPd_util.h>
#include "SceneGetStatus.h"
#include "ToString.h"

namespace nnt { namespace usb { namespace pd {

    namespace {
        // 1フレームの時間 (msec)
        const int64_t FrameLength = 50;

        // API を呼ぶ間隔 (frame)
        const int64_t UpdateInterval = 2;

        // QueryAllInterfaces 用
        const size_t MaxIfCount = 32;

        nn::TimeSpan g_ApiExecutionTime;
        nn::os::Tick g_SceneStart;

        nn::usb::pd::Session g_PdSession;
        nn::usb::pd::Status g_PdStatus;
        nn::usb::pd::Notice g_PdNotice;
        int32_t g_PdActiveNoticeCount;

        nn::omm::OperationMode g_OperationMode;

        nn::psm::ChargerType g_ChargerType;

        nn::vi::Display* g_pDisplay;
        nn::vi::HotplugState g_HotplugState;

        nn::usb::InterfaceQueryOutput g_IfBuffer[MaxIfCount];
        int32_t g_IfCount;

        template <typename T>
        const char* AssertResultToString(T expected, std::pair<bool, T> assertItem) NN_NOEXCEPT
        {
            if (!assertItem.first)
            {
                return "[--]";
            }
            else
            {
                return expected == assertItem.second ? "[OK]" : "[NG]";
            }
        }

        template <typename T>
        const char* AssertResultToStringRange(T expected, std::pair<bool, T> assertMinItem, std::pair<bool, T> assertMaxItem) NN_NOEXCEPT
        {
            if (!assertMinItem.first || !assertMaxItem.first)
            {
                return "[--]";
            }
            else
            {
                return assertMinItem.second <= expected && expected <= assertMaxItem.second ? "[OK]" : "[NG]";
            }
        }

        void PrintPdStatus(std::stringstream& ss, StateAssert& assert) NN_NOEXCEPT
        {
            {
                const char* result = AssertResultToString(g_PdStatus.IsActive(), assert.isActive);
                const char* data = ToString(g_PdStatus.IsActive());
                ss << result << " Active: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetPlugOrientation(), assert.plugOrientation);
                const char* data = ToString(g_PdStatus.GetPlugOrientation());
                ss << result << " PlugOrientation: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetDataRole(), assert.dataRole);
                const char* data = ToString(g_PdStatus.GetDataRole());
                ss << result << " DataRole: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetPowerRole(), assert.powerRole);
                const char* data = ToString(g_PdStatus.GetPowerRole());
                ss << result << " PowerRole: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.m_CurrentPdo.storage, assert.currentPdo);
                char s[10];
                std::sprintf(s, "%08x",g_PdStatus.m_CurrentPdo.storage);
                ss << result << " Current PDO (" << s << "): ";
                ss << g_PdStatus.m_CurrentPdo.GetVoltage() << " mV / ";
                ss << g_PdStatus.m_CurrentPdo.GetMaximumCurrent() << " mA" << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.m_CurrentRdo.storage, assert.currentRdo);
                char s[10];
                std::sprintf(s, "%08x",g_PdStatus.m_CurrentRdo.storage);
                ss << result << " Current RDO (" << s << "): ";
                ss << g_PdStatus.m_CurrentRdo.GetOperatingCurrent() << " mA / ";
                ss << g_PdStatus.m_CurrentRdo.GetMaximumOperatingCurrent() << " mA" << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsElectronicallyMarkedCable(), assert.isElectronicallyMarkedCable);
                const char* data = ToString(g_PdStatus.IsElectronicallyMarkedCable());
                ss << result << " IsElectronicallyMarkedCable: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetAccessoryMode(), assert.accessoryMode);
                const char* data = ToString(g_PdStatus.GetAccessoryMode());
                ss << result << " AccessoryMode: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsDisplayPortAlternateMode(), assert.isDisplayPortAlternateMode);
                const char* data = ToString(g_PdStatus.IsDisplayPortAlternateMode());
                ss << result << " IsDisplayPortAlternateMode: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsCradlePowerShortage(), assert.isCradlePowerShortage);
                const char* data = ToString(g_PdStatus.IsCradlePowerShortage());
                ss << result << " IsCradlePowerShortage: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsCradleWithUnofficialAcAdaptor(), assert.isCradleWithUnofficialAcAdaptor);
                const char* data = ToString(g_PdStatus.IsCradleWithUnofficialAcAdaptor());
                ss << result << " IsCradleWithUnofficialAcAdaptor: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsTableDockPowerShortage(), assert.isTableDockPowerShortage);
                const char* data = ToString(g_PdStatus.IsTableDockPowerShortage());
                ss << result << " IsTableDockPowerShortage: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.IsTableDockWithUnofficialAcAdaptor(), assert.isTableDockWithUnofficialAcAdaptor);
                const char* data = ToString(g_PdStatus.IsTableDockWithUnofficialAcAdaptor());
                ss << result << " IsTableDockWithUnofficialAcAdaptor: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetDeviceType(), assert.deviceType);
                const char* data = ToString(g_PdStatus.GetDeviceType());
                ss << result << " DeviceType: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetRequest(), assert.request);
                const char* data = ToString(g_PdStatus.GetRequest());
                ss << result << " Request: " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_PdStatus.GetError(), assert.error);
                const char* data = ToString(g_PdStatus.GetError());
                ss << result << " Error: " << data << std::endl;
            }
        }

        void Print(State state) NN_NOEXCEPT
        {
            nnt::usb::pd::draw::Clear();
            std::stringstream ss;

            // ヘッダを表示
            ss << "Test Item: " << ToString(state) << std::endl;
            ss << "API Execution Time: " << g_ApiExecutionTime.GetMicroSeconds() << " usec" << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            ss.str("");

            // テスト結果を表示（色つき）
            if (TestState(state, g_PdStatus, g_PdActiveNoticeCount, g_ChargerType, g_IfCount, g_HotplugState, g_OperationMode))
            {
                nnt::usb::pd::draw::Print("Test Result: OK", nnt::usb::pd::draw::Color_Green);
            }
            else
            {
                nnt::usb::pd::draw::Print("Test Result: NG", nnt::usb::pd::draw::Color_Red);
            }

            // 詳細表示
            auto pMap = GetStateAssertMap();
            StateAssert assert = (*pMap)[state];

            PrintPdStatus(ss, assert);

            {
                const char* result = AssertResultToStringRange(g_PdActiveNoticeCount, assert.activeNoticeCountMin, assert.activeNoticeCountMax);
                ss << result << " ActiveNoticeCount: " << g_PdActiveNoticeCount << std::endl;
            }

            {
                const char* result = AssertResultToString(g_ChargerType, assert.chargerType);
                const char* data = ToString(g_ChargerType);
                ss << result << " ChargerType (psm): " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_IfCount, assert.ifCount);
                ss << result << " IfCount (usb:hs): " << g_IfCount << std::endl;
            }

            {
                const char* result = AssertResultToString(g_HotplugState, assert.hotplugState);
                const char* data = ToString(g_HotplugState);
                ss << result << " HotplugState (vi): " << data << std::endl;
            }

            {
                const char* result = AssertResultToString(g_OperationMode, assert.operationMode);
                const char* data = ToString(g_OperationMode);
                ss << result << " OperationMode (omm): " << data << std::endl;
            }

            ss << std::endl;

            // フッタ表示
            float now = (nn::os::GetSystemTick() - g_SceneStart).ToTimeSpan().GetMilliSeconds() / 1000.0f;
            ss << "Time: " << std::fixed << std::setprecision(1) << now << std::endl;
            ss << std::endl;
            ss << "Press B key to go back." << std::endl;

            nnt::usb::pd::draw::Print(ss.str().c_str());
            nnt::usb::pd::draw::Draw();
        }

        void Update(nn::os::SystemEventType *pNoticeEvent, nn::usb::Host *pHost) NN_NOEXCEPT
        {
            auto start = nn::os::GetSystemTick();

            // usb::pd の Status と Notice と NoticeEvent を取得
            nn::usb::pd::GetStatus(&g_PdStatus, &g_PdSession);
            nn::usb::pd::GetNotice(&g_PdNotice, &g_PdSession);
            bool isEvent = nn::os::TryWaitSystemEvent(pNoticeEvent);
            nn::os::ClearSystemEvent(pNoticeEvent);

            if ( !g_PdStatus.IsActive() )
            {
                g_PdActiveNoticeCount = 0;
            }
            else if ( g_PdNotice.IsActiveNotice() && isEvent )
            {
                g_PdActiveNoticeCount++;
            }

            // 動作モード取得
            g_OperationMode = nn::omm::GetOperationMode();

            // 充電器種別取得
            g_ChargerType = nn::psm::GetChargerType();

            // Display port の HPD 信号取得
            NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::GetDisplayHotplugState(&g_HotplugState, g_pDisplay));

            // USB ホストが認識している Interface 数を取得
            auto filter = nn::usb::InvalidDeviceFilter;
            NN_ABORT_UNLESS_RESULT_SUCCESS(pHost->QueryAllInterfaces(&g_IfCount, g_IfBuffer, sizeof(g_IfBuffer), &filter));

            auto end = nn::os::GetSystemTick();
            g_ApiExecutionTime = (end - start).ToTimeSpan();
        }

    } // namespace

    void SceneGetStatus(SceneResult *pOutResult, const SceneGetStatusArg& arg, nn::os::SystemEventType *pNoticeEvent, nn::usb::Host *pHost)
    {
        g_SceneStart = nn::os::GetSystemTick();

        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::vi::OpenDisplay(&g_pDisplay, "External"));

        nn::usb::pd::OpenSession( &g_PdSession );

        nn::usb::pd::BindNoticeEvent(pNoticeEvent, &g_PdSession);

        nn::os::ClearSystemEvent(pNoticeEvent);

        g_PdActiveNoticeCount = 0;

        Update(pNoticeEvent, pHost);

        for(int64_t frame = 0;; frame++)
        {
            input::Update();
            Print(arg.state);

            if (frame % UpdateInterval == 0)
            {
                Update(pNoticeEvent, pHost);
            }

            // キャンセルボタンが押されたらループを抜ける
            if (input::IsCancelPushed())
            {
                break;
            }

            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(FrameLength));
        }

        nn::usb::pd::UnbindNoticeEvent(&g_PdSession);

        nn::usb::pd::CloseSession(&g_PdSession);

        nn::vi::CloseDisplay(g_pDisplay);

        pOutResult->nextSceneType = SceneType_Menu;
        pOutResult->nextSceneArg.menu.type = SceneMenuType_GetStatus;
    }
}}}
