﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt/nntest.h>
#include <nnt/result/testResult_Assert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/updater/updater.h>
#include <nn/updater/updater_DebugApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>

// 現状、あらかじめ BootImagePackage をインストールしておく必要がある

namespace
{
NN_ALIGNAS(4096) char s_Buffer[64 << 10]; // 256KiB

// see: ns_SystemUpdateApplyManager.cpp
nn::updater::BootImageUpdateType GetBootImageUpdateType() NN_NOEXCEPT
{
    int bootImageUpdateType;
    auto size = nn::settings::fwdbg::GetSettingsItemValue(&bootImageUpdateType, sizeof(bootImageUpdateType), "systeminitializer", "boot_image_update_type");
    // 設定項目の取得に失敗した場合は旧来の挙動にする
    if (size != sizeof(bootImageUpdateType))
    {
        return nn::updater::BootImageUpdateType::Original;
    }
    return nn::updater::GetBootImageUpdateType(bootImageUpdateType);
}
}

class UpdaterTest : public testing::Test
{
protected:
    virtual void SetUp()
    {
        nn::ncm::Initialize();

        auto updateType = GetBootImageUpdateType();
        nn::ncm::SystemDataId id;
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::GetBootImagePackageId(&id, nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::UpdateBootImagesFromPackage(id, nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer), updateType));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::VerifyBootImages(id, nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer), updateType));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::MarkVerified(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
        NN_ABORT_UNLESS_RESULT_SUCCESS(nn::updater::MarkVerified(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer)));
    }
    virtual void TearDown()
    {
        nn::ncm::Finalize();
    }
};

TEST_F(UpdaterTest, Flags)
{
    nn::updater::VerifyingRequiredFlags flags;

    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_FALSE(flags.normalRequired);
    EXPECT_FALSE(flags.safeRequired);

    NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_TRUE(flags.normalRequired);
    EXPECT_FALSE(flags.safeRequired);

    NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerified(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_FALSE(flags.normalRequired);
    EXPECT_FALSE(flags.safeRequired);

    NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_FALSE(flags.normalRequired);
    EXPECT_TRUE(flags.safeRequired);

    NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerified(nn::updater::TargetBootMode::Safe, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_FALSE(flags.normalRequired);
    EXPECT_FALSE(flags.safeRequired);
}

TEST_F(UpdaterTest, PowerOffed)
{
    auto updateType = GetBootImageUpdateType();
    bool normal, safe;
    nn::updater::VerifyingRequiredFlags flags;

    // VerifyBootImagesAndUpdateIfNeeded は、いつ呼んでも Success を返す
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::VerifyBootImagesAndUpdateIfNeeded(&normal, &safe, s_Buffer, sizeof(s_Buffer), updateType));
    EXPECT_FALSE(normal); EXPECT_FALSE(safe);

    // フラグを立てるだけ
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::VerifyBootImagesAndUpdateIfNeeded(&normal, &safe, s_Buffer, sizeof(s_Buffer), updateType));
    NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
    EXPECT_FALSE(normal); EXPECT_FALSE(safe);
    EXPECT_FALSE(flags.normalRequired); EXPECT_FALSE(flags.safeRequired);

    // フラグ + 部分的に壊れた状態
    // 本当はその状態で再起動して大丈夫かを見たいが、updater ライブラリのテストとしてはそこまでは求めない
    // phase4 はこのテストでは問題なくなるが、boot プロセスに任せる場合は単純にはいかない
    nn::updater::UpdatePhase phases[] = {
        nn::updater::UpdatePhase::Phase1,
        nn::updater::UpdatePhase::Phase2,
        nn::updater::UpdatePhase::Phase3,
        nn::updater::UpdatePhase::Phase4,
        nn::updater::UpdatePhase::Phase5,
        nn::updater::UpdatePhase::Phase6
    };
    for (auto& phase : phases)
    {
        NNT_EXPECT_RESULT_SUCCESS(nn::updater::MarkVerifyingRequired(nn::updater::TargetBootMode::Normal, s_Buffer, sizeof(s_Buffer)));
        NNT_EXPECT_RESULT_SUCCESS(nn::updater::SetBootImagesIntermediatePhase(nn::updater::TargetBootMode::Normal, phase, s_Buffer, sizeof(s_Buffer)));
        NNT_EXPECT_RESULT_SUCCESS(nn::updater::VerifyBootImagesAndUpdateIfNeeded(&normal, &safe, s_Buffer, sizeof(s_Buffer), updateType));
        NNT_EXPECT_RESULT_SUCCESS(nn::updater::CheckVerifyingRequired(&flags, s_Buffer, sizeof(s_Buffer)));
        EXPECT_TRUE(normal); EXPECT_FALSE(safe);
        EXPECT_FALSE(flags.normalRequired); EXPECT_FALSE(flags.safeRequired);
    }
}
