﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @detail  PC 側の専用テストツール (UartSerialCommunicationTestHost) との通信テスト。
 *          テスト動作には、テスト対象の UART ポートの対向 PC でツールが実行中の状態であることが必要。
 */

// コメント中の用語に関するノート（文言揺れがあるかもしれないので）
// 「ターゲット」「本プログラム側」：このテストプログラムのこと、またそれが動作する環境（ターゲット環境）のこと
// 「ホスト」「対向デバイス」：PC 側のテストツールのこと、またそれが動作する環境（ホスト PC 環境）のこと

#include <cstring>

#include <nn/TargetConfigs/build_Base.h>

#include <nn/os.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/nn_StaticAssert.h>
#include <nn/nn_Log.h>
#include <nn/nn_Abort.h>

#include <nnt/gtest/gtest.h>

#include <nn/uart/uart.h>
#include <nn/uart/uart_PortApiDev.h>

#include "../Common/testUart_Helper.h"

namespace {

// ポートに設定する送受信バッファ
const size_t UartBufferSize = 1 * nn::os::MemoryPageSize;
NN_ALIGNAS(nn::os::MemoryPageSize) char s_SendBuffer[UartBufferSize];
NN_ALIGNAS(nn::os::MemoryPageSize) char s_ReceiveBuffer[UartBufferSize];

// テストデータを置く十分大きな汎用バッファ群
const size_t TestBufferSize = 8 * 1024 * 1024;
NN_ALIGNAS(nn::os::MemoryPageSize) char s_DataBuffer1[TestBufferSize];
NN_ALIGNAS(nn::os::MemoryPageSize) char s_DataBuffer2[TestBufferSize];

// 同じサイズなことを前提にするテストがいるのでこれを保証すること
NN_STATIC_ASSERT(sizeof(s_DataBuffer1) == sizeof(s_DataBuffer2));

#if   defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6)
    const int TestedPortIndex = 0;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
    const int TestedPortIndex = 0;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2)
    const int TestedPortIndex = 2;
#else
    #error NN_BUILD_CONFIG_HARDWARE_ not selected or supported
#endif

///////////////////////////////////////////////////////////////////////////////
// 以下の型定義は、UartSerialCommunicationTestHost 側の同名の型と定義値およびデータサイズを一致させること
enum TestScenario : uint8_t
{
    ShortLoopback = 0x01,
    OverRunDetection = 0x02,
    OutOfBufferDetection = 0x03,
    SendWithFlowControl = 0x04,
    LazyReceiveWithFlowControl = 0x05,
    ReceiveUsingPortEvent = 0x06,
    SendUsingPortEvent = 0x07,
    ReceiveUsingReceiveEndEvent = 0x08,
    TestTermination = 0xff // PC への実行終了シグナル
};
enum TestBaudRateType : uint8_t
{
    TestBaudRateType_57600 = 0x01,
    TestBaudRateType_115200 = 0x02
};
enum TestFlowControlMode : uint8_t
{
    TestFlowControlMode_None = 0x01,
    TestFlowControlMode_Hardware = 0x02
};
struct TestParam
{
    TestScenario        scenario;
    TestBaudRateType    baudRate;
    TestFlowControlMode flowControlMode;
    char                padding;
};
NN_STATIC_ASSERT(sizeof(TestParam)==4);

nn::uart::BaudRate ToBaudRate(TestBaudRateType testBaudRate)
{
    switch (testBaudRate)
    {
    case TestBaudRateType_57600:
        {
            return nn::uart::BaudRate_57600;
        }
    case TestBaudRateType_115200:
        {
            return nn::uart::BaudRate_115200;
        }
    default:
        {
            NN_ABORT();
        }
    }
}
size_t ToSizeTypeBaudRate(TestBaudRateType testBaudRate)
{
    return static_cast<size_t>(ToBaudRate(testBaudRate));
}
nn::uart::FlowControlMode ToFlowControlMode(TestFlowControlMode testFlowControlMode)
{
    switch (testFlowControlMode)
    {
    case TestFlowControlMode_Hardware:
        {
            return nn::uart::FlowControlMode_Hardware;
        }
    case TestFlowControlMode_None:
        {
            return nn::uart::FlowControlMode_None;
        }
    default:
        {
            NN_ABORT();
        }
    }
}

///////////////////////////////////////////////////////////////////////////////
// テストパラメータ生成用の構造体
// ボーレート以外の組み合わせ因子が増えたらここに追加
const TestParam TestParamList[] =
{
    {ShortLoopback, TestBaudRateType_57600, TestFlowControlMode_None},
    {ShortLoopback, TestBaudRateType_115200, TestFlowControlMode_None},
    {SendWithFlowControl, TestBaudRateType_57600, TestFlowControlMode_Hardware},
    {SendWithFlowControl, TestBaudRateType_115200, TestFlowControlMode_Hardware},
    {LazyReceiveWithFlowControl, TestBaudRateType_57600, TestFlowControlMode_Hardware},
    {LazyReceiveWithFlowControl, TestBaudRateType_115200, TestFlowControlMode_Hardware},
    {ReceiveUsingPortEvent, TestBaudRateType_57600, TestFlowControlMode_Hardware},
    {ReceiveUsingPortEvent, TestBaudRateType_115200, TestFlowControlMode_Hardware},
    {SendUsingPortEvent, TestBaudRateType_57600, TestFlowControlMode_Hardware},
    {SendUsingPortEvent, TestBaudRateType_115200, TestFlowControlMode_Hardware},

    // Tegra 専用のテストケース
#if   defined(NN_BUILD_CONFIG_SOC_TEGRA_K1) || \
      defined(NN_BUILD_CONFIG_SOC_TEGRA_X1)
    {ReceiveUsingReceiveEndEvent, TestBaudRateType_57600, TestFlowControlMode_Hardware},
    {ReceiveUsingReceiveEndEvent, TestBaudRateType_115200, TestFlowControlMode_Hardware},
#endif

    // 最後のアイテムを TestTermination にする（ホスト PC への全テスト終了の合図）
    {TestTermination, TestBaudRateType_57600, TestFlowControlMode_None}
};

bool IsTestCaseSupported(TestParam testParam)
{
    nn::uart::Initialize();
    if (!nn::uart::IsSupportedBaudRateForDev(TestedPortIndex, ToBaudRate(testParam.baudRate)))
    {
        return false;
    }
    if (!nn::uart::IsSupportedFlowControlModeForDev(TestedPortIndex, ToFlowControlMode(testParam.flowControlMode)))
    {
        return false;
    }
    nn::uart::Finalize();
    return true;
}

///////////////////////////////////////////////////////////////////////////////
// 共通のテスト処理群
//

void SendAck(nn::uart::PortSession& portSession)
{
    const char Ack[4] = {'A', 'B', 'C', 'D'};
    NN_LOG("Sending Ack\n");
    nn::uart::Send(nullptr, &portSession, Ack, sizeof(Ack));
}

bool WaitAck(nn::uart::PortSession& portSession)
{
    char ack[4];

    NN_LOG("Waiting Ack\n");
    if (!nnt::uart::ReceiveDataSync(nullptr, ack, sizeof(ack), portSession, nn::TimeSpan::FromSeconds(2)))
    {
        return false;
    }

    NN_LOG("Received: 0x%02x%02x%02x%02x\n", ack[0], ack[1], ack[2], ack[3]);
    if (ack[0] != 'A' ||
        ack[1] != 'B' ||
        ack[2] != 'C' ||
        ack[3] != 'D')
    {
        return false;
    }
    return true;
}

bool SyncSignature(nn::uart::PortSession& portSession)
{
    const int SignitureTimeoutMsec = 5000;
    const int SignitureWaitInterval = 1000;
    int signitureWaitedTimeMsec = 0;
    while (signitureWaitedTimeMsec < SignitureTimeoutMsec)
    {
        SendAck(portSession);
        if (WaitAck(portSession))
        {
            return true; // SUCCESS
        }

        // リトライ
        signitureWaitedTimeMsec += SignitureWaitInterval;
    }
    NN_LOG("Host PC is not responding. "
           "Please make sure that the UartSerialCommunicationTestHost is running on PC "
           "and properly connected to tested UART port.\n");
    return false;
}

// PC 側テストツールと既定のレートでハンドシェイクし、テストパラメータを先方に通知する
bool TestHandshake(const TestParam& testParam)
{
    // 前準備
    // ハンドシェイクでの通信モードは固定
    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        nn::uart::BaudRate_57600,
        s_SendBuffer,
        sizeof(s_SendBuffer),
        s_ReceiveBuffer,
        sizeof(s_ReceiveBuffer)
        );
    nn::uart::OpenPortForDev(&portSession, TestedPortIndex, portConfig);

    // 同期
    NN_LOG("Handshake - Sync\n");
    if (!SyncSignature(portSession))
    {
        return false;
    }

    // テストパラメータの送信
    NN_LOG("Handshake - Sending Test Parameters\n");
    size_t doneBytes;
    nn::uart::Send(&doneBytes, &portSession, &testParam, sizeof(TestParam));
    if (doneBytes != sizeof(TestParam))
    {
        NN_LOG("Could not send all Test Parameters - sent %d bytes\n", doneBytes);
        return false;
    }

    // 受理確認
    if (!WaitAck(portSession))
    {
        return false;
    }

    NN_LOG("Handshake - Done\n");
    nn::uart::ClosePort(&portSession);

    return true;
}

/**
    もっとも基本的な送受信が機能していることのテスト

    以下の仕様であることをテストする
        - Send ですべてのデータを対抗に送出できること
        - Receive ですべてのデータを対抗から受信できること
        - 通常の状況であれば、フロー制御なしでも送受信時にオーバーラン等のエラーが発生しない通信品質であること

    テスト手順
        1. 256 バイトのデータを送出し、2 秒以内に送ったデータと同じものが返ってくるかをテストする
 */
void TEST_ShortLoopback(nn::uart::PortSession& portSession)
{
    const size_t TestDataCount = 256;
    char bufferToSend[TestDataCount];
    char bufferToReceive[TestDataCount];

    nnt::uart::GenerateTestData(bufferToSend, TestDataCount);
    nnt::uart::ClearBuffer(bufferToReceive, TestDataCount);

    NN_LOG("Sending\n");
    size_t doneBytes;
    nn::uart::Send(&doneBytes, &portSession, bufferToSend, sizeof(bufferToSend));
    ASSERT_EQ(TestDataCount, doneBytes);

    NN_LOG("Receiving\n");
    ASSERT_TRUE(nnt::uart::ReceiveDataSync(nullptr, bufferToReceive, sizeof(bufferToReceive), portSession, nn::TimeSpan::FromSeconds(2)));

    NN_LOG("Received test data. Verifying\n");
    ASSERT_TRUE(nnt::uart::CompareTestData(bufferToSend, bufferToReceive, TestDataCount));

    NN_LOG("Verify done. Success!\n");
}

/**
    オーバーランエラーを検出できることのテスト（フロー制御オフ時）

    以下の仕様であることをテストする
        - Receive を呼んでいない間の受信バッファサイズを超えない短時間の大量受信時、以下の挙動のどちらかになること
            - 次の Receive が成功し、受信データがすべて揃っている
            - ResultHardwareOverrun エラーが次の Receive で返り、受信データが欠けている

    テスト手順
        1. (TBD)
 */
void TEST_OverRunDetection(nn::uart::PortSession& portSession)
{
    // TODO: Implement me
}

/**
    受信バッファ溢れを検出できることのテスト（フロー制御オフ時）

    以下の仕様であることをテストする
        - Receive を呼んでいない間の受信バッファサイズを超えるデータを受信すると、次の Receive で
          ResultOutOfBuffer エラーが返る

    テスト手順
        1. 対向デバイス側は 1024 バイトなど、少ないサイズの受信バッファを用意して受信開始
           ただし対向側は受信バッファからの取り出しはせず、データを溜めるのみ
        2. 本プログラム側から 4096 バイトのデータを Send() で送信する
        3. 本プログラム側で GetWritableLength() をウォッチし、空きが 1024 バイトから 2 秒間増えないことをテストする
        4. 対向デバイス側は、受信バッファが 1024 バイトでいっぱいになった状態をテストしてから 3 秒待機し、
           データをすべて受信する
        5. 対向デバイス側は、受信したデータ全体をそのまま送信し返す（ループバック）
        6. 本プログラム側は、手順 3 の後 Receive() を繰り返し呼び、送信した分のデータが数秒以内にすべて
           ループバックされてくることをテストする
        7. 本プログラム側でループバックされたデータをベリファイする
 */
void TEST_OutOfBufferDetection(nn::uart::PortSession& portSession)
{
    // TODO: Implement me
}

/**
    フロー制御オンで本プログラム側が送信側になり、オーバーランが発生しないことのテスト

    以下の仕様であることをテストする
        - 対向デバイス側の受信バッファが一杯な間は、送信バッファの送信予約済データが減らないこと
        - その後、対向デバイス側が受信を行うと、正しく続きが送信されること

    テスト手順
        1. 対向デバイス側はなるべく少ないサイズの受信バッファを用意して受信開始
           ただし対向側は受信バッファからの取り出しはせず、データを溜めるのみ
        2. 大きなデータを用意し、本プログラム側から、フロー制御により送信が止まる（送信バッファが満杯になって
           減らなくなる）まで送信する。
           これ以上送れない状態が数十 ms 継続したら成功とする。
           そうなることなく用意したデータが全部送れてしまったら失敗。
           （どれだけ用意しておかないといけないかはホスト側の設定に依存する）
        3. 本プログラム側は 2 秒待機してから残りをすべて送信
        4. 対向デバイス側は、受信バッファがいっぱいになった状態をテストしてから 2 秒待機し、
           データをすべて受信する
        5. 対向デバイス側は、受信したデータ全体をそのまま送信し返す（ループバック）
        6. 本プログラム側は、2. の後 Receive() を繰り返し呼び、送信した分のデータがすべて
           ループバックされてくることをテストする
        7. 本プログラム側でループバックされたデータをベリファイする
 */
void TEST_SendWithFlowControl(nn::uart::PortSession& portSession, const TestParam& testParam)
{
    // 最大 4 秒分ほどデータを用意（ホスト側のバッファをいっぱいにしないといけないのでそれなりに多く必要）
    // 注意：データサイズの値はホスト側でも同じ式で計算しているので、変更時にはホスト側も合わせること
    const size_t TestDataCount = ((ToSizeTypeBaudRate(testParam.baudRate) / 2) < TestBufferSize) ?
                               (ToSizeTypeBaudRate(testParam.baudRate) / 2) : TestBufferSize;
    char* bufferToSend = s_DataBuffer1;
    char* bufferToReceive = s_DataBuffer2;

    // 送受信バッファがいっぱいになりそうな待ち時間
    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(
            UartBufferSize * 1000 / (ToSizeTypeBaudRate(testParam.baudRate) / 8));

    nnt::uart::GenerateTestData(bufferToSend, TestDataCount);
    nnt::uart::ClearBuffer(bufferToReceive, TestDataCount);

    /*
        2. 大きなデータを用意し、本プログラム側から、フロー制御により送信が止まる（送信バッファが満杯になって
           減らなくなる）まで送信する。
           これ以上送れない状態が数十 ms 継続したら成功とする。
           そうなることなく用意したデータが全部送れてしまったら失敗。
           （どれだけ用意しておかないといけないかはホスト側の設定に依存する）
     */
    NN_LOG("Sending\n");
    size_t remainingBytes = TestDataCount;
    char* pCurrent = bufferToSend;
    while (remainingBytes > 0)
    {
        const size_t SendChunkSize = UartBufferSize;
        size_t bytesToSend = (remainingBytes < SendChunkSize) ? remainingBytes : SendChunkSize;
        size_t doneBytes;
        nn::uart::Send(&doneBytes, &portSession, pCurrent, bytesToSend);
        if (doneBytes == 0)
        {
            // 一瞬一杯になっただけなのか、フロー制御により送信が止まっているのか判定
            // 前者なら送信を続ける
            nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(50) );

            size_t writableLength = nn::uart::GetWritableLength(&portSession);
            if (writableLength == 0)
            {
                // 成功：少し待ってから再度見てもバッファが満杯のままなので、フロー制御が効いていると判定
                NN_LOG("OK! remainingBytes = %d\n", remainingBytes);
                break;
            }
            // まだ送信できるようなので続行
            continue;
        }
        else
        {
            remainingBytes -= doneBytes;
            pCurrent += doneBytes;
        }
    }
    ASSERT_GT(remainingBytes, static_cast<size_t>(0)); // 全部送信できてしまったら失敗

    // 1秒待機したら残りをすべて送信
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(1000) );
    ASSERT_TRUE(nnt::uart::SendData(portSession, pCurrent, remainingBytes, UartBufferSize, Interval)); // 今のところ必ず true が返るが一応

    /*
        5. 本プログラム側は、2. の後 Receive() を繰り返し呼び、送信した分のデータがすべて
           ループバックされてくることをテストする
     */
    NN_LOG("Receiving\n");
    ASSERT_TRUE(nnt::uart::ReceiveData(portSession, bufferToReceive, TestDataCount, UartBufferSize, Interval));

    /*
        6. 本プログラム側でループバックされたデータをベリファイする
     */
    NN_LOG("Received test data. Verifying\n");
    ASSERT_TRUE(nnt::uart::CompareTestData(bufferToSend, bufferToReceive, TestDataCount));

    NN_LOG("Verify done. Success!\n");
}


/**
    フロー制御オンで本プログラム側が受信側になり、受信呼び出しを遅延させてもオーバーランが発生しないことのテスト

    以下の仕様であることをテストする
        - Receive を呼んでいない間のデータ受信でも受信バッファの溢れやエラーが起きず、全データを受信できる

    テスト手順
        1. 本プログラム側から、受信バッファのサイズよりも大きい適当なサイズのデータを送信する
        2. 対向デバイス側はデータをすべて受信したらすぐに全体を送信し返す（ループバック）
        3. 本プログラム側は、手順 2 の後 GetReadableLength() をポーリングし、
           受信バッファが一杯になるのを待つ
        4. 本プログラム側は、受信バッファが一杯になったら 3 秒待機し、全データを受信する
        5. 本プログラム側でループバックされたデータをベリファイする
 */
void TEST_LazyReceiveWithFlowControl(nn::uart::PortSession& portSession, const TestParam& testParam)
{
    // こちらの受信バッファが一杯になる程度の適当な半端なサイズ
    // 注意：データサイズの値はホスト側でも同じ式で計算しているので、変更時にはホスト側も合わせること
    const size_t TestDataCount = 10 * 1024 + 19;
    char* bufferToSend = s_DataBuffer1;
    char* bufferToReceive = s_DataBuffer2;

    // 送受信バッファがいっぱいになりそうな待ち時間
    const nn::TimeSpan Interval = nn::TimeSpan::FromMilliSeconds(
            UartBufferSize * 1000 / (ToSizeTypeBaudRate(testParam.baudRate) / 8));

    nnt::uart::GenerateTestData(bufferToSend, TestDataCount);
    nnt::uart::ClearBuffer(bufferToReceive, TestDataCount);

    /*
        1. 本プログラム側から、受信バッファのサイズよりも大きい適当なサイズのデータを送信する
     */
    NN_LOG("Sending\n");
    ASSERT_TRUE(nnt::uart::SendData(portSession, bufferToSend, TestDataCount, UartBufferSize, Interval)); // 今のところ必ず true が返るが一応

    /*
        3. 本プログラム側は、手順 2 の後 GetReadableLength() をポーリングし、
           受信バッファが一杯になるのを待つ
     */
    NN_LOG("Waiting for receive buffer getting full\n");
    ASSERT_TRUE(nnt::uart::WaitDataReady(portSession, UartBufferSize, nn::TimeSpan::FromSeconds(10)));

    /*
        4. 本プログラム側は、受信バッファが一杯になったら 3 秒待機し、全データを受信する
     */
    NN_LOG("Receive buffer is now full, waiting a while before start reading\n");
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(3000) );
    NN_LOG("Receiving all data\n");
    ASSERT_TRUE(nnt::uart::ReceiveData(portSession, bufferToReceive, TestDataCount, UartBufferSize, Interval));

    /*
        6. 本プログラム側でループバックされたデータをベリファイする
     */
    NN_LOG("Received test data. Verifying\n");
    ASSERT_TRUE(nnt::uart::CompareTestData(bufferToSend, bufferToReceive, TestDataCount));

    NN_LOG("Verify done. Success!\n");
}

/**
    本プログラム側が受信側になり、ポートイベントの ReceiveBufferReady を使用して
    受信バッファをポーリングせずにデータを受信できることのテスト

    以下の仕様であることをテストする
        - ReceiveBufferReady イベントに登録した閾値分のデータを受信した際、登録したシステムイベントがシグナルされる
        - シグナル化されたイベントをクリアした後に閾値分のデータを受信すると、再度イベントがシグナルされる

    テスト手順
        手順 1-3 を、テスト対象の各閾値に対して 3 回ずつ行う。

        1. 本プログラム側から、テストに使用するデータのサイズ (= 閾値) を送信する
        2. 本プログラム側は、対向デバイスから閾値分のデータが送られてくるのを待つ
           * 受信待ちはポーリングではなく、BindPortEvent() で登録した ReceiveBufferReady イベントで行う
        3. 本プログラム側は、ReceiveBufferReady イベントが発生したらイベントをクリアし、閾値分のデータを受信する
 */
void TEST_ReceiveUsingPortEvent(nn::uart::PortSession& portSession, const TestParam& testParam)
{
    // テストする閾値のリスト
    //   { (最小), (最小) + 1, (適当な値), (受信バッファサイズ) - 1, (受信バッファサイズ) }
    // 注意： 末尾の 0 はテスト終了として扱う。
    const size_t TestThresholdList[] = { 1, 2, 192, UartBufferSize - 1, UartBufferSize, 0 };

    // 各閾値ごとのテスト回数
    const int TestLoopCount = 3;

    for (auto threshold : TestThresholdList)
    {
        NN_LOG("Threshold = %d\n", threshold);

        // イベントの登録
        nn::os::SystemEventType receiveEvent;
        if (threshold > 0)
        {
            ASSERT_TRUE(nn::uart::IsSupportedPortEventForDev(TestedPortIndex,
                                                             nn::uart::PortEventType_ReceiveBufferReady));
            ASSERT_TRUE(
                    nn::uart::BindPortEvent(&receiveEvent,
                                            &portSession,
                                            nn::uart::PortEventType_ReceiveBufferReady,
                                            threshold));
        }

        for (int i = 0; i < TestLoopCount; ++i)
        {
            // 受信タイムアウト : 10秒
            const nn::TimeSpan ReceiveTimeout = nn::TimeSpan::FromSeconds(10);

            char* bufferToSend    = s_DataBuffer1;
            char* bufferToReceive = s_DataBuffer2;

            /*
                1. 本プログラム側から、テストに使用するデータのサイズ (= 閾値) を送信する
             */

            // テストデータサイズ (閾値) を 4 バイトで送信
            // (対向デバイス側のツールに合わせてリトルエンディアンとする)
            bufferToSend[0] = threshold & 0xFF;
            bufferToSend[1] = (threshold >>  8) & 0xFF;
            bufferToSend[2] = (threshold >> 16) & 0xFF;
            bufferToSend[3] = (threshold >> 24) & 0xFF;

            NN_LOG("Sending test data size\n");
            ASSERT_TRUE(nnt::uart::SendData(portSession, bufferToSend, 4, UartBufferSize, nn::TimeSpan::FromSeconds(1))); // 今のところ必ず true が返るが一応

            // 閾値が 0 の場合はテスト終了扱い
            if (threshold == 0)
            {
                NN_LOG("End of %s\n", NN_CURRENT_FUNCTION_NAME);
                return;
            }

            /*
                2. 本プログラム側は、対向デバイスから閾値分のデータが送られてくるのを待つ
                   (受信待ちはポーリングではなく、BindPortEvent() で登録した ReceiveBufferReady イベントで行う)
             */

            // バッファにデータが溜まるまで待つ
            NN_LOG("Waiting for receiving data...");
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&receiveEvent, ReceiveTimeout));

            /*
                3. 本プログラム側は、ReceiveBufferReady イベントが発生したらイベントをクリアし、
                   閾値分のデータを受信する
             */
            nn::os::ClearSystemEvent(&receiveEvent);

            // バッファ内のデータを受信
            size_t doneBytes;
            nn::Result result = nn::uart::Receive(&doneBytes,
                                                  bufferToReceive,
                                                  threshold,
                                                  &portSession);
            ASSERT_TRUE(result.IsSuccess());
            ASSERT_EQ(doneBytes, threshold);
            NN_LOG("OK (%d bytes)\n", doneBytes);
        }

        // イベントの登録解除
        ASSERT_TRUE(nn::uart::UnbindPortEvent(&receiveEvent, &portSession));

        // システムイベントはドライバ使用者側で破棄する必要がある
        nn::os::DestroySystemEvent(&receiveEvent);
    }
}

/**
    本プログラム側が送信側になり、ポートイベントの SendBufferEmpty および SendBufferReady を使用して、
    送信バッファをポーリングせずにデータ送信の完了を検知できることのテスト

    以下の仕様であることをテストする
        - データ送信後、送信バッファに SendBufferReady イベントに登録した閾値分の空きができた際、登録したシステムイベントがシグナルされる
        - データ送信後、送信バッファが空になった際、SendBufferEmpty に登録したシステムイベントがシグナルされる
        - シグナル化されたイベントをクリアした後にデータを送信し、送信バッファの空きサイズが閾値または 0 に到達すると、再度イベントがシグナルされる

    テスト手順
        手順 1-4 を、テスト対象の各閾値に対して 3 回ずつ行う。

        1. 本プログラム側から、テストで送るデータのサイズを送信する
        2. 本プログラム側から、対向デバイスに送信バッファサイズ分のデータを送信する
        3. 本プログラム側は、送信バッファに閾値分の空きができるのを待つ
        4. 本プログラム側は、送信バッファが空になるのを待つ
           * 送信待ちはポーリングではなく、BindPortEvent() で登録したイベントで行う
 */
void TEST_SendUsingPortEvent(nn::uart::PortSession& portSession, const TestParam& testParam)
{
    // テストする閾値のリスト
    //   { (最小), (最小) + 1, (適当な値), (送信バッファサイズ) - 1, (送信バッファサイズ) }
    // 注意： 末尾の 0 はテスト終了として扱う。
    const size_t TestThresholdList[] = { 1, 2, 192, UartBufferSize - 1, UartBufferSize, 0 };

    // 各閾値ごとのテスト回数
    const int TestLoopCount = 3;

    // SendBufferEmpty イベントの登録
    nn::os::SystemEventType emptyEvent;
    ASSERT_TRUE(nn::uart::IsSupportedPortEventForDev(TestedPortIndex,
                                                     nn::uart::PortEventType_SendBufferEmpty));
    ASSERT_TRUE(nn::uart::BindPortEvent(&emptyEvent,
                                        &portSession,
                                        nn::uart::PortEventType_SendBufferEmpty,
                                        0));

    for (auto threshold : TestThresholdList)
    {
        NN_LOG("Threshold = %d\n", threshold);

        // SendBufferReady イベントの登録
        nn::os::SystemEventType sendEvent;
        if (threshold > 0)
        {
            ASSERT_TRUE(nn::uart::IsSupportedPortEventForDev(TestedPortIndex,
                                                             nn::uart::PortEventType_SendBufferReady));
            ASSERT_TRUE(
                    nn::uart::BindPortEvent(&sendEvent,
                                            &portSession,
                                            nn::uart::PortEventType_SendBufferReady,
                                            threshold));
        }

        for (int i = 0; i < TestLoopCount; ++i)
        {
            // 送信タイムアウト : 10秒
            const nn::TimeSpan SendTimeout = nn::TimeSpan::FromSeconds(10);

            char* bufferToSend = s_DataBuffer1;

            /*
                1. 本プログラム側から、テストで送るデータのサイズを送信する
             */

            // 送信データサイズは、通常は送信バッファサイズ、テスト終了のときのみ 0
            size_t TestDataSize = threshold > 0 ? UartBufferSize : 0;

            // 送信データサイズを 4 バイトで送信
            // (対向デバイス側のツールに合わせてリトルエンディアンとする)
            bufferToSend[0] = TestDataSize & 0xFF;
            bufferToSend[1] = (TestDataSize >>  8) & 0xFF;
            bufferToSend[2] = (TestDataSize >> 16) & 0xFF;
            bufferToSend[3] = (TestDataSize >> 24) & 0xFF;

            nn::os::ClearSystemEvent(&emptyEvent);

            NN_LOG("Sending test data size (%d)\n", TestDataSize);
            ASSERT_TRUE(nnt::uart::SendData(portSession, bufferToSend, 4, UartBufferSize, nn::TimeSpan::FromSeconds(1))); // 今のところ必ず true が返るが一応

            // 送信が完全に終了するのを待つ
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&emptyEvent, SendTimeout));

            // 送信バッファが空であることを確認
            ASSERT_EQ(nn::uart::GetWritableLength(&portSession), UartBufferSize);

            // 閾値が 0 の場合はテスト終了扱い
            if (threshold == 0)
            {
                NN_LOG("End of %s\n", NN_CURRENT_FUNCTION_NAME);
                break;
            }

            nn::os::ClearSystemEvent(&emptyEvent);
            nn::os::ClearSystemEvent(&sendEvent);

            /*
                2. 本プログラム側から、対向デバイスに送信バッファサイズ分のデータを送信する
             */

            nnt::uart::GenerateTestData(bufferToSend, TestDataSize);

            NN_LOG("Start sending test data\n");
            size_t doneBytes;
            nn::uart::Send(&doneBytes, &portSession, bufferToSend, TestDataSize);
            ASSERT_EQ(doneBytes, TestDataSize);

            /*
                3. 本プログラム側は、送信バッファに閾値分の空きができるのを待つ
             */

            NN_LOG("Waiting for sending %d bytes...", threshold);
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&sendEvent, SendTimeout));

            // 送信バッファに閾値以上の空きがあることを確認
            size_t writableLength = nn::uart::GetWritableLength(&portSession);
            ASSERT_GE(writableLength, threshold);
            NN_LOG("OK (%d bytes free in the send buffer)\n", writableLength);

            /*
                4. 本プログラム側は、送信バッファが空になるのを待つ
             */

            NN_LOG("Waiting for sending all data...");
            ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&emptyEvent, SendTimeout));

            // 送信バッファが空であることを確認
            ASSERT_EQ(nn::uart::GetWritableLength(&portSession), UartBufferSize);
            NN_LOG("OK\n");
        }

        if (threshold > 0)
        {
            // イベントの登録解除
            ASSERT_TRUE(nn::uart::UnbindPortEvent(&sendEvent, &portSession));

            // システムイベントはドライバ使用者側で破棄する必要がある
            nn::os::DestroySystemEvent(&sendEvent);
        }
    }

    ASSERT_TRUE(nn::uart::UnbindPortEvent(&emptyEvent, &portSession));
    nn::os::DestroySystemEvent(&emptyEvent);
}

/**
    本プログラム側が受信側になり、ポートイベントの ReceiveEnd を使用して
    受信完了を検知できることのテスト

    以下の仕様であることをテストする
        - 連続したデータの受信完了時に、ReceiveEnd イベントに登録したシステムイベントがシグナルされる
        - シグナル化されたイベントをクリアした後の受信終了で、再度イベントがシグナルされる

    テスト手順
        手順 1-2 をテスト対象の各データサイズに対して行う

        1. 本プログラム側から、テストで受信するデータのサイズを送信する
        2. 本プログラム側は、対向デバイスからのデータ受信が完了するのを待つ
 */
void TEST_ReceiveUsingReceiveEndEvent(nn::uart::PortSession& portSession, const TestParam& testParam)
{
    // テストするデータサイズのリスト
    //   { (最小), (最小) + 1, (適当な値), (受信バッファサイズ) - 1, (受信バッファサイズ) }
    // 注意： 末尾の 0 はテスト終了として扱う。
    const size_t TestDataSizeList[] = { 1, 2, 192, UartBufferSize - 1, UartBufferSize, 0 };

    ASSERT_TRUE(nn::uart::IsSupportedPortEventForDev(TestedPortIndex,
                                                     nn::uart::PortEventType_SendBufferEmpty));
    ASSERT_TRUE(nn::uart::IsSupportedPortEventForDev(TestedPortIndex,
                                                     nn::uart::PortEventType_ReceiveEnd));

    // イベントの登録
    nn::os::SystemEventType emptyEvent;
    nn::os::SystemEventType receiveEndEvent;
    ASSERT_TRUE(nn::uart::BindPortEvent(&emptyEvent,
                                        &portSession,
                                        nn::uart::PortEventType_SendBufferEmpty,
                                        0));
    ASSERT_TRUE(nn::uart::BindPortEvent(&receiveEndEvent,
                                        &portSession,
                                        nn::uart::PortEventType_ReceiveEnd,
                                        0));

    for (const auto TestDataSize : TestDataSizeList)
    {
        // 通信タイムアウト : 10秒
        const nn::TimeSpan XferTimeout = nn::TimeSpan::FromSeconds(10);

        char* bufferToSend    = s_DataBuffer1;
        char* bufferToReceive = s_DataBuffer2;

        /*
            1. 本プログラム側から、テストで受信するデータのサイズを送信する
         */

        // 受信データサイズを 4 バイトで送信
        // (対向デバイス側のツールに合わせてリトルエンディアンとする)
        bufferToSend[0] = TestDataSize & 0xFF;
        bufferToSend[1] = (TestDataSize >>  8) & 0xFF;
        bufferToSend[2] = (TestDataSize >> 16) & 0xFF;
        bufferToSend[3] = (TestDataSize >> 24) & 0xFF;

        nn::os::ClearSystemEvent(&emptyEvent);
        nn::os::ClearSystemEvent(&receiveEndEvent);

        NN_LOG("Sending test data size (%d)\n", TestDataSize);
        ASSERT_TRUE(nnt::uart::SendData(portSession, bufferToSend, 4, UartBufferSize, nn::TimeSpan::FromSeconds(1))); // 今のところ必ず true が返るが一応

        // 送信が完全に終了するのを待つ
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&emptyEvent, XferTimeout));

        // 送信バッファが空であることを確認
        ASSERT_EQ(nn::uart::GetWritableLength(&portSession), UartBufferSize);

        // データサイズが 0 の場合はテスト終了扱い
        if (TestDataSize == 0)
        {
            NN_LOG("End of %s\n", NN_CURRENT_FUNCTION_NAME);
            break;
        }

        /*
            2. 本プログラム側は、対向デバイスからのデータ受信が完了するのを待つ
         */

        // 受信が終わるまで待つ
        NN_LOG("Waiting for receiving data...");
        ASSERT_TRUE(nn::os::TimedWaitSystemEvent(&receiveEndEvent, XferTimeout));

        // 想定したサイズのデータを受信できることを確認
        size_t doneBytes;
        auto result = nn::uart::Receive(&doneBytes, bufferToReceive, TestDataSize, &portSession);
        ASSERT_TRUE(result.IsSuccess());
        ASSERT_EQ(doneBytes, TestDataSize);
        NN_LOG("OK (%d bytes)\n", doneBytes);
    }

    ASSERT_TRUE(nn::uart::UnbindPortEvent(&receiveEndEvent, &portSession));
    ASSERT_TRUE(nn::uart::UnbindPortEvent(&emptyEvent, &portSession));
    nn::os::DestroySystemEvent(&receiveEndEvent);
    nn::os::DestroySystemEvent(&emptyEvent);
}

// ひとつのテストパラメータの実施
void RunTest(const TestParam& testParam)
{
    nn::uart::Initialize();

    ASSERT_TRUE(TestHandshake(testParam));

    nn::uart::PortSession   portSession;
    nn::uart::PortConfig    portConfig(
        ToBaudRate(testParam.baudRate),
        s_SendBuffer,
        sizeof(s_SendBuffer),
        s_ReceiveBuffer,
        sizeof(s_ReceiveBuffer)
        );
    portConfig.SetFlowControlMode(ToFlowControlMode(testParam.flowControlMode));
    nn::uart::OpenPortForDev(&portSession, TestedPortIndex, portConfig);

    // 同期のための待ち時間
    nn::os::SleepThread( nn::TimeSpan::FromMilliSeconds(500) );

    NN_LOG("Running test param - Scenario %02x / BaudRate %02x / FlowControlMode %02x\n",
            testParam.scenario, testParam.baudRate, testParam.flowControlMode);

    switch(testParam.scenario)
    {
    case ShortLoopback:
        {
            NN_LOG("TEST_ShortLoopback : Begin\n");
            TEST_ShortLoopback(portSession);
            NN_LOG("TEST_ShortLoopback : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case OverRunDetection:
        {
            NN_LOG("TEST_OverRunDetection : Begin\n");
            TEST_OverRunDetection(portSession);
            NN_LOG("TEST_OverRunDetection : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case OutOfBufferDetection:
        {
            NN_LOG("TEST_OutOfBufferDetection : Begin\n");
            TEST_OutOfBufferDetection(portSession);
            NN_LOG("TEST_OutOfBufferDetection : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case SendWithFlowControl:
        {
            NN_LOG("TEST_SendWithFlowControl : Begin\n");
            TEST_SendWithFlowControl(portSession, testParam);
            NN_LOG("TEST_SendWithFlowControl : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case LazyReceiveWithFlowControl:
        {
            NN_LOG("TEST_LazyReceiveWithFlowControl : Begin\n");
            TEST_LazyReceiveWithFlowControl(portSession, testParam);
            NN_LOG("TEST_LazyReceiveWithFlowControl : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case ReceiveUsingPortEvent:
        {
            NN_LOG("TEST_ReceiveUsingPortEvent : Begin\n");
            TEST_ReceiveUsingPortEvent(portSession, testParam);
            NN_LOG("TEST_ReceiveUsingPortEvent : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case SendUsingPortEvent:
        {
            NN_LOG("TEST_SendUsingPortEvent : Begin\n");
            TEST_SendUsingPortEvent(portSession, testParam);
            NN_LOG("TEST_SendUsingPortEvent : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case ReceiveUsingReceiveEndEvent:
        {
            NN_LOG("TEST_ReceiveUsingReceiveEndEvent : Begin\n");
            TEST_ReceiveUsingReceiveEndEvent(portSession, testParam);
            NN_LOG("TEST_ReceiveUsingReceiveEndEvent : End\n");
            if (::testing::Test::HasFatalFailure())
            {
                return;
            }
        }
        break;
    case TestTermination:
        {
            // PC への実行終了シグナル。
            // シグナル自体はハンドシェイクによって通知成功した状態なので、
            // ここでは何もせず成功とする。
            NN_LOG("TestTermination handshake with Host PC is done.\n");
        }
        break;
    default:
        {
            FAIL() << "Undefined test scenario.\n";
        }
    }

    nn::uart::ClosePort(&portSession);

    nn::uart::Finalize();
}  // NOLINT(readability/fn_size)

}

TEST(SerialCommunicationWithHost, All)
{
    for (const TestParam testParam : TestParamList)
    {
        if (testParam.scenario == TestTermination || IsTestCaseSupported(testParam))
        {
            RunTest(testParam);
            if (HasFatalFailure())
            {
                return;
            }
        }
    }
}
