﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>

#include <nn/uart/uart.h>
#include <nn/gpio/gpio.h>
#include <nn/pinmux/pinmux.h>

#include <nnt/gtest/gtest.h>

#include "testUart_ExtCon.h"

NN_OS_ALIGNAS_THREAD_STACK char g_UThreadStack[nn::os::ThreadStackAlignment * 16];
NN_OS_ALIGNAS_THREAD_STACK char g_SThreadStack[nn::os::ThreadStackAlignment * 16];

void ExtConUTest(void* pArg) NN_NOEXCEPT
{
    NN_LOG("ExtConU Test\n");
    nnt::uart::extcon::TEST_Loopback    (nn::uart::PortName_ExtConU, nn::uart::BaudRate_3M,     nn::uart::FlowControlMode_Hardware);
}
void ExtConSTest(void* pArg) NN_NOEXCEPT
{
    NN_LOG("ExtConS Test\n");
    nnt::uart::extcon::TEST_Loopback    (nn::uart::PortName_ExtConS, nn::uart::BaudRate_3M,     nn::uart::FlowControlMode_Hardware);
}

TEST(TargetSpecificNx, ExtConU)
{
    nn::pinmux::PinmuxSession uTxSession;
    nn::gpio::GpioPadSession gpioExtConDetU;

    nn::pinmux::Initialize();
    nn::gpio::Initialize();
    nn::uart::Initialize();

    nn::pinmux::OpenSession(&uTxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
    nn::pinmux::SetPinAssignment(&uTxSession, nn::pinmux::PinAssignment_ExtConTxGpio);

    nn::gpio::OpenSession(&gpioExtConDetU, nn::gpio::GpioPadName_ExtconDetU);
    nn::gpio::SetDirection(&gpioExtConDetU, nn::gpio::Direction_Input);

    while (1)
    {
        NN_LOG("Waiting for ExtCon detection... : ExtConU:%s\n",
            (nn::gpio::GetValue(&gpioExtConDetU) == nn::gpio::GpioValue_Low) ? "True" : "False");

        if (nn::gpio::GetValue(&gpioExtConDetU) == nn::gpio::GpioValue_Low)
        {
            NN_LOG("ExtConU is detected!\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    nn::gpio::CloseSession(&gpioExtConDetU);

    nn::pinmux::CloseSession(&uTxSession);

    nn::pinmux::OpenSession(&uTxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
    nn::pinmux::SetPinAssignment(&uTxSession, nn::pinmux::PinAssignment_ExtConTxUart);

    nnt::uart::extcon::TEST_BasicCommand(nn::uart::PortName_ExtConU, nn::uart::BaudRate_115200, nn::uart::FlowControlMode_Hardware);

    NN_LOG("BasicCommand Test : DONE\n");

    nn::os::ThreadType uThread;


    nn::os::CreateThread(&uThread,
        ExtConUTest,
        NULL,
        g_UThreadStack,
        sizeof(g_UThreadStack),
        nn::os::DefaultThreadPriority);

    nn::os::StartThread(&uThread);

    nn::os::WaitThread(&uThread);

    nn::os::DestroyThread(&uThread);

    NN_LOG("Loopback Test : DONE\n");

    nn::pinmux::CloseSession(&uTxSession);

    nn::pinmux::Finalize();
    nn::gpio::Finalize();
    nn::uart::Finalize();
}


TEST(TargetSpecificNx, ExtConS)
{
    nn::pinmux::PinmuxSession sTxSession;
    nn::gpio::GpioPadSession gpioExtConDetS;

    nn::pinmux::Initialize();
    nn::gpio::Initialize();
    nn::uart::Initialize();

    nn::pinmux::OpenSession(&sTxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
    nn::pinmux::SetPinAssignment(&sTxSession, nn::pinmux::PinAssignment_ExtConTxGpio);

    nn::gpio::OpenSession(&gpioExtConDetS, nn::gpio::GpioPadName_ExtconDetS);
    nn::gpio::SetDirection(&gpioExtConDetS, nn::gpio::Direction_Input);

    while (1)
    {
        NN_LOG("Waiting for ExtCon detection... : ExtConS:%s\n",
            (nn::gpio::GetValue(&gpioExtConDetS) == nn::gpio::GpioValue_Low) ? "True" : "False");

        if (nn::gpio::GetValue(&gpioExtConDetS) == nn::gpio::GpioValue_Low)
        {
            NN_LOG("ExtConS is detected!\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    nn::gpio::CloseSession(&gpioExtConDetS);

    nn::pinmux::CloseSession(&sTxSession);

    nn::pinmux::OpenSession(&sTxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
    nn::pinmux::SetPinAssignment(&sTxSession, nn::pinmux::PinAssignment_ExtConTxUart);

    nnt::uart::extcon::TEST_BasicCommand(nn::uart::PortName_ExtConS, nn::uart::BaudRate_115200, nn::uart::FlowControlMode_Hardware);

    NN_LOG("BasicCommand Test : DONE\n");

    nn::os::ThreadType sThread;


    nn::os::CreateThread(&sThread,
        ExtConSTest,
        NULL,
        g_SThreadStack,
        sizeof(g_SThreadStack),
        nn::os::DefaultThreadPriority);

    nn::os::StartThread(&sThread);

    nn::os::WaitThread(&sThread);

    nn::os::DestroyThread(&sThread);

    NN_LOG("Loopback Test : DONE\n");

    nn::pinmux::CloseSession(&sTxSession);

    nn::pinmux::Finalize();
    nn::gpio::Finalize();
    nn::uart::Finalize();
}

TEST( TargetSpecificNx, ExtCon )
{
    nn::pinmux::PinmuxSession uTxSession;
    nn::pinmux::PinmuxSession sTxSession;
    nn::gpio::GpioPadSession gpioExtConDetU;
    nn::gpio::GpioPadSession gpioExtConDetS;

    nn::pinmux::Initialize();
    nn::gpio::Initialize();
    nn::uart::Initialize();

    nn::pinmux::OpenSession(&uTxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
    nn::pinmux::OpenSession(&sTxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
    nn::pinmux::SetPinAssignment(&uTxSession, nn::pinmux::PinAssignment_ExtConTxGpio);
    nn::pinmux::SetPinAssignment(&sTxSession, nn::pinmux::PinAssignment_ExtConTxGpio);

    nn::gpio::OpenSession(&gpioExtConDetU, nn::gpio::GpioPadName_ExtconDetU);
    nn::gpio::OpenSession(&gpioExtConDetS, nn::gpio::GpioPadName_ExtconDetS);
    nn::gpio::SetDirection(&gpioExtConDetU, nn::gpio::Direction_Input);
    nn::gpio::SetDirection(&gpioExtConDetS, nn::gpio::Direction_Input);

    while (1)
    {
        NN_LOG("Waiting for ExtConU detection... : ExtConU:%s, ExtConS:%s\n",
               (nn::gpio::GetValue(&gpioExtConDetU) == nn::gpio::GpioValue_Low) ? "True" : "False",
               (nn::gpio::GetValue(&gpioExtConDetS) == nn::gpio::GpioValue_Low) ? "True" : "False");

        if (nn::gpio::GetValue(&gpioExtConDetU) == nn::gpio::GpioValue_Low &&
            nn::gpio::GetValue(&gpioExtConDetS) == nn::gpio::GpioValue_Low)
        {
            NN_LOG("ExtConU and ExtConS are detected!\n");
            break;
        }

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    nn::gpio::CloseSession(&gpioExtConDetU);
    nn::gpio::CloseSession(&gpioExtConDetS);

    nn::pinmux::CloseSession(&uTxSession);
    nn::pinmux::CloseSession(&sTxSession);

    nn::pinmux::OpenSession(&uTxSession, nn::pinmux::AssignablePinGroupName_ExtConUTx);
    nn::pinmux::OpenSession(&sTxSession, nn::pinmux::AssignablePinGroupName_ExtConSTx);
    nn::pinmux::SetPinAssignment(&uTxSession, nn::pinmux::PinAssignment_ExtConTxUart);
    nn::pinmux::SetPinAssignment(&sTxSession, nn::pinmux::PinAssignment_ExtConTxUart);

    nnt::uart::extcon::TEST_BasicCommand(nn::uart::PortName_ExtConU, nn::uart::BaudRate_115200, nn::uart::FlowControlMode_Hardware);
    nnt::uart::extcon::TEST_BasicCommand(nn::uart::PortName_ExtConS, nn::uart::BaudRate_115200, nn::uart::FlowControlMode_Hardware);

    NN_LOG("BasicCommand Test : DONE\n");

    nn::os::ThreadType uThread;
    nn::os::ThreadType sThread;


    nn::os::CreateThread(&uThread,
                         ExtConUTest,
                         NULL,
                         g_UThreadStack,
                         sizeof(g_UThreadStack),
                         nn::os::DefaultThreadPriority);


    nn::os::CreateThread(&sThread,
                         ExtConSTest,
                         NULL,
                         g_SThreadStack,
                         sizeof(g_SThreadStack),
                         nn::os::DefaultThreadPriority);


    nn::os::StartThread(&uThread);
    nn::os::StartThread(&sThread);

    nn::os::WaitThread(&uThread);
    nn::os::WaitThread(&sThread);

    nn::os::DestroyThread(&uThread);
    nn::os::DestroyThread(&sThread);

    NN_LOG("Loopback Test : DONE\n");

    nn::pinmux::CloseSession(&uTxSession);
    nn::pinmux::CloseSession(&sTxSession);

    nn::pinmux::Finalize();
    nn::gpio::Finalize();
    nn::uart::Finalize();
}

