﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Includes.h"

//////////////////////////////////////////////////////////////////////////////
void Output::Initialize()
{
    m_ThreadIsUsed  = false;
    m_IsAttached    = false;

    // setup Isoc frams byte count for runtime
    for (uint32_t i = 0; i < MillisecondsPerFrameCount; i++)
    {
        m_OutputBytes[i] = SamplesPerMillisecondCount * OutputChannelsCount * BytesPerSampleCount;
    }
}


//////////////////////////////////////////////////////////////////////////////
void Output::Finalize()
{
}


//////////////////////////////////////////////////////////////////////////////
bool Output::IsAttached()
{
    return m_IsAttached;
}


//////////////////////////////////////////////////////////////////////////////
void Output::PostTransferRequest()
{
    nn::Result result = m_Interface.PostTransferAsync(
                                                            m_pMix->GetOutputFrame(),
                                                            m_OutputBytes,
                                                            MillisecondsPerFrameCount,
                                                            m_NextUsbFrame,
                                                            m_NextUsbFrame
                                                            );

    m_NextUsbFrame += MillisecondsPerFrameCount;

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("output error %d %d\n", result.GetModule(), result.GetDescription());
    }
}


//////////////////////////////////////////////////////////////////////////////
void Output::Resync()
{
NN_SDK_LOG("output %08x resync %d %d %d\n", this, m_NextUsbFrame, m_XferReport->context, m_OutCount);

    uint32_t pendingCount = (MillisecondsPerFrameCount * BuffersCount) - m_OutCount;

    while (pendingCount)
    {
        nn::Result result = m_Interface.GetXferReport(&m_OutCount, m_XferReport, MillisecondsPerFrameCount * BuffersCount);

        if (result.IsSuccess())
        {
            pendingCount -= m_OutCount;
        }
    }

    SetStartFrame();
    PostTransferRequest();
}


//////////////////////////////////////////////////////////////////////////////
void Output::CheckCompletions()
{
    m_OutCount = 0;

    while (m_OutCount == 0)
    {
        UacWaitForCompletion(&m_Interface);

        nn::Result result = m_Interface.GetXferReport(&m_OutCount, m_XferReport, MillisecondsPerFrameCount * BuffersCount);

        if (result.IsSuccess())
        {
            if (m_OutCount == MillisecondsPerFrameCount)  // Most commone case, 1 buffer completed
            {
                for (uint32_t i = 0; i < m_OutCount; i++)
                {
                    nn::usb::XferReport *pXferReport = &m_XferReport[i];

                    if (!pXferReport->result.IsSuccess())
                    {
                        // - If we have a frame error we need to resync
                        // - For anything else treat it as a data error
                        if (pXferReport->result <= nn::usb::ResultNotInFrameWindow())
                        {
                            Resync();

                            return;
                        }

                        // For output, if we have a data error it is already too late to do anything :0
                        // The output HW may have some sort of depop for data errors
                    }
                }
            }
            else if (m_OutCount > MillisecondsPerFrameCount)  // More than one buffer completed, time to re-sync
            {
                Resync();
            }

            // If m_OutCount == 0 then then this is an outstanding notification from when resync happened, ignore it and handle the next completion event
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void Output::SetStartFrame()
{
    // Setup current frame
    // - push frame out to n audio frames ahead of current frame
    // - place output frames on next NUM_MS_PER_FRAME + n boundary after
    // - process the output buffer n ms after input frames complete
    nn::Result result = m_Interface.GetCurrentFrame(&m_NextUsbFrame);

    if (result.IsSuccess())
    {
        m_NextUsbFrame += (MillisecondsPerFrameCount * 10) - (m_NextUsbFrame % MillisecondsPerFrameCount) + 1;
    }
}


//////////////////////////////////////////////////////////////////////////////
void Output::Processing()
{
    SetStartFrame();
    PostTransferRequest();

    while (TryWaitSystemEvent(m_Interface.GetStateChangeEvent()) == false)
    {
        PostTransferRequest();
        CheckCompletions();
    }

    m_Interface.Finalize();
    m_IsAttached = false;

    NN_SDK_LOG("Output %08x detached\n", this);
}


//////////////////////////////////////////////////////////////////////////////
void Output::WriteThread(void *pContext)
{
    reinterpret_cast<Output*>(pContext)->Processing();
}


//////////////////////////////////////////////////////////////////////////////
nn::Result Output::Start(nn::cduac::Host *pHost, nn::cduac::Parser *pParser, Mix *pMix, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    NN_SDK_ASSERT(pHost);
    NN_SDK_ASSERT(pParser);
    NN_SDK_ASSERT(pMix);
    NN_SDK_ASSERT(pInterfaceProfile);

    m_pHost     = pHost;
    m_pParser   = pParser;
    m_pMix      = pMix;

    nn::Result result = m_Interface.Initialize(m_pHost, pInterfaceProfile);

    if (result.IsSuccess())
    {
        // Find alt setting with 2 channels in PCM format
        uint8_t bAlternateSetting = 0;

        for (uint8_t i = 1; i < nn::usb::HsLimitMaxAltInterfaceSettingsCount; i++)
        {
            nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = pParser->GetInterface(pInterfaceProfile, i);

            if (pUsbInterfaceDescriptor)
            {
                nn::cduac::AudioStreamingGeneral    *pAudioStreamingGeneral     = pParser->GetAudioStreamingGeneral(pInterfaceProfile, i);
                nn::cduac::AudioStreamingFormatType *pAudioStreamingFormatType  = pParser->GetAudioStreamingFormatType(pInterfaceProfile, i);

                if (pAudioStreamingGeneral && pAudioStreamingFormatType)
                {
                    if (pAudioStreamingGeneral->wFormatTag == nn::cduac::AudioDataFormat::AudioDataFormat_Pcm) // PCM format
                    {
                        if (pAudioStreamingFormatType->bNrChannles == 2) // Stereo output, 2 channels L/R
                        {
                            bAlternateSetting = i;
                            break;
                        }
                    }
                }
            }
        }

        if (bAlternateSetting)
        {
            nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor = pParser->GetEndpoint(pInterfaceProfile, bAlternateSetting, 0);

            if (pUsbEndpointDescriptor)
            {
                //UacPrint(m_pParser, &m_Interface);        // export from ../UacCommon/UacPrint.cpp
                UacSetControls(m_pParser, &m_Interface);  // export from ../UacCommon/UacSetControls.cpp

                result = m_Interface.SetAltSetting(
                                                        bAlternateSetting,
                                                        pUsbEndpointDescriptor,
                                                        BuffersCount * MillisecondsPerFrameCount,
                                                        SamplesPerFrameCount * OutputChannelsCount * BuffersCount
                                                        );

                if (!result.IsSuccess())
                {
                    NN_SDK_LOG("SetAltSetting failed module %d description %d\n", result.GetModule(), result.GetDescription());
                }

                UacSetEndpoint(m_pParser, &m_Interface);  // set sample rate, export from ../UacCommon/UacSetEndpoint.cpp

                if (m_ThreadIsUsed)
                {
                    nn::os::WaitThread(&m_Thread);
                    nn::os::DestroyThread(&m_Thread);
                }

                nn::os::CreateThread(
                                        &m_Thread,
                                        this->WriteThread,
                                        this,
                                        m_ThreadStack,
                                        sizeof(m_ThreadStack),
                                        nn::os::HighestThreadPriority
                                        );

                nn::os::StartThread(&m_Thread);

                m_ThreadIsUsed  = true;
                m_IsAttached    = true;
            }
        }
    }

    return result;
}

