﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "Includes.h"

//////////////////////////////////////////////////////////////////////////////
void Input::Initialize()
{
    m_InputFrames   = 0;
    m_ThreadIsUsed  = false;
    m_IsAttached    = false;

    m_Depop.Initialize();

    nn::os::InitializeMutex(&m_Mutex, false, 0);

    // below is stuff that never changes at runtime, initialize once
    // setup Isoc frams byte count for runtime
    for (uint32_t i = 0; i < MillisecondsPerFrameCount; i++)
    {
        m_InputBytes[i] = SamplesPerMillisecondCount * InputChannelsCount * BytesPerSampleCount;
    }

    // setup buffer pointers for completion context
    for (uint32_t i = 0; i < BuffersCount; i++)
    {
        m_InputContext[i].pBuffer = m_InputBuffer[i];
    }
}


//////////////////////////////////////////////////////////////////////////////
void Input::Finalize()
{
    m_Depop.Finalize();
}


//////////////////////////////////////////////////////////////////////////////
bool Input::IsAttached()
{
    return m_IsAttached;
}


//////////////////////////////////////////////////////////////////////////////
int16_t * Input::GetInputFrame()
{
    bool depop = true;

    if (m_IsAttached)
    {
        nn::os::LockMutex(&m_Mutex);

        if (m_InputFrames == 1)
        {
            depop = false;
        }

        m_InputFrames = 0;

        nn::os::UnlockMutex(&m_Mutex);
    }

    return m_Depop.ProcessFrame(m_InputSamples, depop);   // bypass the depop for now
}


//////////////////////////////////////////////////////////////////////////////
void Input::PostTransferRequest()
{
    int32_t bufferIndex = m_InputCounter % BuffersCount;
    InputContext *pInputContext = &m_InputContext[bufferIndex];

    pInputContext->startFrame = m_NextUsbFrame;

    nn::Result result = m_Interface.PostTransferAsync(
                                                            m_InputBuffer[bufferIndex],
                                                            m_InputBytes,
                                                            MillisecondsPerFrameCount,
                                                            m_NextUsbFrame,
                                                            bufferIndex
                                                            );

    m_NextUsbFrame += MillisecondsPerFrameCount;
    m_InputCounter++;

    if (!result.IsSuccess())
    {
        NN_SDK_LOG("input %08x error %d %d\n", this, result.GetModule(), result.GetDescription());
    }
}


//////////////////////////////////////////////////////////////////////////////
void Input::Resync()
{
NN_SDK_LOG("input %08x resync %d %d %d\n", this, m_NextUsbFrame, m_XferReport->context, m_OutCount);

    uint32_t pendingCount = (MillisecondsPerFrameCount * BuffersCount) - m_OutCount;

    while (pendingCount)
    {
        nn::Result result = m_Interface.GetXferReport(&m_OutCount, m_XferReport, MillisecondsPerFrameCount * BuffersCount);

        if (result.IsSuccess())
        {
            pendingCount -= m_OutCount;
        }
    }

    SetStartFrame();
    PostTransferRequest();
}


//////////////////////////////////////////////////////////////////////////////
void Input::CheckCompletions()
{
    m_OutCount = 0;

    while (m_OutCount == 0)
    {
        UacWaitForCompletion(&m_Interface);

        nn::Result result = m_Interface.GetXferReport(&m_OutCount, m_XferReport, MillisecondsPerFrameCount * BuffersCount);

        if (result.IsSuccess())
        {
            if (m_OutCount == MillisecondsPerFrameCount)  // Most commone case, 1 buffer completed
            {
                uint32_t dataError = 0;
                InputContext *pInputContext = &m_InputContext[m_XferReport[0].context];

                for (uint32_t i = 0; i < m_OutCount; i++)
                {
                    nn::usb::XferReport *pXferReport = &m_XferReport[i];

                    if (!pXferReport->result.IsSuccess())
                    {
                        // If we have a frame error we need to resync
                        if (pXferReport->result <= nn::usb::ResultNotInFrameWindow())
                        {
                            DepopFrame();
                            Resync();

                            return;
                        }
                        // For anything else treat it as a data error
                        else
                        {
                            dataError++;
                        }
                    }
                }

                if (dataError == 0) // No data error
                {
                    nn::os::LockMutex(&m_Mutex);
                    memcpy(m_InputSamples, pInputContext->pBuffer, SamplesPerFrameCount * BytesPerSampleCount);
                    m_InputFrames++;
                    nn::os::UnlockMutex(&m_Mutex);
                }
                else // Error, depop input
                {
                    DepopFrame();
                }
            }
            else if (m_OutCount > MillisecondsPerFrameCount)  // More than one buffer completed, time to re-sync
            {
                DepopFrame();
                Resync();
            }

            // If m_OutCount == 0 then then this is an outstanding notification from when resync happened, ignore it and handle the next completion event
        }
    }
}


//////////////////////////////////////////////////////////////////////////////
void Input::SetStartFrame()
{
    // Setup current frame
    // - push frame out to 10 ms ahead of current frame
    // - place input frames on next NUM_MS_PER_FRAME boundary after
    nn::Result result = m_Interface.GetCurrentFrame(&m_NextUsbFrame);

    if (result.IsSuccess())
    {
        m_NextUsbFrame += (MillisecondsPerFrameCount * 10) - (m_NextUsbFrame % MillisecondsPerFrameCount);
    }
}


//////////////////////////////////////////////////////////////////////////////
void Input::Processing()
{
    SetStartFrame();
    PostTransferRequest();

    while (TryWaitSystemEvent(m_Interface.GetStateChangeEvent()) == false)
    {
        PostTransferRequest();
        CheckCompletions();
    }

    m_Interface.Finalize();
    m_IsAttached = false;

    NN_SDK_LOG("Input %08x detached\n", this);
}


//////////////////////////////////////////////////////////////////////////////
void Input::DepopFrame()
{
    nn::os::LockMutex(&m_Mutex);

    memset(m_InputSamples, 0, SamplesPerFrameCount * BytesPerSampleCount);
    m_InputFrames = 0; // Force the depopper to depop this frame

    nn::os::UnlockMutex(&m_Mutex);
}


//////////////////////////////////////////////////////////////////////////////
void Input::ReadThread(void *pContext)
{
    reinterpret_cast<Input*>(pContext)->Processing();
}


//////////////////////////////////////////////////////////////////////////////
nn::Result Input::Start(nn::cduac::Host *pHost, nn::cduac::Parser *pParser, nn::cduac::InterfaceProfile *pInterfaceProfile)
{
    NN_SDK_ASSERT(pHost);
    NN_SDK_ASSERT(pParser);
    NN_SDK_ASSERT(pInterfaceProfile);

    m_pHost     = pHost;
    m_pParser   = pParser;

    nn::Result result = m_Interface.Initialize(m_pHost, pInterfaceProfile);

    if (result.IsSuccess())
    {
        // Find alt setting with 1 channel in PCM format
        uint8_t bAlternateSetting = 0;

        for (uint8_t i = 1; i < nn::usb::HsLimitMaxAltInterfaceSettingsCount; i++)
        {
            nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = pParser->GetInterface(pInterfaceProfile, i);

            if (pUsbInterfaceDescriptor)
            {
                nn::cduac::AudioStreamingGeneral    *pAudioStreamingGeneral     = pParser->GetAudioStreamingGeneral(pInterfaceProfile, i);
                nn::cduac::AudioStreamingFormatType *pAudioStreamingFormatType  = pParser->GetAudioStreamingFormatType(pInterfaceProfile, i);

                if (pAudioStreamingGeneral && pAudioStreamingFormatType)
                {
                    if (pAudioStreamingGeneral->wFormatTag == nn::cduac::AudioDataFormat::AudioDataFormat_Pcm) // PCM format
                    {
                        if (pAudioStreamingFormatType->bNrChannles == 1) // Mono output, 1 channel
                        {
                            bAlternateSetting = i;
                            break;
                        }
                    }
                }
            }
        }

        if (bAlternateSetting)
        {
            nn::usb::UsbEndpointDescriptor *pUsbEndpointDescriptor = pParser->GetEndpoint(pInterfaceProfile, bAlternateSetting, 0);

            if (pUsbEndpointDescriptor)
            {
                //UacPrint(m_pParser, &m_Interface);        // prints out descriptors, export from ../UacCommon/UacPrint.cpp
                UacSetControls(m_pParser, &m_Interface);  // set "Audio Control" controls, export from ../UacCommon/UacSetControls.cpp

                result = m_Interface.SetAltSetting(
                                                        bAlternateSetting,                                          // alternate setting to set
                                                        pUsbEndpointDescriptor,                                     // pointer to UsbEndpointDescriptor
                                                        BuffersCount * MillisecondsPerFrameCount,                   // maximum URB count
                                                        SamplesPerFrameCount * BytesPerSampleCount * BuffersCount   // maximum outstanding transfer size total
                                                        );

                if (!result.IsSuccess())
                {
                    NN_SDK_LOG("SetAltSetting failed module %d description %d\n", result.GetModule(), result.GetDescription());
                }

                UacSetEndpoint(m_pParser, &m_Interface);  // set sample rate, export from ../UacCommon/UacSetEndpoint.cpp

                if (m_ThreadIsUsed)
                {
                    nn::os::WaitThread(&m_Thread);
                    nn::os::DestroyThread(&m_Thread);
                }

                // reset runtime params
                m_InputCounter  = 0;
                m_InputFrames   = 0;

                nn::os::CreateThread(
                                        &m_Thread,
                                        this->ReadThread,
                                        this,
                                        m_ThreadStack,
                                        sizeof(m_ThreadStack),
                                        nn::os::HighestThreadPriority
                                        );

                nn::os::StartThread(&m_Thread);

                m_ThreadIsUsed  = true;
                m_IsAttached    = true;
            }
        }
    }

    return result;
}

