﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/os.h>
#include <nn/nn_Abort.h>
#include <nn/nn_SdkLog.h>
#include <nn/nn_Common.h>
#include <nn/nn_Log.h>

#include <nn/cduac/cduac_Spec.h>
#include <nn/cduac/cduac_Api.h>

#include "./UacCommon.h"

//////////////////////////////////////////////////////////////////////////////
void UacSetControls(nn::cduac::Parser *pParser, nn::cduac::Interface *pInterface)
{
    NN_SDK_LOG("%s\n", __FUNCTION__);

    nn::cduac::AudioControlInputTerminal  *pInputTerminal = pParser->GetInputTerminal(pInterface->GetInterfaceProfile());
    nn::cduac::AudioControlFeatureUnit    *pFeatureUnit   = pParser->GetFeatureUnit(pInterface->GetInterfaceProfile());

    if (pFeatureUnit && pInputTerminal)
    {
        uint8_t *pBmaControls = pFeatureUnit->bmaControls;

        for (uint8_t i = 0; i <= pInputTerminal->bNrChannels; i++)
        {
            uint16_t controls = 0;

            for (uint8_t j = 0; j < pFeatureUnit->bControlSize; j++)
            {
                controls <<= 8;
                controls |= *pBmaControls;

                pBmaControls++;
            }

            // We only care about mute and volume for this test
            if (controls & nn::cduac::Control_Mute)
            {
                NN_SDK_LOG("Channel %d Mute Control\n", i);

                uint8_t data;
                nn::Result result = pInterface->GetAudioControl(
                                                        nn::cduac::Request_GetCur,
                                                        (nn::cduac::Control_Mute << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(data),
                                                        &data
                                                        );

                if (result.IsSuccess())
                {
                    NN_SDK_LOG("mute %02x\n", data);

                    // If mic is muted, unmute the mic
                    if (data)
                    {
                        data = 0;

                        result = pInterface->SetAudioControl(
                                                        nn::cduac::Request_GetCur,
                                                        (nn::cduac::Control_Mute << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(data),
                                                        &data
                                                        );
                        if (result.IsSuccess())
                        {
                            NN_SDK_LOG("mute set to %d\n", data);
                        }
                    }
                }
            }

            if (controls & nn::cduac::Control_Volume)
            {
                NN_SDK_LOG("Channel %d Volume Control\n", i);

                int16_t vCur, vMin, vMax, vRes;
                nn::Result result;

                result = pInterface->GetAudioControl(
                                                        nn::cduac::Request_GetCur,
                                                        (nn::cduac::Control_Volume << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(vCur),
                                                        &vCur
                                                        );

                result = pInterface->GetAudioControl(
                                                        nn::cduac::Request_GetMin,
                                                        (nn::cduac::Control_Volume << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(vMin),
                                                        &vMin
                                                        );

                result = pInterface->GetAudioControl(
                                                        nn::cduac::Request_GetMax,
                                                        (nn::cduac::Control_Volume << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(vMax),
                                                        &vMax
                                                        );

                result = pInterface->GetAudioControl(
                                                        nn::cduac::Request_GetRes,
                                                        (nn::cduac::Control_Volume << 8) | i,
                                                        pFeatureUnit->bUnitId,
                                                        sizeof(vRes),
                                                        &vRes
                                                        );

                NN_SDK_LOG("Cur Volume %f dB\n", static_cast<float>(vCur) / vRes);
                NN_SDK_LOG("Max Volume %f dB\n", static_cast<float>(vMax) / vRes);
                NN_SDK_LOG("Min Volume %f dB\n", static_cast<float>(vMin) / vRes);


                // Lets set volume to max
                if (vCur != vMax)
                {
                    result = pInterface->SetAudioControl(
                                                            nn::cduac::Request_SetCur,
                                                            (nn::cduac::Control_Volume << 8) | i,
                                                            pFeatureUnit->bUnitId,
                                                            sizeof(vMax),
                                                            &vMax
                                                            );

                    if (result.IsSuccess())
                    {
                        NN_SDK_LOG("Cur Volume set to %f dB\n", static_cast<float>(vMax) / vRes);
                    }
                }
            }
        }
    }
} // NOLINT(impl/function_size)

