﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*
 * HW評価アプリ：温度センサー：データロガー
 * 参考： http://spdlybra.nintendo.co.jp/confluence/pages/viewpage.action?pageId=113804557
 */

#include <cstdlib> // new, free
#include <vector>
#include <nn/os.h>
#include <nn/init.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/nn_Log.h>

#include <nn/font.h>
#include <nn/fs.h>
#include <nn/fs/fs_SdCardForDebug.h> // SD への FAT アクセス
#include <nn/hid.h>

#include <nn/time/time_Api.h>
#include <nn/time/time_TimeZoneApi.h>
#include <nn/time/time_StandardUserSystemClock.h>
#include <nv/nv_MemoryManagement.h>

#include <eval_Screen.h>

#include <nnd/tmp451/tmp451.h>

namespace {

const int AppVerMajor = 1;
const int AppVerMinor = 1;

struct ConfigVal{
    uint8_t                         id;
    ::nnd::tmp451::MeasurementRange range;
    int                             rate;
    bool                            timeout;

    uint8_t                         hys;
    ::nnd::tmp451::AlertConsecutive consec;
    ::nnd::tmp451::DigitalFilter    filter;
    ::nnd::tmp451::Temperature      offset;
    int8_t                          eta;
};
ConfigVal g_ConfigVlaue;

::nn::hid::TouchScreenState<nn::hid::TouchStateCountMax> g_TouchState;

static int g_FileCount = 0;
static bool g_IsLogfileWriting = false;
static bool g_Finalize = false;
static bool g_Push = false;
static bool g_SwitchMode = false;
static bool g_IsFirstTouch = true;
static ::nn::os::Tick g_BeginTouchTick;
static ::nn::os::Tick g_BeginMeasureTick;

static int64_t g_WriteOffset = 0;

// マウントするファイルシステムに対する名前
::nn::fs::FileHandle g_FileHandle;
const ::std::string SdcardMountName = "sd";

struct LogDate {
    ::nnd::tmp451::Temperature  local;
    ::nnd::tmp451::Temperature  remote;
    ::std::string               dateStr;
    ::std::string               timeStr;
};

// 画面解像度
//const float DisplayWidth = 1280;
const float DisplayHeight = 720;

// Defaultとするフォントサイズ
const float FontSizeDefault = 33.0f;
const float FontSizeLarge = FontSizeDefault * 1.5f;

struct Position {
    float x;
    float y;
};

int         WaitTimeMs = 0;     // 計測時間に併せて待つための時間[ミリ秒]

// 枠線 --------------------------------
// 縦線の X 座標
const float     PosFrameVerticalX           =  900.0f;

// タイトルの下枠
const Position  PosFrameHorizontalTitle     = { 0.0f, FontSizeDefault};

// 縦線から右に線を引く時のマージンとX座標
const float     PosFrameVerticalMargine     = FontSizeDefault * 0.36f; // 実測による比率
const float     PosFrameHorizontalRightX    = PosFrameVerticalX + PosFrameVerticalMargine;

// 縦線から右に引く枠線
const Position  PosFrameHorizontalStartStop = { PosFrameHorizontalRightX, 590.0f};
//const Position  PosFrameHorizontalConfig    = { PosFrameHorizontalRightX, 200.0f};

// テキスト ----------------------------
// 外周からの余白
const float MarginTextFromFrame = 5.0f;

const float PosTextHorizontalRightX = PosFrameHorizontalRightX + 10.0f;

// タイトル
const Position PosTextTitle = {
        MarginTextFromFrame,
        MarginTextFromFrame };

// 日時
const Position PosTextDate = {
        PosTextHorizontalRightX,
        MarginTextFromFrame };

// START/STOP
const Position PosTextStartStop = {
        PosTextHorizontalRightX,
        PosFrameHorizontalStartStop.y + (DisplayHeight - PosFrameHorizontalStartStop.y - FontSizeLarge) / 2 };

// コンフィグ値
const Position PosTextConfig = {
        PosTextHorizontalRightX,
        PosFrameHorizontalTitle.y + FontSizeDefault / 2};

// ログデータの項目名
const float PosTextLogTitleY = PosFrameHorizontalTitle.y + FontSizeDefault * 3 / 2;
const float PosTextLogDataY = PosFrameHorizontalTitle.y + FontSizeDefault * 5 / 2;

const float PosTextDateX = MarginTextFromFrame; // 日付
const float PosTextTimeX = PosTextDateX + 180;
const float PosTextLocal = PosTextTimeX + 190;
const float PosTextRemote = PosTextLocal + 180;


// フラグ -----------------------
bool    g_IsSdMounted = false;

// For nv::InitializeGraphics
void* Allocate(size_t size, size_t alignment, void*)
{
    return aligned_alloc(alignment, size);
}

void Free(void* addr, void*)
{
    free(addr);
}

void* Reallocate(void* addr, size_t newSize, void*)
{
    return realloc(addr, newSize);
}
}


::std::string GetCurrentTimeStrForFileName(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.year) + ::std::to_string(calendarTime.month) + ::std::to_string(calendarTime.day)
        + "_" + ::std::to_string(calendarTime.hour) + ::std::to_string(calendarTime.minute) + ::std::to_string(calendarTime.second);

    return out;
}

::std::string GetCurrentDateStr(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.year) + "/" + ::std::to_string(calendarTime.month) + "/" + ::std::to_string(calendarTime.day);

    return out;
}

::std::string GetCurrentTimeStr(::nn::time::PosixTime posixTime)
{
    nn::time::CalendarTime calendarTime;
    nn::time::CalendarAdditionalInfo calendarAdditionalInfo;
    NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::ToCalendarTime(&calendarTime, &calendarAdditionalInfo, posixTime) );
//    NN_LOG("Current calendar(%s): %04d/%02d/%02d %02d:%02d:%02d\n",
//        calendarAdditionalInfo.timeZone.standardTimeName,
//        calendarTime.year, calendarTime.month, calendarTime.day, calendarTime.hour, calendarTime.minute, calendarTime.second);

    ::std::string out = ::std::to_string(calendarTime.hour) + ":" + ::std::to_string(calendarTime.minute) + ":" + ::std::to_string(calendarTime.second);

    return out;
}


void DrawHorizontalLine(nn::gfx::util::DebugFontTextWriter* pTextWriter, Position pos, int size, uint32_t flag)
{
    float width = pTextWriter->CalculateStringWidth("─");

//    pTextWriter->SetDrawFlag(flag);

    pTextWriter->SetCursor(pos.x, pos.y);
    for(auto i = 0; i < size ; i++)
    {
        pTextWriter->Print("─");

        if (flag & ::nn::font::TextWriter::PositionFlag_HorizontalOriginRight)
        {
            pos.x -= width;
            pTextWriter->SetCursor(pos.x, pos.y);
        }
    }
}

void DrawVerticalLine(nn::gfx::util::DebugFontTextWriter* pTextWriter, Position pos, float size)
{
    float height = pTextWriter->GetLineHeight() * 0.76f; // 実測による比率
    for (auto i = 0; i < size; i ++)
    {
        pTextWriter->SetCursor(pos.x, pos.y);
        pTextWriter->Print("│");
        pos.y += height;
    }
}


void DrawOutline(nn::gfx::util::DebugFontTextWriter* pTextWriter)
{
    // 縦線 ----------------------
    Position pos;
    uint32_t flag = ::nn::font::TextWriter::DefaultDrawFlag;
    pTextWriter->SetFontSize(FontSizeDefault);
    pTextWriter->SetTextColor(::nnt::eval::White);

    pos = { PosFrameVerticalX, 0.0f};
    DrawVerticalLine(pTextWriter, pos, 30);

    // 横線 ----------------------
    // タイトル&時計の枠
    pos = PosFrameHorizontalTitle;
    DrawHorizontalLine(pTextWriter, pos, 50, flag);


    pTextWriter->SetTextColor(::nnt::eval::Gray);
    // 表の項目名の枠
    pos.x = PosFrameVerticalX + PosFrameVerticalMargine;
    pos.y += pTextWriter->GetFontHeight() * 2;
    flag = ::nn::font::TextWriter::PositionFlag_HorizontalOriginRight;
    DrawHorizontalLine(pTextWriter, pos, 40, flag);
    flag = ::nn::font::TextWriter::DefaultDrawFlag;

//    // デバイス動作設定値の枠
//    pos = PosFrameHorizontalConfig;
//    DrawHorizontalLine(textWriter, pos, 15, flag);

    // 制御ボタンの枠
    pTextWriter->SetTextColor(::nnt::eval::White);
    pos = PosFrameHorizontalStartStop;
    DrawHorizontalLine(pTextWriter, pos, 15, flag);
}


void DisplayTextOnLcd(::nnt::eval::Screen* pScreen, ::std::vector<LogDate>& inData)
{
//    NN_LOG("--- Display text on LCD.\n");

    // LCD へのテキスト表示
    nn::gfx::util::DebugFontTextWriter* pTextWriter = pScreen->GetDebugFontTextWriter();

    // 枠線
    DrawOutline(pTextWriter);

    Position pos;

    //----------- テキスト
    pTextWriter->SetTabWidth(2);
    pTextWriter->SetTextColor(::nnt::eval::White);

    // タイトル
    pos = PosTextTitle;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->Print("[HW Eval: TS DataLogger (Ver.%d.%d)] ", AppVerMajor, AppVerMinor);

    // SDのマウント状態
    if (!g_IsSdMounted)
    {
        pTextWriter->SetTextColor(::nnt::eval::Red);
        pTextWriter->Print("SD card is NOT mounted");
        pTextWriter->SetTextColor(::nnt::eval::White);
    }
    else
    {
        pTextWriter->SetTextColor(::nnt::eval::Green);
        pTextWriter->Print("SD card is mounted");
        pTextWriter->SetTextColor(::nnt::eval::White);
    }

    // 時計
    pos = PosTextDate;
    pTextWriter->SetCursor(pos.x, pos.y);
    ::nn::time::PosixTime posixTime;
    ::nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    ::std::string date = GetCurrentDateStr(posixTime);
    ::std::string time = GetCurrentTimeStr(posixTime);
    pTextWriter->Print("%s %s", date.c_str(), time.c_str());

    // デバイス設定値
    pTextWriter->SetFontSize(FontSizeDefault);
    pos = PosTextConfig;
    pTextWriter->SetCursor(pos.x, pos.y);
//    pTextWriter->Print("ID\nRange\nRate\nTimeout\nHysterisis\nConsective\nFilter\nOffset\nEta-factor\n");
    pTextWriter->Print("ID\n");
    pTextWriter->Print("Range\n");
    pTextWriter->Print("Rate [ms]\n");
    pTextWriter->Print("Timeout\n");
    pTextWriter->Print("Hysterisis\n");
    pTextWriter->Print("Consective\n");
    pTextWriter->Print("Filter\n");
    pTextWriter->Print("Offset [℃]\n");
    pTextWriter->Print("Eta-factor\n");

    pos = PosTextConfig;
    pos.x += pTextWriter->CalculateStringWidth("Consective ");
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->Print(": 0x%X\n", g_ConfigVlaue.id);
    pTextWriter->Print(": %s\n", (g_ConfigVlaue.range == ::nnd::tmp451::MeasurementRange::Standard  ? "Standard": "Extended"));
    pTextWriter->Print(": %.4f\n", (1000.0f / pow(2, g_ConfigVlaue.rate)));
    pTextWriter->Print(": %s\n", (g_ConfigVlaue.timeout ? "true" : "false"));
    pTextWriter->Print(": %d\n", g_ConfigVlaue.hys);
    pTextWriter->Print(": %d\n", (g_ConfigVlaue.consec == ::nnd::tmp451::AlertConsecutive::Consecutive1  ? 1:
                (g_ConfigVlaue.consec == ::nnd::tmp451::AlertConsecutive::Consecutive2  ? 2:
                (g_ConfigVlaue.consec == ::nnd::tmp451::AlertConsecutive::Consecutive3  ? 3: 4))));
    pTextWriter->Print(": %s\n", (g_ConfigVlaue.filter == ::nnd::tmp451::DigitalFilter::FilterOff  ? "Off":
                (g_ConfigVlaue.filter == ::nnd::tmp451::DigitalFilter::Filter4Results  ? "4": "8")));
    pTextWriter->Print(": %d.%d\n", g_ConfigVlaue.offset.integer, g_ConfigVlaue.offset.decimal);
    pTextWriter->Print(": %.6f\n", (1.008f * 2088.f) / (2088 + g_ConfigVlaue.eta));

    // 列名称
    pTextWriter->SetFontSize(FontSizeDefault);
    pTextWriter->SetTextColor(::nnt::eval::Gray);
    pTextWriter->SetCursor(PosTextDateX, PosTextLogTitleY);
    pTextWriter->Print("%s", "Date");
    pTextWriter->SetCursorX(PosTextTimeX);
    pTextWriter->Print("%s", "Time");
    pTextWriter->SetCursorX(PosTextLocal);
    pTextWriter->Print("%s", "Local [℃]");
    pTextWriter->SetCursorX(PosTextRemote);
    pTextWriter->Print("%s", "Remote [℃]");
    pTextWriter->SetTextColor(::nnt::eval::White);


    // ログデータ
    // 計測停止状態ならグレーにする
    if (!g_IsLogfileWriting)
    {
        pTextWriter->SetTextColor(::nnt::eval::Gray);
    }
    // TODO: vector で過去のログも出力したい
    pTextWriter->SetCursor(PosTextDateX, PosTextLogDataY);
    pTextWriter->Print("%s", inData[0].dateStr.c_str());
    pTextWriter->SetCursorX(PosTextTimeX);
    pTextWriter->Print("%s", inData[0].timeStr.c_str());
    pTextWriter->SetCursorX(PosTextLocal);
    pTextWriter->Print("%3d.%04d", inData[0].local.integer, inData[0].local.decimal);
    pTextWriter->SetCursorX(PosTextRemote);
    pTextWriter->Print("%3d.%04d", inData[0].remote.integer, inData[0].remote.decimal);
    pTextWriter->SetTextColor(::nnt::eval::White);

    // 操作ボタン
    pos = PosTextStartStop;
    pTextWriter->SetCursor(pos.x, pos.y);
    pTextWriter->SetFontSize(FontSizeLarge);
    if (g_IsLogfileWriting)
    {
        pTextWriter->SetTextColor(::nnt::eval::Red);
        pTextWriter->Print("Tap to STOP\n");
//        NN_LOG("Tap to STOP\n");
    }
    else
    {
        pTextWriter->SetTextColor(::nnt::eval::Blue);
        pTextWriter->Print("Tap to START\n");
//        NN_LOG("Tap to START\n");
    }
    pTextWriter->SetFontSize(FontSizeDefault);
    pTextWriter->SetTextColor(::nnt::eval::White);

    // 描画
    pScreen->Draw();
}

void DetectTouchState()
{
    //-----------------------------------
    // タッチ状態の確認
    //-----------------------------------
    // 計測状態の切り替え要求を確認
    nn::hid::GetTouchScreenState(&g_TouchState);
    // 始めてタッチを検出した時に計測開始
    if (g_IsFirstTouch && ((g_TouchState.count == 1) || (g_TouchState.count == 4)))
    {
        g_BeginTouchTick = ::nn::os::GetSystemTick();
        g_IsFirstTouch = false;
    }
    if (g_TouchState.count == 0)
    {
        g_IsFirstTouch = true;
        g_Push = false;
    }
    else
    {
        g_IsFirstTouch = false;
        auto endTick = ::nn::os::GetSystemTick();
        auto totalTick = endTick - g_BeginTouchTick;
        auto durationMsec = ::nn::os::ConvertToTimeSpan(totalTick).GetMilliSeconds();
        if (g_TouchState.count == 1)
        {
            // タッチが行われており「START/STOP」の位置なら
            if (!g_Push && g_TouchState.touches[0].x > PosFrameHorizontalStartStop.x
                    && g_TouchState.touches[0].y > PosFrameHorizontalStartStop.y)
            {
                if (durationMsec > 100)
                {
                    NN_LOG("Touch!! START/STOP button.\n");
                    g_Push = true;
                    g_SwitchMode = true;
                }
            }
        }
        else if (g_TouchState.count == 4 && !g_IsLogfileWriting)
        {
            // 4本の指で5秒タッチされたらアプリ終了
            if (durationMsec > 5000)
            {
                NN_LOG("Touch!! Finalize Application.\n");
                g_Finalize = true;
            }
        }
    }
}

void OpenOutputFile()
{
    // 時間からファイル名を生成        // 時間を取得
    ::nn::time::PosixTime posixTime;
    ::nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime);
    ::std::string timeStr = GetCurrentTimeStrForFileName(posixTime);
    const ::std::string OutFileName = "AlsDataLogger-" + timeStr + "-" + ::std::to_string(g_FileCount) +".txt";

    // 出力するファイルのパス
    const ::std::string filePath = SdcardMountName + ":/" + OutFileName;
    NN_LOG("FilePath: %s\n", filePath.c_str());

    ::nn::fs::DirectoryEntryType type;
    auto result = ::nn::fs::GetEntryType(&type, filePath.c_str());
    if (::nn::fs::ResultPathNotFound().Includes(result))
    {
        result = ::nn::fs::CreateFile(filePath.c_str(), 0);
        NN_ASSERT(result.IsSuccess(), "Failed to create file");
    }
    else
    {
        NN_ASSERT(result.IsSuccess(), "Failed to get entry type");
    }

    // ファイルをオープン
    result = ::nn::fs::OpenFile(&g_FileHandle, filePath.c_str(),
            ::nn::fs::OpenMode_Write | ::nn::fs::OpenMode_AllowAppend);
    NN_ASSERT(result.IsSuccess(), "Failed to open log");
}

void CloseOutputFile()
{
    // ファイルをクローズ
    ::nn::fs::CloseFile(g_FileHandle);
    g_WriteOffset = 0;
    g_FileCount++;
}


extern "C" void nninitStartup()
{
    const size_t MemoryHeapSize = 512 * 1024 * 1024;
    const size_t MallocHeapSize = 256 * 1024 * 1024;

    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::SetMemoryHeapSize(MemoryHeapSize));

    uintptr_t address = uintptr_t();
    NN_ABORT_UNLESS_RESULT_SUCCESS(
        ::nn::os::AllocateMemoryBlock(&address, MallocHeapSize));

    ::nn::init::InitializeAllocator(
        reinterpret_cast<void*>(address), MallocHeapSize);
}


extern "C" void nnMain()
{
    NN_LOG("/// START ///\n");

    const size_t NvMemorySize = 8 * 1024 * 1024;
    nv::SetGraphicsAllocator(Allocate, Free, Reallocate, NULL);
    nv::SetGraphicsDevtoolsAllocator(Allocate, Free, Reallocate, NULL);
    nv::InitializeGraphics(std::malloc(NvMemorySize), NvMemorySize);

    // フォント表示ライブラリの初期化
    ::nnt::eval::Screen* pScreen = new ::nnt::eval::Screen();
    pScreen->Initialize();

    // ライブラリの初期化
    ::nnd::tmp451::Initialize();

    // 操作制御用 hid の初期化
    ::nn::hid::InitializeTouchScreen();

    ::nn::time::Initialize();

    // SD カードをマウント
    auto result = ::nn::fs::MountSdCardForDebug(SdcardMountName.c_str());
    if (!result.IsSuccess())
    {
        NN_LOG("!!!!! Failed to mount SD card. !!!!!!\n");
        g_IsSdMounted = false;
    }
    else
    {
        g_IsSdMounted = true;
    }

    // 各種 計測設定値の取得
    ::nnd::tmp451::ReadManufacturerId(&g_ConfigVlaue.id);
    ::nnd::tmp451::GetMeasurementRange(&g_ConfigVlaue.range);
    ::nnd::tmp451::GetMeasurementRateBy2ToThePowerN(&g_ConfigVlaue.rate);
    ::nnd::tmp451::GetBusTimeoutFunctionEnabled(&g_ConfigVlaue.timeout);
    ::nnd::tmp451::GetThermHysterisis(&g_ConfigVlaue.hys);
    ::nnd::tmp451::GetConsecutiveAlert(&g_ConfigVlaue.consec);
    ::nnd::tmp451::GetRemoteDigitalFilterControl(&g_ConfigVlaue.filter);
    ::nnd::tmp451::GetRemoteTemperatureOffset(&g_ConfigVlaue.offset);
    ::nnd::tmp451::GetRemoteEtaFactorCorrectionValueByN(&g_ConfigVlaue.eta);

    WaitTimeMs = ceil(1000.0f / (2 ^ g_ConfigVlaue.rate));

    ::nnd::tmp451::SetMeasurementMode(::nnd::tmp451::MeasurementMode::Continuous);

    // TODO: 時間情報も含んだvectorにする
    ::std::vector<LogDate> dataBuffer;
    dataBuffer.reserve(30);
    // 時計表示部分のバッファの初期化
    dataBuffer[0].dateStr = "0000/00/00";
    dataBuffer[0].timeStr = "00:00:00";

    while (!g_Finalize)
    {
        //-----------------------------------
        // タッチ状態の確認
        //-----------------------------------
        // 計測状態の切り替え要求を確認
        DetectTouchState();

        //-------------------------------------
        // ログファイルの出力状態を切り替え
        //-------------------------------------
        if (g_Push && g_SwitchMode)
        {
            // フラグをクリア
            g_SwitchMode = false;

            // モードを反転
            g_IsLogfileWriting = !g_IsLogfileWriting;
            if (g_IsLogfileWriting)
            {
                g_BeginMeasureTick = ::nn::os::GetSystemTick();
            }

            // SDカードがマウントされている場合のみログファイルをオープン
            if (g_IsLogfileWriting && g_IsSdMounted)
            {
                OpenOutputFile();
            }
            else
            {
                CloseOutputFile();
            }
        }

        // --------------------------------
        // センサー値の取得とログ出力
        // --------------------------------
        if (g_IsLogfileWriting)
        {
            auto endTick = ::nn::os::GetSystemTick();
            auto totalTick = endTick - g_BeginMeasureTick;
            auto durationMsec = ::nn::os::ConvertToTimeSpan(totalTick).GetMilliSeconds();

            // posixTime を WaitTimeMs で割って
            if (durationMsec > WaitTimeMs)
            {
                g_BeginMeasureTick = ::nn::os::GetSystemTick();

                // センサー値の取得
                // TODO: vector なので複数バッファを利用したい。
                ::nnd::tmp451::Temperature readData;
                ::nnd::tmp451::ReadTemperature(&readData, ::nnd::tmp451::Location::Local);
                dataBuffer[0].local = readData;
                ::nnd::tmp451::ReadTemperature(&readData, ::nnd::tmp451::Location::Remote);
                dataBuffer[0].remote = readData;

                // 時間取得
                // 協定世界時 (UTC) の 西暦1970年1月1日午前0時0分0秒 からの経過秒数で現在時刻を取得します。
                nn::time::PosixTime posixTime;
                NN_ABORT_UNLESS_RESULT_SUCCESS( nn::time::StandardUserSystemClock::GetCurrentTime(&posixTime) );
                dataBuffer[0].dateStr = GetCurrentDateStr(posixTime);
                dataBuffer[0].timeStr = GetCurrentTimeStr(posixTime);

                // ログファイルへの書き出し
                ::std::string OutStrData;
                OutStrData = dataBuffer[0].dateStr.c_str();
                OutStrData += "\t";
                OutStrData += dataBuffer[0].timeStr.c_str();
                OutStrData += "\t";
                OutStrData += ::std::to_string(dataBuffer[0].local.integer) + "." + ::std::to_string(dataBuffer[0].local.decimal);
                OutStrData += "\t";
                OutStrData += ::std::to_string(dataBuffer[0].remote.integer) + "." + ::std::to_string(dataBuffer[0].remote.decimal);
                OutStrData += "\n";
                result = ::nn::fs::WriteFile(g_FileHandle, g_WriteOffset, OutStrData.c_str(),
                        OutStrData.size(), ::nn::fs::WriteOption::MakeValue(0));
                g_WriteOffset += OutStrData.size();
                NN_ASSERT(result.IsSuccess(), "Failed to write log");

                // コンソールにも出力
                NN_LOG("%s", OutStrData.c_str());

                // ファイルへの書き出し
                result = ::nn::fs::FlushFile(g_FileHandle);
                NN_ASSERT(result.IsSuccess(), "Failed to flush log");
            }
        }
        else
        {
            // 計測も停止しないが、ログファイルへの出力は行わない。LCDの更新も止める。
        }

        DisplayTextOnLcd(pScreen, dataBuffer);

    }

    // 終了処理 -----------------------------------------

    ::nn::time::Finalize();

    // ファイルシステムのマウントを解除
    ::nn::fs::Unmount(SdcardMountName.c_str());

    // FIXME: lbl プロセスで照度値が取得できなくなる
    ::nnd::tmp451::Finalize();

    pScreen->Finalize();
    delete pScreen;

    NN_LOG("/// END ///\n");

#ifndef CI_EXECUTION
    // CI の実行テストで無視したい処理はで通らない
    NN_LOG("waiting for something...\n");
#endif

#ifdef CI_EXECUTION
    // CI の実行テストで通る
    NN_LOG("CI Pass\n");
#endif
} // NOLINT(impl/function_size)

