﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Log.h>
#include <nn/nn_TimeSpan.h>

#include <nn/os.h>
#include <nn/ts.h>

#include <nnt/nntest.h>

#include "../../Samples/Common/Ts_SampleCommon.h"

namespace {

using Temperature  = int;
struct LocationInfo
{
    nn::ts::Location m_Location;
    char             m_Name[32];
    Temperature  m_TempLow;
    Temperature  m_TempHigh;
};

// 正常動作する温度の上界、下界(ts_TargetSpecTmp451-hardware.nx.h を参照)
LocationInfo g_LocationTable[] = {
    {nn::ts::Location_ThermalSensorInternal, "Location_ThermalSensorInternal", 25, 85},
    {nn::ts::Location_ThermalSensorExternal, "Location_ThermalSensorExternal", 25, 96},
};

}

class TemperatureTest : public ::testing::Test
{
protected:
    virtual void SetUp() NN_OVERRIDE
    {
        NN_LOG("Initialize()\n");
        ::nn::ts::Initialize();
    }
    virtual void TearDown() NN_OVERRIDE
    {
        NN_LOG("Finalize()\n");
        ::nn::ts::Finalize();
    }
};

::testing::AssertionResult IsTemperatureInRange(int n, Temperature low, Temperature high)
{
    bool isTemperatureInRange = low <= n && n <= high;
    return isTemperatureInRange
        ? ::testing::AssertionSuccess()
        : ::testing::AssertionFailure();
}
TEST_F(TemperatureTest, GetTemperatureRange)
{
    for( const auto &locationInfo : g_LocationTable)
    {
        ::nn::ts::LimitTemperature limit = ::nn::ts::GetTemperatureRange(locationInfo.m_Location);
        EXPECT_EQ(limit.low,  0  ) << "Lower bound of temperature range is unexpected value";
        EXPECT_EQ(limit.high, 127) << "Upper bound of temperature range is unexpected value";
    }
}
TEST_F(TemperatureTest, CanGetTemperature)
{
    for(const auto &locationInfo : g_LocationTable)
    {
        NN_LOG("%s: ", locationInfo.m_Name);
        for(int count = 0; count < MeasurementLoopCount; count++)
        {
            int temperature;
            NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::ts::GetTemperature(&temperature, locationInfo.m_Location));

            EXPECT_TRUE( IsTemperatureInRange(temperature, locationInfo.m_TempLow, locationInfo.m_TempHigh))
                << temperature             << " is not in expected range"
                << locationInfo.m_TempLow  << "C ~ "
                << locationInfo.m_TempHigh << "C";

            NN_LOG("%3d,", temperature);
            ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(MeasurementInterbalMs));
        }
        NN_LOG("\n");
    }
}
TEST_F(TemperatureTest, CanGetTemperatureMilliC)
{

    for(const auto &locationInfo : g_LocationTable)
    {
        NN_LOG("%s: ", locationInfo.m_Name);
        for(int i = 0; i < MeasurementLoopCount; i++)
        {
            int temperatureMilliC;
            NN_ABORT_UNLESS_RESULT_SUCCESS(::nn::ts::GetTemperatureMilliC(&temperatureMilliC, locationInfo.m_Location));
            const auto tempLow  = locationInfo.m_TempLow  * 1000;
            const auto tempHigh = locationInfo.m_TempHigh * 1000;
            EXPECT_TRUE(IsTemperatureInRange( temperatureMilliC,
                                              tempLow,
                                              tempHigh))
                << temperatureMilliC << " is not in expected range"
                << tempLow           << "Milli C ~ "
                << tempHigh          << "Milli C";

            NN_LOG("%6d,", temperatureMilliC);
            ::nn::os::SleepThread(::nn::TimeSpan::FromMilliSeconds(MeasurementInterbalMs));
        }
        NN_LOG("\n");
    }
}
