﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunnerTest.TestStatementInfoTest
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TestRunner;

    /// <summary>
    /// [TestRunner.TestStatementInfo]::IsWrongYaml用のテスト関数を定義します。
    /// </summary>
    [TestClass]
    public class IsWrongYamlTest
    {
        /// <summary>
        /// 境界値分析: ステートメントの初期化に失敗
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest001()
        {
            var msg = string.Empty;

            Assert.IsTrue(new TestStatementInfo(null).IsWrongYaml(out msg));
            Assert.AreEqual(msg, "Mapping Node expected but not found");

            Assert.IsTrue(new TestStatementInfo(new object()).IsWrongYaml(out msg));
            Assert.AreEqual(msg, "Mapping Node expected but not found");

            Assert.IsFalse(new TestStatementInfo(new Dictionary<string, object>()).IsWrongYaml(out msg));
            Assert.AreEqual(msg, string.Empty);
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Constants 要素の型が有効なマッピングノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest002()
        {
            HasExpectedMappingNodeOrNotTest(new Dictionary<string, object>(), "Constants", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Platforms 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest003()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "Platforms", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Modules 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest004()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "Modules", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Categories 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest005()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "Categories", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Parallelizable 要素がブール値でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest006()
        {
            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), "Parallelizable", true);
        }

        /// <summary>
        /// 境界値分析: ステートメント中の BreakOn 要素の型が規定のブレークレベルを表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest007()
        {
            var key = "BreakOn";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "None");

            HasExpectedValueOrNotTest(new Dictionary<string, object>(),
                                      key,
                                      new[] { "Failure", "Timeout", "Error", "None" },
                                      new[] { "Foo" });
        }

        /// <summary>
        /// 境界値分析: ステートメント中の GlobalRepeat 要素が正の整数を表す文字列でない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest008()
        {
            var key = "GlobalRepeat";

            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), key, "1");

            HasExpectedValueOrNotTest(new Dictionary<string, object>(), key, new[] { "2" }, new[] { "0", "-1", "value" });
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Tests 要素の型がシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest009()
        {
            HasExpectedTypeOrNotTest(new Dictionary<string, object>(), "Tests", new List<object>());
        }

        /// <summary>
        /// 境界値分析: ステートメント中の BuildTypes 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest010()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "BuildTypes", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の FailurePatterns 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest011()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "FailurePatterns", "Bar");
        }

        /// <summary>
        /// 境界値分析: ステートメント中の Resources 要素の型が有効なシーケンスノードでない
        /// </summary>
        [TestMethod]
        public void IsWrongYamlBoundaryTest012()
        {
            HasExpectedSequenceNodeOrNotTest(new Dictionary<string, object>(), "Resources", "Bar");
        }

        private static void HasExpectedTypeOrNotTest<T>(Dictionary<string, object> mappingNode, string key, T value)
        {
            var msg = string.Empty;

            mappingNode[key] = new object();

            Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid node type for '{key}'", msg);

            mappingNode[key] = value;

            Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        private static void HasExpectedValueOrNotTest<T>(Dictionary<string, object> mappingNode,
                                                         string key,
                                                         IEnumerable<T> validValues,
                                                         IEnumerable<T> invalidValues)
        {
            var msg = string.Empty;

            foreach (var invalidValue in invalidValues)
            {
                mappingNode[key] = invalidValue;

                Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
                Assert.AreEqual($"Invalid value '{invalidValue}' for '{key}'", msg);
            }

            foreach (var validValue in validValues)
            {
                mappingNode[key] = validValue;

                Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
                Assert.AreEqual(string.Empty, msg);
            }
        }

        private static void HasExpectedMappingNodeOrNotTest<T>(Dictionary<string, object> mappingNode, string key, T value)
        {
            var msg = string.Empty;

            mappingNode[key] = new object();

            Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid node type for '{key}'", msg);

            var node = new Dictionary<string, object>();

            mappingNode[key] = node;

            Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node["Foo"] = new object();

            Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid node type for an entry of '{key}'", msg);

            node["Foo"] = value;

            Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }

        private static void HasExpectedSequenceNodeOrNotTest<T>(Dictionary<string, object> mappingNode, string key, T value)
        {
            var msg = string.Empty;

            mappingNode[key] = new object();

            Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid node type for '{key}'", msg);

            var node = new List<object>();

            mappingNode[key] = node;

            Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);

            node.Add(new object());

            Assert.IsTrue(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual($"Invalid node type for an entry of '{key}'", msg);

            node.Clear();
            node.Add(value);

            Assert.IsFalse(new TestStatementInfo(mappingNode).IsWrongYaml(out msg));
            Assert.AreEqual(string.Empty, msg);
        }
    }
}
