﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace TestRunnerTest.TestStatementInfoTest
{
    using System.Collections.Generic;
    using System.Linq;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using TestRunner;
    using TestRunner.Executer;

    /// <summary>
    /// [TestRunner.TestStatementInfo]::GetTestNodeInfos系用のテスト関数を定義します。
    /// </summary>
    [TestClass]
    public class GetTestNodeInfosTest
    {
        /// <summary>
        /// 不変条件: ノードの情報は生成元のステートメントの情報に基づく
        /// </summary>
        [TestMethod]
        public void GetTestNodeInfosInvariantTest001()
        {
            var nodes = new TestStatementInfo(new Dictionary<string, object>()
            {
                {
                    "Tests",
                    new List<object>()
                    {
                        new Dictionary<string, object>() { { "Path", "Foo" } }
                    }
                }
            }).GetTestNodeInfos();

            Assert.AreEqual(1, nodes.Count);

            Assert.IsTrue(nodes[0].Parallelizable);
            Assert.AreEqual(BreakLevel.None, nodes[0].BreakLevel);
            Assert.AreEqual(1u, nodes[0].TestId);
            Assert.AreEqual(1u, nodes[0].GlobalRepeatId);
            Assert.AreEqual(1u, nodes[0].LocalRepeatId);
            Assert.AreEqual(1, nodes[0].LocalRepeat);
            Assert.AreEqual(0, nodes[0].Resources.Count);
            Assert.AreEqual(930, nodes[0].Timeout);
            Assert.AreEqual(string.Empty, nodes[0].Platform);
            Assert.AreEqual(string.Empty, nodes[0].BuildType);
            Assert.AreEqual("FOO${BAR}BAZ", nodes[0].VariableManager.ExpandVariables("FOO${BAR}BAZ"));

            nodes = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "Constants", new Dictionary<string, object>() { { "BAR", "QUX" } } },
                { "Parallelizable", false },
                { "BreakOn", "Failure" },
                { "Resources", new List<object> { "Hoge.exe" } },
                { "FailurePatterns", new List<object> { "Failed" } },
                { "Timeout", "30" },
                { "Platform", "Win64" },
                { "BuildType", "Release" },
                { "TargetName", "MY-TARGET-NAME" },
                { "TargetInterface", "MY-TARGET-INTERFACE" },
                {
                    "Tests",
                    new List<object>()
                    {
                        new Dictionary<string, object>() { { "Path", "Foo" } },
                        new Dictionary<string, object>() { { "Path", "Bar" } }
                    }
                }
            }).UpdateResultDirectory("Baz", 1u, "Qux").GetTestNodeInfos();

            Assert.AreEqual(2, nodes.Count);

            for (var i = 0; i < nodes.Count; ++i)
            {
                Assert.IsFalse(nodes[i].Parallelizable);
                Assert.AreEqual(BreakLevel.Failure, nodes[i].BreakLevel);
                Assert.AreEqual((uint)(i + 1), nodes[i].TestId);
                Assert.AreEqual(1u, nodes[i].GlobalRepeatId);
                Assert.AreEqual(1u, nodes[i].LocalRepeatId);
                Assert.AreEqual(1, nodes[i].LocalRepeat);
                Assert.AreEqual("Hoge.exe", nodes[i].Resources[0]);
                Assert.AreEqual("Failed", nodes[i].FailurePatterns[0]);
                Assert.AreEqual(30, nodes[i].Timeout);
                Assert.AreEqual("Win64", nodes[i].Platform);
                Assert.AreEqual("Release", nodes[i].BuildType);
                Assert.AreEqual("MY-TARGET-NAME", nodes[i].TargetNamePattern);
                Assert.AreEqual("MY-TARGET-INTERFACE", nodes[i].TargetInterfacePattern);
                Assert.AreEqual("FOOQUXBAZ", nodes[i].VariableManager.ExpandVariables("FOO${BAR}BAZ"));
            }
        }

        /// <summary>
        /// 不変条件: ノードの繰り返し回数は生成元のステートメントの情報に基づく
        /// </summary>
        [TestMethod]
        public void GetTestNodeInfosInvariantTest002()
        {
            var nodes = new TestStatementInfo(new Dictionary<string, object>()
            {
                {
                    "Tests",
                    new List<object>()
                    {
                        new Dictionary<string, object>() { { "Path", "Foo" } },
                        new Dictionary<string, object>() { { "Path", "Bar" } }
                    }
                }
            }).GetExpandedTestNodeInfos();

            Assert.AreEqual(2, nodes.Count);

            for (var i = 0; i < nodes.Count; ++i)
            {
                Assert.AreEqual((uint)(i + 1), nodes[i].TestId);
                Assert.AreEqual(1u, nodes[i].GlobalRepeatId);
                Assert.AreEqual(1u, nodes[i].LocalRepeatId);
                Assert.AreEqual(1, nodes[i].LocalRepeat);
            }

            Assert.AreEqual("Foo", nodes[0].GetTestContexts().First().TargetPath);
            Assert.AreEqual("Bar", nodes[1].GetTestContexts().First().TargetPath);

            nodes = new TestStatementInfo(new Dictionary<string, object>()
            {
                { "GlobalRepeat", "2" },
                { "LocalRepeat", "3" },
                {
                    "Tests",
                    new List<object>()
                    {
                        new Dictionary<string, object>() { { "Path", "Foo" } },
                        new Dictionary<string, object>() { { "Path", "Bar" } }
                    }
                }
            }).GetExpandedTestNodeInfos();

            Assert.AreEqual(12, nodes.Count);

            var j = 0;

            for (var k = 0; k < 2; ++k)
            {
                foreach (var targetPath in new[] { "Foo", "Bar" })
                {
                    for (var l = 0; l < 3; ++l)
                    {
                        Assert.AreEqual((uint)(j + 1), nodes[j].TestId);
                        Assert.AreEqual((uint)(k + 1), nodes[j].GlobalRepeatId);
                        Assert.AreEqual((uint)(l + 1), nodes[j].LocalRepeatId);
                        Assert.AreEqual(3, nodes[j].LocalRepeat);
                        Assert.AreEqual(targetPath, nodes[j].GetTestContexts().First().TargetPath);
                        ++j;
                    }
                }
            }
        }
    }
}
