﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Globalization;
using System.Threading;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using TeamCityLogParser;

namespace TeamCityLogParserTest
{
    [TestClass]
    public class UnitTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            // ロケールを固定
            Thread.CurrentThread.CurrentCulture = new CultureInfo("ja-JP");
        }

        [TestMethod]
        public void TestParseDate()
        {
            var date = LogParser.ParseDate("Started 'Wed Apr 09 17:15:44 JST 2014' on 'DEVSEDCISL01' by 'Git'");
            var expected = DateTime.ParseExact(
                "Wed Apr 09 17:15:44 +09:00 2014",
                "ddd MMM dd HH:mm:ss zzz yyyy",
                System.Globalization.DateTimeFormatInfo.InvariantInfo,
                System.Globalization.DateTimeStyles.None);

            Assert.AreEqual(
                expected.Ticks,
                date.Ticks);
        }

        [TestMethod]
        public void TestParseRevision()
        {
            var revision = LogParser.ParseRevision("[17:15:43]:	 [Available server properties] system.build.vcs.number.1 = df32428f53aad9a17f3797a637da04b92147afcd");
            var expected = "df32428f53aad9a17f3797a637da04b92147afcd";

            Assert.AreEqual(
                expected,
                revision);
        }

        [TestMethod]
        public void TestParseBranch()
        {
            var branch = LogParser.ParseBranch("[09:16:01]:	 [Available server properties] teamcity.build.vcs.branch.SigloSdk_SigloSdk = refs/heads/feature/SIGLO-1100_modifying_nw4f_sln");
            var expected = "refs/heads/feature/SIGLO-1100_modifying_nw4f_sln";

            Assert.AreEqual(
                expected,
                branch);
        }

        [TestMethod]
        public void ParseToJsonTextFromText()
        {
            var data = @"
Build 'SIGLO-SDK :: build_packages_tests' #202
Started 'Wed Apr 09 17:15:44 JST 2014' on 'DEVSEDCISL01' by 'Git'
Finished 'Wed Apr 09 17:20:13 JST 2014' with status 'NORMAL Success'
TeamCity URL http://devsedci01.ncl.nintendo.co.jp/TeamCity/viewLog.html?buildId=8182&buildTypeId=SigloSdk_BuildPackagesTests
TeamCity server version is 8.0.6 (build 27767)

[17:15:41]: bt28 (4m:32s)
[17:15:41]: Checking for changes (running for 4m:32s)
[17:15:43]: Available server properties
[17:15:43]:	 [Available server properties] build.counter = 202
[17:15:43]:	 [Available server properties] vcsroot.submoduleCheckout = CHECKOUT
[17:15:43]:	 [Available server properties] build.number = 202
[17:15:43]:	 [Available server properties] system.build.vcs.number.1 = df32428f53aad9a17f3797a637da04b92147afcd
[17:15:43]:	 [Available server properties] build.vcs.number.SigloSdk_SigloSdk = df32428f53aad9a17f3797a637da04b92147afcd
[17:15:43]:	 [Available server properties] vcsroot.reportTagRevisions = true
[17:15:43]:	 [Available server properties] teamcity.build.vcs.branch.SigloSdk_SigloSdk = refs/heads/feature/SIGLO-1100_modifying_nw4f_sln";

            var json = LogParser.ParseToJsonTextFromText(data);

            var date = DateTime.ParseExact(
                "Wed Apr 09 17:15:44 +09:00 2014",
                "ddd MMM dd HH:mm:ss zzz yyyy",
                System.Globalization.DateTimeFormatInfo.InvariantInfo,
                System.Globalization.DateTimeStyles.None);

            var dateString =  date.ToString().Replace("/", "\\/");

            var expected = "{\"date\":\"" + dateString + "\",\"revision\":\"df32428f53aad9a17f3797a637da04b92147afcd\",\"branch\":\"refs\\/heads\\/feature\\/SIGLO-1100_modifying_nw4f_sln\"}";

            Assert.AreEqual(
                expected,
                json);
        }
    }
}
