﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo;
using Nintendo.ServiceFramework;

namespace Nintendo.ServiceFrameworkTest
{
    public static class AssertionUtility
    {
        /// <summary>
        /// 指定したアクションの実行中に、指定した型の例外が起き、その例外オブジェクトに対する assert を行う
        /// </summary>
        /// <typeparam name="ExpectedException">期待する例外型</typeparam>
        /// <param name="a">実行するアクション</param>
        /// <param name="assertion">発生した例外に対する assert アクション</param>
        public static void AssertException<ExpectedException>(Action a, Action<ExpectedException> assertion)
            where ExpectedException : Exception
        {
            try
            {
                a();
            }
            catch (ExpectedException e)
            {
                assertion(e);
                return;
            }
            catch (Exception e)
            {
                Assert.Fail(@"{0} 型の例外を期待しましたが、{1} 型の例外が発生しました", typeof(ExpectedException), e.GetType());
                throw;
            }
            Assert.Fail(@"{0} 型の例外を期待しましたが、例外が発生しませんでした", typeof(ExpectedException));
        }

        /// <summary>
        /// 指定したアクションの実行中に、例外が発生しないことを assert する
        /// </summary>
        /// <param name="a">実行するアクション</param>
        public static void AssertNoException(Action a)
        {
            try
            {
                a();
            }
            catch (Exception e)
            {
                Assert.Fail(@"例外が起きないことを期待しましたが、{0} 型の例外が発生しました", e.GetType());
                throw;
            }
        }

        /// <summary>
        /// 指定されたユーザエラー例外が、指定されたエラーコードを持ち、指定された文字列を含むかどうかを assert する
        /// </summary>
        /// <param name="e">ユーザエラー例外オブジェクト</param>
        /// <param name="errorCode">期待されるエラーコード</param>
        /// <param name="expectedStrings">期待される文字列(配列)</param>
        public static void AssertUserErrorException(UserErrorException e, string errorCode, params string[] expectedStrings)
        {
            Assert.AreEqual(errorCode, e.ErrorCode);
            foreach (var s in expectedStrings)
            {
                Assert.IsTrue(e.Message.Contains(s), string.Format(@"<{0}> は <{1}> を含むことを期待しましたが、含みませんでした。", e.Message, s));
            }
        }

        public static void AssertAreSetEqual<T>(IEnumerable<T> expected, IEnumerable<T> target)
        {
            Assert.IsTrue(new HashSet<T>(expected).SetEquals(target));
        }
    }
}
