﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework;

namespace Nintendo.ServiceFrameworkTest
{
    [TestClass]
    public class SfInterfaceTest
    {
        #region CanAccept/GetEntity

        private interface IGetEntity : nn.sf.IServiceObject
        {
        }

        private interface IGetEntity2 : nn.sf.IServiceObject
        {
        }
        [TestMethod]
        public void TestCanAccept()
        {
            Assert.IsTrue(SfInterface.CanAccept(typeof(IGetEntity)));
        }

        [TestMethod]
        public void TestGetEntity()
        {
            // 共通テスト
            SfEntityTestUtility.TestGetEntityCommon<IGetEntity, IGetEntity2>();
            // エンティティの型が SfInterface であることテスト
            var e = typeof(IGetEntity).GetSfEntity();
            Assert.AreSame(typeof(SfInterface), e.GetType());
        }

        #endregion

        #region TestTypes

        private class TypesAttributeBase : Attribute
        {
            public Type[] Types { get; private set; }

            protected TypesAttributeBase(Type[] types)
            {
                this.Types = types;
            }

            protected static Type[] GetTypesImpl<A>(Type type)
                where A : TypesAttributeBase
            {
                var a = type.GetCustomAttribute<A>();
                return a == null ? new Type[0] : a.Types;
            }
        }

        [AttributeUsage(AttributeTargets.Interface)]
        private class ExpectedReferredTypesAttribute : TypesAttributeBase
        {
            public ExpectedReferredTypesAttribute(params Type[] types)
                : base(types)
            {
            }

            public static Type[] GetTypes(Type type)
            {
                return GetTypesImpl<ExpectedReferredTypesAttribute>(type);
            }
        }

        [AttributeUsage(AttributeTargets.Interface)]
        private class ExpectedStronglyReferredTypesAttribute : TypesAttributeBase
        {
            public ExpectedStronglyReferredTypesAttribute(params Type[] types)
                : base(types)
            {
            }

            public static Type[] GetTypes(Type type)
            {
                return GetTypesImpl<ExpectedStronglyReferredTypesAttribute>(type);
            }
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject))]
        private interface IGetReferredTypes1 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F();
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int))]
        private interface IGetReferredTypes2 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            int F();
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int))]
        private interface IGetReferredTypes3 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(int x);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int))]
        private interface IGetReferredTypes4 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(int x);
            [MethodId(1)]
            void G(int x);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int), typeof(float))]
        private interface IGetReferredTypes5 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(int x, float y);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int), typeof(float), typeof(double))]
        private interface IGetReferredTypes6 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            double F(int x, float y);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int), typeof(float), typeof(double), typeof(bool))]
        private interface IGetReferredTypes7 : nn.sf.IServiceObject
        {
            [MethodId(0)]
            double F(int x, float y);
            [MethodId(1)]
            bool G(int x, float y);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject), typeof(int))]
        private interface IGetReferredTypesOut : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(nn.sf.Out<int> x);
        }

        private interface IChild : nn.sf.IServiceObject
        {
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject))]
        [ExpectedReferredTypesAttribute(typeof(nn.sf.IServiceObject), typeof(IChild))]
        private interface IGetReferredTypesChild_Return : nn.sf.IServiceObject
        {
            [MethodId(0)]
            IChild F();
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject))]
        [ExpectedReferredTypesAttribute(typeof(nn.sf.IServiceObject), typeof(IChild))]
        private interface IGetReferredTypesChild_In : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(IChild x);
        }

        [ExpectedStronglyReferredTypes(typeof(nn.sf.IServiceObject))]
        [ExpectedReferredTypesAttribute(typeof(nn.sf.IServiceObject), typeof(IChild))]
        private interface IGetReferredTypesChild_Out : nn.sf.IServiceObject
        {
            [MethodId(0)]
            void F(nn.sf.Out<IChild> pOut);
        }

        private static Type[] TestTypes = new[]
        {
            typeof(IGetReferredTypes1),
            typeof(IGetReferredTypes2),
            typeof(IGetReferredTypes3),
            typeof(IGetReferredTypes4),
            typeof(IGetReferredTypes5),
            typeof(IGetReferredTypes6),
            typeof(IGetReferredTypes7),
            typeof(IGetReferredTypesOut),
            typeof(IGetReferredTypesChild_Return),
            typeof(IGetReferredTypesChild_In),
            typeof(IGetReferredTypesChild_Out),
        };

        #endregion

        [TestMethod]
        public void TestGetReferredEntities()
        {
            foreach (var type in TestTypes)
            {
                var target = new SfInterface(type).GetReferredEntities().Select(x => (Type)x);
                var expected = Enumerable.Concat(
                    ExpectedReferredTypesAttribute.GetTypes(type),
                    ExpectedStronglyReferredTypesAttribute.GetTypes(type));
                AssertionUtility.AssertAreSetEqual(expected, target);
            }
        }

        [TestMethod]
        public void TestGetStronglyReferredEntities()
        {
            foreach (var type in TestTypes)
            {
                var target = new SfInterface(type).GetStronglyReferredEntities().Select(x => (Type)x);
                var expected = ExpectedStronglyReferredTypesAttribute.GetTypes(type);
                AssertionUtility.AssertAreSetEqual(expected, target);
            }
        }
    }
}
