﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.IO;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework.CppCode;

namespace Nintendo.ServiceFrameworkTest.CppCode
{
    [TestClass]
    public class RawCppCodeGeneratorTest
    {
        [TestMethod]
        public void TestWriteLine()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.WriteLine(@"#pragma once");
                    g.Finish();
                }
                Assert.AreEqual(@"
#pragma once
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.WriteLine(@"#pragma {0}", "once");
                    g.Finish();
                }
                Assert.AreEqual(@"
#pragma once
", b.ToString());
            }
        }

        [TestMethod]
        public void TestNewline()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline(0);
                    g.Finish();
                }
                Assert.AreEqual(string.Empty, b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.Finish();
                }
                Assert.AreEqual(@"
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline(1);
                    g.Finish();
                }
                Assert.AreEqual(@"
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline(2);
                    g.Finish();
                }
                Assert.AreEqual(@"

", b.ToString());
            }
        }

        [TestMethod]
        public void TestAutoNewline()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.AutoNewLine();
                    g.Finish();
                }
                Assert.AreEqual(string.Empty, b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.AutoNewLine();
                    g.AutoNewLine();
                    g.Finish();
                }
                Assert.AreEqual(string.Empty, b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.AutoNewLine();
                    g.WriteLine(@"A");
                    g.AutoNewLine();
                    g.WriteLine(@"B");
                    g.AutoNewLine();
                    g.Finish();
                }
                Assert.AreEqual(@"
A

B
", b.ToString());
            }
        }

        [TestMethod]
        public void TestNamespace()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

}}
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

}}
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.ChangeCurrentNamespace(CppName.Create(@"nn2::bar"));
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

}}

namespace nn2 { namespace bar {

}}
", b.ToString());
            }
        }

        [TestMethod]
        public void TestInclude()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.Include(@"HEADER");
                    g.Finish();
                }
                Assert.AreEqual(@"
#include HEADER
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.Include(@"""Sample.h""");
                    g.Finish();
                }
                Assert.AreEqual(@"
#include ""Sample.h""
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.Include(@"<cstdio>");
                    g.Finish();
                }
                Assert.AreEqual(@"
#include <cstdio>
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.Include(@"HEADER");
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

}}

#include HEADER
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.Include(@"HEADER", true);
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

}}

#include HEADER
", b.ToString());
            }
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.ForceInclude(@"HEADER");
                    g.ForceInclude(@"HEADER");
                    g.Include(@"HEADER", false);
                    g.Include(@"HEADER2", false);
                    g.Include(@"HEADER2", false);
                    g.Finish();
                }
                Assert.AreEqual(@"
namespace nn { namespace foo {

#include HEADER
#include HEADER
#include HEADER
#include HEADER2

}}
", b.ToString());
            }
        }

        [TestMethod]
        public void TestIndent()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.WriteLine(@"class C");
                    g.WriteLine(@"{");
                    using (g.Indent())
                    {
                        g.WriteLine(@"void F()");
                        g.WriteLine(@"{");
                        using (g.Indent(8))
                        {
                            g.WriteLine(@"return;");
                        }
                        g.WriteLine(@"}");
                    }
                    g.WriteLine(@"};");
                    g.Finish();
                }
                Assert.AreEqual(@"
class C
{
    void F()
    {
            return;
    }
};
", b.ToString());
            }
        }

        [TestMethod]
        public void TestIntegrated()
        {
            {
                var b = new StringBuilder();
                using (var w = new StringWriter(b))
                {
                    var g = new RawCppCodeGenerator(w);
                    g.Newline();
                    g.WriteLine(@"#pragma once");
                    g.Newline();
                    g.WriteLine(@"// このファイルは自動生成です。手動で修正しないでください。");
                    g.WriteLine(@"// This file is auto-generated. Do not modify manually.");
                    g.Newline();
                    g.Include(@"""Header.h""");
                    g.Newline();
                    g.Include(@"<nn/Common.h>");
                    g.Include(@"<nn/Result.h>");
                    g.Include(@"<nn/Result.h>");
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.WriteLine(@"class Sample");
                    g.WriteLine(@"{");
                    using (g.Indent())
                    {
                        g.WriteLine(@"void F();");
                    }
                    g.WriteLine(@"};");
                    g.AutoNewLine();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::foo"));
                    g.WriteLine(@"void Sample::F()");
                    g.WriteLine(@"{");
                    using (g.Indent())
                    {
                        g.WriteLine(@"return;");
                    }
                    g.WriteLine(@"}");
                    g.AutoNewLine();
                    g.ChangeCurrentNamespace(CppName.Create(@"nn::bar"));
                    g.WriteLine(@"class Sample2");
                    g.WriteLine(@"{");
                    using (g.Indent())
                    {
                        g.WriteLine(@"void F();");
                    }
                    g.WriteLine(@"};");
                    g.AutoNewLine();
                    g.CloseCurrentNamespace();
                    g.Finish();
                }
                Assert.AreEqual(@"
#pragma once

// このファイルは自動生成です。手動で修正しないでください。
// This file is auto-generated. Do not modify manually.

#include ""Header.h""

#include <nn/Common.h>
#include <nn/Result.h>

namespace nn { namespace foo {

class Sample
{
    void F();
};

void Sample::F()
{
    return;
}

}}

namespace nn { namespace bar {

class Sample2
{
    void F();
};

}}
", b.ToString());
            }
        }
    }
}
