﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using Nintendo.ServiceFramework.CppCode;

namespace Nintendo.ServiceFrameworkTest.CppCode
{
    [TestClass]
    public class CppNameTest
    {
        [TestMethod]
        public void TestCppName()
        {
            {
                var n = CppName.Create(@"nn");
                Assert.IsTrue(n.NameChain.SequenceEqual(new[] { "nn" }));
                Assert.AreEqual(@"nn", n.FullName);
            }
            {
                var n = CppName.Create(@"nn::foo::Sample");
                Assert.IsTrue(n.NameChain.SequenceEqual(new[] { "nn", "foo", "Sample" }));
                Assert.AreEqual(@"nn::foo::Sample", n.FullName);
            }
        }

        [TestMethod]
        public void TestOperatorPlus()
        {
            var n = CppName.Create(@"nn");
            Assert.AreEqual(@"nn", (n + string.Empty).FullName);
            Assert.AreEqual(@"nn", (n + CppName.Create(string.Empty)).FullName);
            Assert.AreEqual(@"nn::foo", (n + "foo").FullName);
            Assert.AreEqual(@"nn::foo::Sample", (n + "foo::Sample").FullName);
            Assert.AreEqual(@"nn::foo::Sample", (n + CppName.Create("foo::Sample")).FullName);
            Assert.AreEqual(@"nn::foo::Sample", (n + CppName.Create("foo") + CppName.Create("Sample")).FullName);
        }

        [TestMethod]
        public void TestRelativeName()
        {
            var n = CppName.Create(@"nn::foo");
            Assert.AreEqual(@"Sample", CppName.Create(@"nn::foo::Sample").GetRelativeName(n));
            Assert.AreEqual(@"bar::Sample", CppName.Create(@"nn::foo::bar::Sample").GetRelativeName(n));
            Assert.AreEqual(@"nnn::foo::Sample", CppName.Create(@"nnn::foo::Sample").GetRelativeName(n));
            Assert.AreEqual(@"Sample", CppName.Create(@"Sample").GetRelativeName(n));
            Assert.AreEqual(@"Result", CppName.Create(@"nn::Result").GetRelativeName(n));
        }

        [TestMethod]
        public void TestRank()
        {
            Assert.AreEqual(0, CppName.Create().Rank);
            Assert.AreEqual(1, CppName.Create("nn").Rank);
            Assert.AreEqual(2, CppName.Create("nn::foo").Rank);
            Assert.AreEqual(3, CppName.Create("nn::foo::Sample").Rank);
        }

        [TestMethod]
        public void TestGetParent()
        {
            {
                var n = CppName.Create(@"nn::foo::Sample");
                Assert.AreEqual("nn::foo", n.GetNameSpaceName().FullName);
                Assert.AreEqual("nn", n.GetNameSpaceName().GetNameSpaceName().FullName);
                Assert.AreEqual(string.Empty, n.GetNameSpaceName().GetNameSpaceName().GetNameSpaceName().FullName);
            }
            {
                var n = CppName.Create(@"Sample");
                Assert.AreEqual(string.Empty, n.GetNameSpaceName().FullName);
            }
        }

        [TestMethod]
        public void TestGetSingleName()
        {
            {
                var n = CppName.Create(@"nn::foo::Sample");
                Assert.AreEqual("Sample", n.GetSingleName());
            }
            {
                var n = CppName.Create(@"Sample");
                Assert.AreEqual("Sample", n.GetSingleName());
            }
        }

        [TestMethod]
        public void TestEquality()
        {
            Assert.AreEqual(CppName.Create(), CppName.Create());
            var n = CppName.Create(@"nn::foo::Sample");
            Assert.AreEqual(n, CppName.Create(@"nn::foo::Sample"));
            Assert.AreEqual(n, CppName.Create(@"nn::foo") + "Sample");
            Assert.AreEqual((CppName)null, (CppName)null);
            Assert.AreNotEqual(n, CppName.Create(@"nn::foo::Sampl"));
            Assert.AreNotEqual(n, CppName.Create(@"nn::foo") + "Sampl");
            Assert.AreNotEqual((CppName)null, n);
            Assert.AreNotEqual(n, (CppName)null);
            Assert.AreEqual(n.GetHashCode(), CppName.Create(@"nn::foo::Sample").GetHashCode());
            Assert.AreEqual(n.GetHashCode(), (CppName.Create(@"nn::foo") + "Sample").GetHashCode());
        }
    }
}
