﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace RetailInteractiveDisplayTest
{
    [TestClass]
    public class RetailInteractiveDisplayTest
    {
        public TestContext TestContext { get; set; }

        private void RebootDevice()
        {
            var command = new DevMenuCommandTest.DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[]
            {
                "power reboot",
            }));

            System.Threading.Thread.Sleep(15000);
        }

        private string GetContentListPath(string name)
        {
            var testPath = new TestUtility.TestPath(TestContext);
            return @"""" + testPath.GetSigloRoot() + @"\Tests\Tools\Sources\Tests\RetailInteractiveDisplayTest\ContentList\" + name + @"""";
        }

        [TestInitialize]
        public void TestInitialize()
        {
            var command = new DevMenuCommandTest.DevMenuCommandSystem(this.TestContext);
            var appMaker = new DevMenuCommandTest.MakeTestApplication(this.TestContext, "Application");

            // 全てのコンテンツをアンインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "application uninstall --all",
            }));

            // 全ての unknown なシステムプログラムをアンインストール
            Assert.IsTrue(command.Execute(new string[]
            {
                "systemprogram uninstall all-unknown",
            }));

            // ダミーの試遊代メニューをインストール
            appMaker.MakeApplication(0x0100069000078000);
            Assert.IsTrue(command.Execute(new string[]
            {
                "application install " + appMaker.GetOutputDirectory() + " --all",
            }));

            RebootDevice();
        }

        [TestMethod]
        public void SystemUpdateTest()
        {
            var command = new DevMenuCommandTest.DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "rid systemupdate"
            },
                @"Last state is \""Completed\""."
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "rid systemupdate"
            },
                @"Last state is \""NeedNoUpdate\""."
            ));
        }

        [TestMethod]
        public void MenuUpdateTest()
        {
            var command = new DevMenuCommandTest.DevMenuCommandSystem(this.TestContext);

            Assert.IsTrue(command.Execute(new string[]
            {
                "patch list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("0 patch(es) found."));

            Assert.IsTrue(command.ResetAndExecute(new string[]
            {
                "rid menuupdate"
            }));

            Assert.IsTrue(command.Execute(new string[]
            {
                "patch list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("1 patch(es) found."));

            Assert.IsTrue(command.Execute(new string[]
            {
                "rid menuupdate"
            },
                @"Last state is \""NeedNoUpdate\""."
            ));
        }

        [TestMethod]
        public void ApplicationUpdateTest()
        {
            var command = new DevMenuCommandTest.DevMenuCommandSystem(this.TestContext);

            // 3 つのコンテンツを指定して正しくインストールされるかの確認
            Assert.IsTrue(command.Execute(new string[]
            {
                "rid applicationupdate " + GetContentListPath("ContentList_3.json")
            },
                @"Last state is \""Completed\""."
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("4 application(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28000"));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28001"));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28002"));

            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("3 add-on-content(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("1"));
            Assert.IsTrue(command.LastOutput.Contains("4"));
            Assert.IsTrue(command.LastOutput.Contains("10"));

            // 1つのコンテンツを指定して指定したもの以外が消えるかの確認
            Assert.IsTrue(command.Execute(new string[]
            {
                "rid applicationupdate " + GetContentListPath("ContentList_1.json")
            },
                @"Last state is \""Completed\""."
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("2 application(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28000"));

            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("1 add-on-content(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("1"));

            // 3 つのコンテンツ(1つはmovie only)を指定して正しくインストールされるかの確認
            Assert.IsTrue(command.Execute(new string[]
            {
                "rid applicationupdate " + GetContentListPath("ContentList_3_include_movieonly.json")
            },
                @"Last state is \""Completed\""."
            ));

            Assert.IsTrue(command.Execute(new string[]
            {
                "application list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("3 application(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28000"));
            Assert.IsTrue(command.LastOutput.Contains("0x0100b15003a28001"));

            Assert.IsTrue(command.Execute(new string[]
            {
                "addoncontent list"
            }));

            Assert.IsTrue(command.LastOutput.Contains("3 add-on-content(s) found."));
            Assert.IsTrue(command.LastOutput.Contains("1"));
            Assert.IsTrue(command.LastOutput.Contains("4"));
            Assert.IsTrue(command.LastOutput.Contains("10"));
        }
    }
}
