﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

// args: html document を読む対象の ApplicationID

#include <cstdio>
#include <cstdlib>
#include <cstring>

#include <string>

#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/os.h>
#include <nn/oe.h>

#include <nn/ncm/ncm_Service.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_ContentMetaExtendedData.h>
#include <nn/ncm/ncm_ContentMetaUtil.h>
#include <nn/ncm/ncm_ApplyDeltaTask.h>

#include <nnt/nnt_Argument.h>
#include <nn/ns/ns_Result.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationRecordSystemApi.h>
#include <nn/ns/ns_ApplicationViewApi.h>
#include <nn/ns/ns_DocumentApi.h>
#include <nn/ns/ns_InitializationApi.h>
#include <nn/ns/ns_RequestServerStopper.h>
#include <nn/ns/ns_SdCardApi.h>

#include <nn/fs.h>
#include <nn/fssystem/fs_PartitionFileSystem.h>
#include <nn/fs/fs_FileStorage.h>
#include <nn/fs/fsa/fs_Registrar.h>
#include <nn/fs/fs_GameCard.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/nifm.h>
#include <nn/nifm/nifm_Request.h>
#include <nn/nifm/nifm_ApiRequest.h>
#include <nn/nim/nim_Result.h>
#include <nn/nim/nim_NetworkInstallManagerApi.h>

namespace
{
uint8_t g_HeapMemory[64 << 10];
nn::lmem::HeapHandle g_HeapHandle;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

nn::Result PushApplicationRecordBasedOnDatabase(nn::ncm::ApplicationId id)
{
    // 現在インストールしたものではなく、現状のコンテントメタデータベースから Record を再生成する
    const int maxKeys = 2048;
    std::unique_ptr<nn::ncm::ContentMetaKey[]> keys(new nn::ncm::ContentMetaKey[maxKeys]);
    std::unique_ptr<nn::ncm::StorageContentMetaKey[]> skeys(new nn::ncm::StorageContentMetaKey[maxKeys * 2]);
    int countSkeys = 0;

    // Application / Patch の探索 (なくてもカードのために登録する)
    auto f = [&] (nn::ncm::StorageId storage) -> nn::Result
        {
            nn::ncm::ContentMetaDatabase db;
            NN_RESULT_TRY(OpenContentMetaDatabase(&db, storage))
            NN_RESULT_CATCH(nn::ncm::ResultContentMetaDatabaseNotActive)
            {
                // Inactive な場合は、何もせずに成功扱い
                NN_RESULT_SUCCESS;
            }
            NN_RESULT_END_TRY
            auto listcount = db.ListContentMeta(keys.get(), maxKeys, nn::ncm::ContentMetaType::Unknown, id);
            for (int i = 0; i < listcount.listed; ++i)
            {
                if (keys[i].type == nn::ncm::ContentMetaType::Application
                    || keys[i].type == nn::ncm::ContentMetaType::Patch
                    || keys[i].type == nn::ncm::ContentMetaType::AddOnContent)
                {
                    skeys[countSkeys] = {keys[i], storage};
                    countSkeys++;
                }
            }
            NN_RESULT_SUCCESS;
        };
    // builtin
    {
        NN_RESULT_DO(f(nn::ncm::StorageId::BuildInUser));
    }
    // card
    if(nn::fs::IsGameCardInserted())
    {
        NN_RESULT_DO(f(nn::ncm::StorageId::Card));
    }
    // sd
    if(nn::ns::CheckSdCardMountStatus().IsSuccess())
    {
        NN_RESULT_DO(f(nn::ncm::StorageId::SdCard));
    }
    NN_RESULT_TRY(nn::ns::DeleteApplicationRecord(id))
        NN_RESULT_CATCH(nn::ns::ResultApplicationRecordNotFound) {} // 見つからないは無視する
    NN_RESULT_END_TRY
    if (countSkeys > 0)
    {
        NN_RESULT_DO(nn::ns::PushApplicationRecord(id, nn::ns::ApplicationEvent::LocalInstalled, skeys.get(), countSkeys));
    }
    NN_RESULT_SUCCESS;
}

// copied from ns_ApplicationViewMananger.cpp
int FilterFragmentOnlyContentMetaKey(nn::ncm::StorageContentMetaKey* list, int count) NN_NOEXCEPT
{
    int currentIndex = 0;
    for (int i = 0; i < count; ++i)
    {
        if(list[i].key.type == nn::ncm::ContentMetaType::Patch && list[i].key.installType == nn::ncm::ContentInstallType::FragmentOnly)
        {
            list[currentIndex] = list[i];
            currentIndex++;
        }
    }
    return currentIndex;
}
// copied from ns_ApplicationViewMananger.cpp
nn::Result SortAndConvertToContentMetaKey(nn::ncm::ContentMetaKey* list, nn::ncm::StorageContentMetaKey* list2, int count) NN_NOEXCEPT
{
    // 全ストレージが一致していることのチェック
    auto storageId = list2[0].storageId;
    for (int i = 0; i < count; ++i)
    {
        NN_RESULT_THROW_UNLESS(storageId == list2[i].storageId, nn::ns::ResultUnmatchStorageIds());
    }
    // ContentMetaKey に潰す
    for (int i = 0; i < count; ++i)
    {
        std::memmove(&(list[i]), &(list2[i].key), sizeof(nn::ncm::ContentMetaKey));
        // list[i] = list2[i].key;
    }
    std::sort(list, list + count);
    NN_RESULT_SUCCESS;
}

// copied from ns_ApplicationViewMananger.cpp
nn::Result RegisterApplyDeltaTask(nn::ncm::StorageContentMetaKey* list, int count, nn::ncm::ApplicationId id) NN_NOEXCEPT
{
    count = FilterFragmentOnlyContentMetaKey(list, count);
    if (count > 0)
    {
        // 同じ領域を使ってソートして、登録する
        nn::ncm::ContentMetaKey* keyList = reinterpret_cast<nn::ncm::ContentMetaKey*>(list);
        nn::ncm::StorageId storageId = list[0].storageId;
        NN_RESULT_DO(SortAndConvertToContentMetaKey(keyList, list, count));

        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));

        nn::ncm::ContentMetaKey patch;
        auto listcount = db.ListContentMeta(&patch, 1, nn::ncm::ContentMetaType::Patch, id);
        NN_RESULT_THROW_UNLESS(listcount.listed == 1, nn::ns::ResultSourcePatchNotFound());

        nn::nim::ApplyDeltaTaskId deltaTaskId;
        NN_RESULT_DO(nn::nim::CreateApplyDeltaTask(&deltaTaskId, id, patch, keyList, count, storageId));
        NN_LOG("[RegisterApplyDeltaTask] task created\n");
    }
    NN_RESULT_SUCCESS;
}

// GetApplicationView では、RuntimeHelper が動いていると Commit してくれない
// そのため、こちらでコミットをする必要がある
nn::Result CommitDownloadTask(nn::ncm::ApplicationId id)
{
    nn::nim::NetworkInstallTaskId taskId;
    int count = nn::nim::ListApplicationNetworkInstallTask(&taskId, 1, id);
    if (count > 0)
    {
        nn::nim::NetworkInstallTaskInfo info;
        NN_RESULT_DO(nn::nim::GetNetworkInstallTaskInfo(&info, taskId));
        if (info.progress.state == nn::ncm::Downloaded)
        {
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);

            NN_RESULT_DO(nn::nim::CommitNetworkInstallTask(taskId));
            NN_LOG("[download] Commit\n");
            const int CountKeys = 16;
            nn::ncm::StorageContentMetaKey keys[CountKeys];
            int count;
            NN_RESULT_DO(nn::nim::ListNetworkInstallTaskContentMeta(&count, keys, CountKeys, 0, taskId));
            NN_RESULT_DO(RegisterApplyDeltaTask(keys, count, id));
            NN_RESULT_DO(nn::nim::DestroyNetworkInstallTask(taskId));
        }
    }
    NN_RESULT_SUCCESS;
}


nn::Result MountApplicationHtmlDocument()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);

    nn::ncm::ApplicationId id;
    auto buffer = nnt::GetHostArgv()[2];
    id.value = std::strtoull(buffer, nullptr, 16);

    NN_RESULT_DO(nn::ns::MountApplicationHtmlDocument("Manual", id));

    NN_RESULT_SUCCESS;
}

const char* HexToString(const nn::Bit8* data, size_t size)
{
    // snprintf 用バッファ
    static char str[128];
    for(int i = 0; i < size && i < (sizeof(str) / 2); ++i)
    {
        nn::util::SNPrintf(&(str[i * 2]), sizeof(str) - (i * 2), "%02x", data[i]);
    }
    return str;
}
void HexStringToBits(nn::Bit8* data, size_t size, const char* string)
{
    char buffer[3] = {};
    for(int i = 0; i < size; ++i)
    {
        std::memcpy(buffer, &(string[i * 2]), 2);
        data[i] = static_cast<nn::Bit8>(std::strtoul(buffer, nullptr, 16));
    }
}

void DumpPatchMeta(const nn::ncm::PatchMetaExtendedDataReader& reader)
{
    auto header = reader.GetHeader();

    for(int i = 0; i < header->historyCount; ++i)
    {
        auto history = reader.GetPatchHistoryHeader(i);
        NN_LOG("History %d\n", i);
        NN_LOG(" - Key: %016llx, %u, %d\n", history->key.id, history->key.version, history->key.type);
        NN_LOG(" - Digest: %s\n", HexToString(history->digest.data, sizeof(nn::ncm::Hash)));
        for(int j = 0; j < history->contentCount; ++j)
        {
            auto content = reader.GetPatchHistoryContentInfo(i, j);
            NN_LOG(" - Contents %d\n", j);
            NN_LOG("   - ContentId: %s\n", HexToString(content->id.data, sizeof(nn::ncm::ContentId)));
            NN_LOG("   - Type: %d\n", content->type);
            NN_LOG("   - Size: %lld\n", content->GetSize());
        }
    }
    for(int i = 0; i < header->deltaHistoryCount; ++i)
    {
        auto history = reader.GetPatchDeltaHistory(i);
        NN_LOG("Delta History %d\n", i);
        NN_LOG(" - Source\n");
        NN_LOG("   - ID: %016llx\n", history->sourceId);
        NN_LOG("   - Version: %u\n", history->sourceVersion);

        NN_LOG(" - Destination\n");
        NN_LOG("   - ID: %016llx\n", history->destinationId);
        NN_LOG("   - Version: %u\n", history->destinationVersion);
        NN_LOG(" - Size: %lld\n", history->downloadSize);
    }
    for(int i = 0; i < header->deltaCount; ++i)
    {
        auto delta = reader.GetPatchDeltaHeader(i);
        NN_LOG("Delta %d\n", i);
        NN_LOG(" - Source\n");
        NN_LOG("   - ID: %016llx\n", delta->delta.sourceId);
        NN_LOG("   - Version: %u\n", delta->delta.sourceVersion);

        NN_LOG(" - Destination\n");
        NN_LOG("   - ID: %016llx\n", delta->delta.destinationId);
        NN_LOG("   - Version: %u\n", delta->delta.destinationVersion);

        for(int j = 0; j < delta->contentCount; ++j)
        {
            auto content = reader.GetPatchDeltaPackagedContentInfo(i, j);
            NN_LOG(" - Contents %d\n", j);
            NN_LOG("   - ContentId: %s\n", HexToString(content->info.id.data, sizeof(nn::ncm::ContentId)));
            NN_LOG("   - Type: %d\n", content->info.type);
            NN_LOG("   - Size: %lld\n", content->info.GetSize());
        }
        for(int j = 0; j < delta->delta.fragmentSetCount; ++j)
        {
            auto fragmentSet = reader.GetFragmentSet(i, j);
            NN_LOG(" - FragmentSet %d\n", j);
            NN_LOG("   - TargetContentType: %d\n", fragmentSet->targetContentType);
            NN_LOG("   - Source\n");
            NN_LOG("     - ContentId: %s\n", HexToString(fragmentSet->sourceContentId.data, sizeof(nn::ncm::ContentId)));
            NN_LOG("     - Size: %lld\n", fragmentSet->GetSourceSize());
            NN_LOG("   - Destination\n");
            NN_LOG("     - ContentId: %s\n", HexToString(fragmentSet->destinationContentId.data, sizeof(nn::ncm::ContentId)));
            NN_LOG("     - Size: %lld\n", fragmentSet->GetDestinationSize());
            NN_LOG("   - Fragments\n");
            for(int k = 0; k < fragmentSet->fragmentCount; ++k)
            {
                auto fragmentIndicator = reader.GetFragmentIndicator(i, j, k);
                NN_LOG("     - ContentInfoIndex: %d\n", fragmentIndicator->contentInfoIndex);
                NN_LOG("     - FragmentIndex: %d\n", fragmentIndicator->fragmentIndex);
            }
        }
    }
}
void DumpDeltaMeta(const nn::ncm::DeltaMetaExtendedDataReader& reader)
{
    auto header = reader.GetHeader();
    // source
    NN_LOG("Source\n");
    NN_LOG(" - ID: %016llx\n", header->sourceId);
    NN_LOG(" - Version: %u\n", header->sourceVersion);

    // destination
    NN_LOG("Destination\n");
    NN_LOG(" - ID: %016llx\n", header->destinationId);
    NN_LOG(" - Version: %u\n", header->destinationVersion);

    // fragment set
    for(int i = 0; i < header->fragmentSetCount; ++i)
    {
        auto fragmentSet = reader.GetFragmentSet(i);
        NN_LOG("FragmentSet %d\n", i);
        NN_LOG(" - Source\n");
        NN_LOG("   - ContentId: %s\n", HexToString(fragmentSet->sourceContentId.data, sizeof(nn::ncm::ContentId)));
        NN_LOG("   - Size: %lld\n", fragmentSet->GetSourceSize());
        NN_LOG(" - Destination\n");
        NN_LOG("   - ContentId: %s\n", HexToString(fragmentSet->destinationContentId.data, sizeof(nn::ncm::ContentId)));
        NN_LOG("   - Size: %lld\n", fragmentSet->GetDestinationSize());
        NN_LOG(" - Fragments\n");
        for(int j = 0; j < fragmentSet->fragmentCount; ++j)
        {
            auto fragmentIndicator = reader.GetFragmentIndicator(i, j);
            NN_LOG("   - ContentInfoIndex: %d\n", fragmentIndicator->contentInfoIndex);
            NN_LOG("   - FragmentIndex: %d\n", fragmentIndicator->fragmentIndex);
        }
    }
}

nn::Result ReadContentMeta(nn::ncm::AutoBuffer* meta, nn::ncm::ContentMetaType metaType)
{
    NN_ASSERT(nnt::GetHostArgc() == 3);

    auto buffer = nnt::GetHostArgv()[2];
    auto id = std::strtoull(buffer, nullptr, 16);

    nn::ncm::ContentMetaDatabase db;
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));

    nn::ncm::ContentMetaKey key;
    nn::ncm::ListCount count;
    count = db.ListContentMeta(&key, 1, metaType, nn::ncm::ContentMetaDatabase::AnyApplicationId, id, id);

    if (count.listed != 1)
    {
        NN_LOG("[Failed] id not found\n");

        NN_RESULT_THROW(nn::ncm::ResultContentNotFound());
    }

    nn::ncm::Path path;
    nn::ncm::StorageContentMetaKey storageKey = {key, storageId};
    NN_RESULT_DO(nn::ncm::GetContentMetaPath(&path, storageKey));
    NN_RESULT_DO(nn::ncm::ReadContentMetaPath(meta, path.string));

    NN_RESULT_SUCCESS;
}

nn::Result ReadDeltaMetaExtendedData()
{
    nn::ncm::AutoBuffer meta;
    NN_RESULT_DO(ReadContentMeta(&meta, nn::ncm::ContentMetaType::Delta));

    nn::ncm::PackagedContentMetaReader reader(meta.Get(), meta.GetSize());

    size_t size = reader.GetExtendedDataSize();
    if(size == 0)
    {
        NN_LOG("[Success] ExtData size 0\n");
        NN_RESULT_SUCCESS;
    }
    auto ptr = reader.GetExtendedData();

    nn::ncm::DeltaMetaExtendedDataReader metaReader(ptr, size);
    DumpDeltaMeta(metaReader);

    NN_RESULT_SUCCESS;
}
nn::Result ReadPatchMetaExtendedData()
{
    nn::ncm::AutoBuffer meta;
    NN_RESULT_DO(ReadContentMeta(&meta, nn::ncm::ContentMetaType::Patch));

    nn::ncm::PackagedContentMetaReader reader(meta.Get(), meta.GetSize());

    size_t size = reader.GetExtendedDataSize();
    if(size == 0)
    {
        NN_LOG("[Success] ExtData size 0\n");
        NN_RESULT_SUCCESS;
    }
    auto ptr = reader.GetExtendedData();

    nn::ncm::PatchMetaExtendedDataReader metaReader(ptr, size);
    DumpPatchMeta(metaReader);

    NN_RESULT_SUCCESS;
}


static nn::os::ThreadType thread;
nn::Result ExecuteInThread(nn::ncm::ApplyDeltaTaskBase* task)
{
    const int StackSize = 8192;
    NN_OS_ALIGNAS_THREAD_STACK static nn::Bit8 s_ThreadStack[StackSize];
    nn::os::ThreadFunction f = [](void* task) -> void
        {
            auto result = reinterpret_cast<nn::ncm::ApplyDeltaTaskBase*>(task)->Execute();
            if (result.IsFailure() && !nn::ncm::ResultCancelled::Includes(result))
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(result);
            }
        };
    NN_RESULT_DO(nn::os::CreateThread(&thread, f, task, s_ThreadStack, StackSize, nn::os::DefaultThreadPriority));
    nn::os::StartThread(&thread);

    NN_RESULT_SUCCESS;
}
void JoinThread()
{
    nn::os::DestroyThread(&thread);
}

nn::Result WaitDownload(nn::ncm::ApplicationId id)
{
    for(;;)
    {
        // ダウンロード待ち
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (view.IsWaitingPatchInstall() || view.IsWaitingAocCommit())
        {
            NN_RESULT_DO(CommitDownloadTask(id));
            NN_RESULT_SUCCESS;
        }
        if (!view.IsDownloading())
        {
            break;
        }
        if (view.progress.state != nn::ns::ApplicationDownloadState::Runnable)
        {
            NN_LOG("Download suspended as 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
            NN_RESULT_THROW(view.progress.lastResult);
        }

        NN_LOG("[download] %12lld / %lld\n", view.progress.downloaded, view.progress.total);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }

    NN_RESULT_SUCCESS;
}

nn::Result Wait(nn::ncm::ApplicationId id)
{
    nn::ns::ApplicationView view;
    for(int i = 0; i < 300; ++i)
    {
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (view.IsDownloading())
        {
            NN_LOG("[download] %12lld / %lld %d\n", view.progress.downloaded, view.progress.total, static_cast<int>(view.progress.state));
        }
        else if (view.IsApplyingDelta())
        {
            NN_LOG("[apply] %12lld / %lld %d\n", view.applyProgress.applied, view.applyProgress.total, static_cast<int>(view.applyProgress.state));
        }
        else
        {
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    NN_RESULT_SUCCESS;
}

nn::Result ApplyDelta(bool withCancel = false)
{
    NN_ASSERT(nnt::GetHostArgc() == 4);

    nn::ncm::ApplicationId sourceId = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    nn::ncm::DeltaId deltaId = {std::strtoull(nnt::GetHostArgv()[3], nullptr, 16)};

    nn::ncm::ApplyDeltaTask task;
    const int ApplyBufferSize = 64 << 10; // 64KB
    std::unique_ptr<nn::Bit8[]> buffer(new nn::Bit8[ApplyBufferSize]);
    NN_RESULT_DO(task.SetBuffer(buffer.get(), ApplyBufferSize));

    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;

    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));

    nn::ncm::ContentMetaKey key;
    nn::ncm::ListCount count;
    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Patch, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] single patch not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }
    nn::ncm::StorageContentMetaKey source = {key, storageId};

    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Delta, nn::ncm::ContentMetaDatabase::AnyApplicationId , deltaId.value, deltaId.value);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] single delta not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }
    nn::ncm::StorageContentMetaKey delta = {key, storageId};
    nn::ncm::ApplyDeltaTaskBase::TaskState taskState;

    NN_RESULT_DO(task.Initialize(source, delta, &taskState));
    NN_RESULT_DO(task.Prepare());
    NN_RESULT_DO(ExecuteInThread(&task));
    for(;;)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        if (withCancel)
        {
            NN_RESULT_DO(task.Cancel());
            JoinThread();
        }
        auto progress = task.GetProgress();
        NN_LOG("[Progress] %d %lld / %lld\n", progress.state, progress.appliedDeltaSize, progress.totalDeltaSize);
        if (progress.state == nn::ncm::ApplyDeltaProgressState_DeltaApplied)
        {
            break;
        }
        if (withCancel)
        {
            NN_RESULT_DO(ExecuteInThread(&task));
        }
    }
    NN_RESULT_DO(task.Commit());

    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Patch, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] single patch not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }

    nn::ncm::StorageContentMetaKey keyList = {key, storageId};
    NN_RESULT_DO(nn::ns::PushApplicationRecord(sourceId, nn::ns::ApplicationEvent::LocalInstalled, &keyList, 1));

    NN_RESULT_SUCCESS;
}

nn::Result InstallContentMetaAsPlaceHolder(nn::ncm::PlaceHolderId* outPlaceHolderId, nn::ncm::ContentId* outContentId, size_t* outSize, nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser);

nn::Result InstallContentMetaAsPlaceHolder(nn::ncm::PlaceHolderId* outPlaceHolderId, nn::ncm::ContentId* outContentId, size_t* outSize, nn::ncm::StorageId storageId)
{
    NN_ASSERT(nnt::GetHostArgc() >= 3);
    auto nspPath = nnt::GetHostArgv()[2];

    // nsp を解析して、ContentMeta だけ取得して、Placeholder として書き込んでおく。
    // Placeholder の ID を表示する

    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, nspPath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

    nn::fs::FileHandleStorage fsStorage(file);

    std::unique_ptr<nn::fssystem::PartitionFileSystem> fileSystem(new nn::fssystem::PartitionFileSystem());
    fileSystem->Initialize(&fsStorage);

    nn::ncm::ContentStorage contentStorage;
    NN_RESULT_DO(nn::ncm::OpenContentStorage(&contentStorage, storageId));
    *outPlaceHolderId = contentStorage.GeneratePlaceHolderId();

    auto mountNamePlain = "nspMount";
    auto mountName = "nspMount:/";
    NN_RESULT_DO(nn::fs::fsa::Register(mountNamePlain, std::move(fileSystem)));
    NN_UTIL_SCOPE_EXIT {nn::fs::fsa::Unregister(mountNamePlain); };

    nn::fs::DirectoryHandle directory;
    NN_RESULT_DO(nn::fs::OpenDirectory(&directory, mountName, nn::fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseDirectory(directory); };

    for(;;)
    {
        nn::fs::DirectoryEntry entry;
        int64_t outRead;
        NN_RESULT_DO(nn::fs::ReadDirectory(&outRead, &entry, directory, 1));
        if (outRead == 0)
        {
            break;
        }
        // outContentId の長さが 32byte ということを利用してさぼっている
        if (std::strncmp(&(entry.name[32]), ".cnmt.nca", std::strlen(".cnmt.nca")) == 0)
        {
            // GetContentId
            HexStringToBits(&(outContentId->data[0]), sizeof(outContentId->data), entry.name);
            NN_RESULT_DO(contentStorage.CreatePlaceHolder(*outPlaceHolderId, *outContentId, entry.fileSize));

            char path[256];
            nn::util::SNPrintf(path, sizeof(path), "%s%s", mountName, entry.name);
            nn::fs::FileHandle ncaFile;
            NN_RESULT_DO(nn::fs::OpenFile(&ncaFile, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT {nn::fs::CloseFile(ncaFile); };

            char buffer[4096];
            int64_t totalWrite = 0;
            while (totalWrite < entry.fileSize)
            {
                size_t readSize = std::min(sizeof(buffer), static_cast<size_t>(entry.fileSize - totalWrite));
                NN_RESULT_DO(nn::fs::ReadFile(ncaFile, totalWrite, buffer, readSize));
                NN_RESULT_DO(contentStorage.WritePlaceHolder(*outPlaceHolderId, totalWrite, buffer, readSize));

                totalWrite += readSize;
            }
            NN_LOG("Make placeholder success\n");
            NN_LOG("PlaceHolderId: %s\n", HexToString(&(outPlaceHolderId->uuid.data[0]), sizeof(*outPlaceHolderId)));
            NN_LOG("ContentId: %s\n", HexToString(&(outContentId->data[0]), sizeof(*outContentId)));
            NN_LOG("Size: %lld\n", entry.fileSize);
            *outSize = entry.fileSize;
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(nn::ncm::ResultContentNotFound());
}

nn::Result InstallContentAsPlaceHolder(nn::ncm::PlaceHolderId* outPlaceHolderId, const nn::ncm::ContentId& contentId, nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser);

nn::Result InstallContentAsPlaceHolder(nn::ncm::PlaceHolderId* outPlaceHolderId, const nn::ncm::ContentId& contentId, nn::ncm::StorageId storageId)
{
    NN_ASSERT(nnt::GetHostArgc() >= 3);
    auto nspPath = nnt::GetHostArgv()[2];

    nn::fs::FileHandle file;
    NN_RESULT_DO(nn::fs::OpenFile(&file, nspPath, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT{ nn::fs::CloseFile(file); };

    nn::fs::FileHandleStorage fsStorage(file);

    std::unique_ptr<nn::fssystem::PartitionFileSystem> fileSystem(new nn::fssystem::PartitionFileSystem());
    fileSystem->Initialize(&fsStorage);

    nn::ncm::ContentStorage contentStorage;
    NN_RESULT_DO(nn::ncm::OpenContentStorage(&contentStorage, storageId));
    *outPlaceHolderId = contentStorage.GeneratePlaceHolderId();

    auto mountNamePlain = "nspMount";
    auto mountName = "nspMount:/";
    NN_RESULT_DO(nn::fs::fsa::Register(mountNamePlain, std::move(fileSystem)));
    NN_UTIL_SCOPE_EXIT {nn::fs::fsa::Unregister(mountNamePlain); };

    nn::fs::DirectoryHandle directory;
    NN_RESULT_DO(nn::fs::OpenDirectory(&directory, mountName, nn::fs::OpenDirectoryMode_File));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseDirectory(directory); };

    for(;;)
    {
        nn::fs::DirectoryEntry entry;
        int64_t outRead;
        NN_RESULT_DO(nn::fs::ReadDirectory(&outRead, &entry, directory, 1));
        if (outRead == 0)
        {
            break;
        }
        auto contentIdString = HexToString(contentId.data, sizeof(contentId));
        if (std::strncmp(entry.name, contentIdString, std::strlen(contentIdString)) == 0)
        {
            NN_RESULT_DO(contentStorage.CreatePlaceHolder(*outPlaceHolderId, contentId, entry.fileSize));

            char path[256];
            nn::util::SNPrintf(path, sizeof(path), "%s%s", mountName, entry.name);
            nn::fs::FileHandle ncaFile;
            NN_RESULT_DO(nn::fs::OpenFile(&ncaFile, path, nn::fs::OpenMode_Read));
            NN_UTIL_SCOPE_EXIT {nn::fs::CloseFile(ncaFile); };

            static char buffer[128 << 10];
            int64_t totalWrite = 0;
            while (totalWrite < entry.fileSize)
            {
                size_t readSize = std::min(sizeof(buffer), static_cast<size_t>(entry.fileSize - totalWrite));
                NN_RESULT_DO(nn::fs::ReadFile(ncaFile, totalWrite, buffer, readSize));
                NN_RESULT_DO(contentStorage.WritePlaceHolder(*outPlaceHolderId, totalWrite, buffer, readSize));

                totalWrite += readSize;
            }
            NN_RESULT_SUCCESS;
        }
    }

    NN_RESULT_THROW(nn::ncm::ResultContentNotFound());
}

nn::Result FindKey(nn::ncm::StorageContentMetaKey* outKey, nn::ncm::ApplicationId applicationId, nn::ncm::ContentMetaType type, nn::ncm::ContentInstallType installType)
{
    nn::ncm::StorageId storageIds[] = {nn::ncm::StorageId::BuiltInUser, nn::ncm::StorageId::SdCard};

    for (auto& storageId : storageIds)
    {
        nn::ncm::ContentMetaDatabase db;
        NN_RESULT_TRY(nn::ncm::OpenContentMetaDatabase(&db, storageId))
            NN_RESULT_CATCH(nn::ncm::ResultContentMetaDatabaseNotActive) { continue; }
        NN_RESULT_END_TRY

        nn::ncm::ContentMetaKey key;
        auto count = db.ListContentMeta(&key, 1, type, applicationId, 0, 0xffffffffffffffff, installType);
        if (count.listed == 1)
        {
            *outKey = {key, storageId};
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_THROW(nn::ncm::ResultContentMetaNotFound());
}

// パッチのメタを使って、パッチを更新する
// この時点では、あらかじめ delta をインストールしておく必要がある
#if 0
nn::Result ApplyLegacyPatchDelta()
{
    NN_ASSERT(nnt::GetHostArgc() == 7);

    nn::ncm::ApplicationId sourceId = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    nn::ncm::ApplicationId deltaId = {std::strtoull(nnt::GetHostArgv()[3], nullptr, 16)};
    nn::ncm::PlaceHolderId placeHolderId;
    nn::ncm::ContentId     contentId;
    HexStringToBits(&(placeHolderId.uuid.data[0]), sizeof(placeHolderId.uuid.data), nnt::GetHostArgv()[4]);
    HexStringToBits(&(contentId.data[0]), sizeof(contentId.data), nnt::GetHostArgv()[5]);

    int64_t size = std::strtoll(nnt::GetHostArgv()[6], nullptr, 10);

    nn::ncm::ContentInfo contentInfo = nn::ncm::ContentInfo::Make(contentId, size, nn::ncm::ContentType::Meta, 0);

    nn::ncm::ApplyLegacyPatchDeltaTask task;
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;

    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));

    nn::ncm::ContentMetaKey key;
    nn::ncm::ListCount count;
    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Patch, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] single patch not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }
    nn::ncm::StorageContentMetaKey source = {key, storageId};
    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Delta, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] single delta not found %016llx\n", deltaId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }
    NN_RESULT_DO(task.Initialize(source, placeHolderId, contentInfo));
    NN_RESULT_DO(task.Prepare());
    NN_RESULT_DO(ExecuteInThread(&task));
    auto progress = task.GetProgress();
    while(progress.state != nn::ncm::ApplyDeltaProgressState_DeltaApplied)
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        progress = task.GetProgress();
        NN_LOG("[Progress] %d %lld / %lld\n", progress.state, progress.appliedDeltaSize, progress.totalDeltaSize);
    }
    NN_RESULT_DO(task.Commit());

    count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Patch, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] id not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }

    nn::ncm::StorageContentMetaKey keyList = {key, storageId};
    NN_RESULT_DO(nn::ns::PushApplicationRecord(sourceId, nn::ns::ApplicationEvent::LocalInstalled, &keyList, 1));

    NN_RESULT_SUCCESS;
}
#endif
nn::Result ApplyPatchDelta(bool withCancel = false)
{
    NN_ASSERT(nnt::GetHostArgc() == 3);

    nn::ncm::ApplicationId sourceId = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::ncm::StorageContentMetaKey source;
    NN_RESULT_DO(FindKey(&source, sourceId, nn::ncm::ContentMetaType::Patch, nn::ncm::ContentInstallType::Full));
    nn::ncm::StorageContentMetaKey destination;
    NN_RESULT_DO(FindKey(&destination, sourceId, nn::ncm::ContentMetaType::Patch, nn::ncm::ContentInstallType::FragmentOnly));

    NN_LOG("Found,\n");
    NN_LOG("Source: %016llx %d\n", source.key.id, static_cast<int>(source.storageId));
    NN_LOG("Destination: %016llx %d\n", destination.key.id, static_cast<int>(destination.storageId));

    nn::ncm::ApplyDeltaTaskBase::TaskState taskState;
    bool isInitial = true;
    bool isComplete = false;

    int counter = 0;
    while(!isComplete)
    {
        counter++;
        nn::ncm::ApplyPatchDeltaTask task;
        const int ApplyBufferSize = 64 << 10; // 64KB
        std::unique_ptr<nn::Bit8[]> buffer(new nn::Bit8[ApplyBufferSize]);
        NN_RESULT_DO(task.SetBuffer(buffer.get(), ApplyBufferSize));

        if (isInitial)
        {
            NN_RESULT_DO(task.Initialize(source, destination, &taskState));
            NN_RESULT_DO(task.Prepare());
            isInitial = false;
        }
        else
        {
            NN_RESULT_DO(task.InitializeForResume(&taskState));
        }
        NN_RESULT_DO(ExecuteInThread(&task));
        for (;;)
        {
            // キャンセルをしすぎるとプログレスが進まないので、たまに緩める
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(counter % 10 == 9 ? 3000 : 100));
            if (withCancel)
            {
                NN_RESULT_DO(task.Cancel());
                JoinThread();
            }
            auto progress = task.GetProgress();
            NN_LOG("[Progress] %d %lld / %lld\n", progress.state, progress.appliedDeltaSize, progress.totalDeltaSize);
            if (progress.state == nn::ncm::ApplyDeltaProgressState_DeltaApplied)
            {
                NN_RESULT_DO(task.Commit());
                isComplete = true;
                break;
            }
            if (withCancel)
            {
                break;
            }
        }
    }
    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, source.storageId));

    nn::ncm::ContentMetaKey key;
    auto count = db.ListContentMeta(&key, 1, nn::ncm::ContentMetaType::Patch, sourceId);
    if (count.listed != 1)
    {
        NN_LOG("[Failed] id not found %016llx\n", sourceId.value);

        // Success じゃないのだけど・・
        NN_RESULT_SUCCESS;
    }

    nn::ncm::StorageContentMetaKey keyList = {key, source.storageId};
    NN_RESULT_DO(nn::ns::PushApplicationRecord(sourceId, nn::ns::ApplicationEvent::LocalInstalled, &keyList, 1));

    NN_RESULT_SUCCESS;
}

nn::Result InstallFragmentsOnly()
{
    nn::ncm::PlaceHolderId id;
    nn::ncm::ContentId     contentId;
    size_t size;
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuildInUser;
    if (nnt::GetHostArgc() > 3)
    {
        if (nnt::GetHostArgv()[3] == std::string("sdcard"))
        {
            storageId = nn::ncm::StorageId::SdCard;
        }
        else
        {
            NN_LOG("[Failed] Unknown storage: %s\n", nnt::GetHostArgv()[3]);
            NN_RESULT_SUCCESS;
        }
    }
    NN_RESULT_DO(InstallContentMetaAsPlaceHolder(&id, &contentId, &size, storageId));

    // placeholder を読み込み、InstallContentMeta を生成し、DB に push して commit する
    nn::ncm::Path path;
    nn::ncm::AutoBuffer meta;
    nn::ncm::ContentMetaDatabase db;
    nn::ncm::ContentStorage storage;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));

    storage.GetPlaceHolderPath(&path, id);
    NN_RESULT_DO(nn::ncm::ReadContentMetaPath(&meta, path.string));

    // 実質利用されるのは ContentInfo のみのはずなので、そこだけ設定
    nn::ncm::InstallContentInfo info = {};
    {
        info.info = nn::ncm::ContentInfo::Make(contentId, size, nn::ncm::ContentType::Meta, 0);
    }

    // packageReader -> installContentMeta
    nn::ncm::PackagedContentMetaReader packagedReader(meta.Get(), meta.GetSize());
    size_t installContentMetaSize;
    NN_RESULT_DO(packagedReader.CalculateConvertFragmentOnlyInstallContentMetaSize(&installContentMetaSize, 0));
    std::unique_ptr<nn::Bit8[]> installContentMeta(new nn::Bit8[installContentMetaSize]);
    packagedReader.ConvertToFragmentOnlyInstallContentMeta(installContentMeta.get(), installContentMetaSize, info, 0);

    // installContentMeta -> contentMeta
    nn::ncm::InstallContentMetaReader installReader(installContentMeta.get(), installContentMetaSize);
    auto contentMetaSize = installReader.CalculateConvertSize();
    std::unique_ptr<nn::Bit8[]> contentMeta(new nn::Bit8[contentMetaSize]);
    installReader.ConvertToContentMeta(contentMeta.get(), contentMetaSize);

    for (int i = 0; i < installReader.CountContent(); ++i)
    {
        auto installInfo = installReader.GetContentInfo(i);
        if (installInfo->info.type != nn::ncm::ContentType::Meta)
        {
            NN_LOG("Install content: %d, %s\n", static_cast<int>(installInfo->info.type), HexToString(installInfo->info.id.data, sizeof(installInfo->info.id)));
            nn::ncm::PlaceHolderId placeHolderId;
            NN_RESULT_DO(InstallContentAsPlaceHolder(&placeHolderId, installInfo->info.id, storageId));
            NN_RESULT_TRY(storage.Register(placeHolderId, installInfo->info.id))
                NN_RESULT_CATCH(nn::ncm::ResultContentAlreadyExists) {}
            NN_RESULT_END_TRY
        }
    }

    // register
    NN_RESULT_DO(storage.Register(id, contentId));
    NN_RESULT_DO(db.Set(installReader.GetKey(), contentMeta.get(), contentMetaSize));
    NN_RESULT_DO(db.Commit());

    NN_RESULT_SUCCESS;
}

nn::Result RegisterApplyPatchTaskImpl(nn::nim::ApplyDeltaTaskId* taskId, nn::ncm::ApplicationId id, nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser);

nn::Result RegisterApplyPatchTaskImpl(nn::nim::ApplyDeltaTaskId* taskId, nn::ncm::ApplicationId id, nn::ncm::StorageId storageId)
{
    nn::ncm::ContentMetaDatabase db;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    const int maxPatches = 16;
    nn::ncm::ContentMetaKey keys[maxPatches];
    auto count = db.ListContentMeta(keys, maxPatches, nn::ncm::ContentMetaType::Patch, id, 0, 0xffffffffffffffff, nn::ncm::ContentInstallType::FragmentOnly);

    for (int i = 0; i < count.listed; ++i)
    {
        NN_LOG("Found Key: %016llx, %u\n", keys[i].id, keys[i].version);
    }

    nn::ncm::ContentMetaKey sourcePatch;
    auto count2 = db.ListContentMeta(&sourcePatch, 1, nn::ncm::ContentMetaType::Patch, id);
    NN_UNUSED(count2);

    NN_RESULT_DO(nn::nim::CreateApplyDeltaTask(taskId, id, sourcePatch, keys, count.listed, storageId));

    NN_RESULT_SUCCESS;
}


nn::Result RegisterApplyPatchTaskImpl(nn::ncm::ApplicationId* id, nn::nim::ApplyDeltaTaskId* taskId)
{
    NN_ASSERT(nnt::GetHostArgc() >= 3);
    *id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser;
    if (nnt::GetHostArgc() > 3 && nnt::GetHostArgv()[3] == std::string("sdcard"))
    {
        storageId = nn::ncm::StorageId::SdCard;
    }

    NN_RESULT_DO(RegisterApplyPatchTaskImpl(taskId, *id, storageId));

    NN_RESULT_SUCCESS;
}

nn::Result RegisterApplyPatchTask()
{
    nn::ncm::ApplicationId id;
    nn::nim::ApplyDeltaTaskId taskId;
    NN_RESULT_DO(RegisterApplyPatchTaskImpl(&id, &taskId));

    // Application ID を指定すると、インストール済みの Meta-only パッチを探して nim に登録する
    nn::nim::AsyncResult async;
    NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, taskId));
    while(!async.TryWait())
    {
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
        nn::nim::ApplyDeltaTaskInfo info;
        NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, taskId));
        NN_LOG("[Progress] %lld / %lld\n", info.progress.applied, info.progress.total);
    }
    NN_RESULT_DO(async.Get());
    async.Finalize();

    NN_RESULT_DO(nn::nim::CommitApplyDeltaTask(taskId));
    NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(taskId));

    NN_RESULT_DO(PushApplicationRecordBasedOnDatabase(id));

    NN_RESULT_SUCCESS;
}
nn::Result ExpectNotEnoughSpaceImpl(nn::ncm::ApplicationId id)
{
    nn::ns::ApplicationView view;
    for(int i = 0; i < 300; ++i)
    {
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (view.progress.state == nn::ns::ApplicationDownloadState::NotEnoughSpace)
        {
            NN_LOG("Detect not enough space\n");

            int64_t requiredSize;
            nn::ncm::StorageId storageId;
            NN_RESULT_DO(nn::ns::CalculateApplicationDownloadRequiredSize(&storageId, &requiredSize, id));
            NN_LOG("Required size: %lld, Storage: %d\n", requiredSize, static_cast<int>(storageId));

            NN_RESULT_SUCCESS;
        }
        else if(view.applyProgress.state == nn::ns::ApplicationApplyDeltaState::NotEnoughSpace)
        {
            NN_LOG("Detect not enough space\n");

            int64_t requiredSize;
            nn::ncm::StorageId storageId;
            NN_RESULT_DO(nn::ns::CalculateApplicationApplyDeltaRequiredSize(&storageId, &requiredSize, id));
            NN_LOG("Required size: %lld, Storage: %d\n", requiredSize, static_cast<int>(storageId));

            NN_RESULT_SUCCESS;
        }

        if (view.IsDownloading())
        {
            NN_LOG("[download] %12lld / %lld %d\n", view.progress.downloaded, view.progress.total, static_cast<int>(view.progress.state));
        }
        if (view.IsApplyingDelta())
        {
            NN_LOG("[apply] %12lld / %lld %d\n", view.applyProgress.applied, view.applyProgress.total, static_cast<int>(view.applyProgress.state));
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
    }
    // not detect なのでエラーのほうがベター
    NN_RESULT_SUCCESS;
}

nn::Result RegisterAndExpectNotEnoughSpace()
{
    nn::ncm::ApplicationId id;

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        nn::nim::ApplyDeltaTaskId taskId;
        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&id, &taskId));
    }

    NN_RESULT_DO(ExpectNotEnoughSpaceImpl(id));

    NN_RESULT_SUCCESS;
}

nn::Result ExpectNotEnoughSpace()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(ExpectNotEnoughSpaceImpl(id));

    NN_RESULT_SUCCESS;
}

nn::Result ResumeApplyDelta()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(nn::ns::ResumeApplicationApplyDelta(id));

    NN_RESULT_SUCCESS;
}

nn::Result WaitAndCancelApplyDelta()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(WaitDownload(id));

    for(;;)
    {
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsApplyingDelta())
        {
            break;
        }
        NN_LOG("[apply] %12lld / %lld\n", view.applyProgress.applied, view.applyProgress.total);

        if (view.applyProgress.applied > 0)
        {
            NN_LOG("[apply] call cancel\n");
            NN_RESULT_DO(nn::ns::CancelApplicationApplyDelta(id));
            break;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));
    }
    NN_RESULT_SUCCESS;
}

nn::Result WaitAndSuspendApplyDelta()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(WaitDownload(id));

    int64_t currentProgress = 0;
    for(;;)
    {
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsApplyingDelta())
        {
            break;
        }
        NN_LOG("[apply] %12lld / %lld\n", view.applyProgress.applied, view.applyProgress.total);

        if (view.applyProgress.applied > currentProgress)
        {
            NN_LOG("[apply] stop request server\n");
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));
            currentProgress = view.applyProgress.applied;
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));
    }
    NN_RESULT_SUCCESS;
}
// ApplyDelta を途中で止める。上位層で電源断されることを前提としたもの
nn::Result WaitAndSuspendApplyDeltaPermanently()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(WaitDownload(id));
    for(;;)
    {
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsApplyingDelta())
        {
            break;
        }
        NN_LOG("[apply] %12lld / %lld\n", view.applyProgress.applied, view.applyProgress.total);

        if (view.applyProgress.applied > 0)
        {
            NN_LOG("[apply] stop request server\n");
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(300));
        }
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(300));
    }
    NN_RESULT_SUCCESS;
}

nn::Result WaitAndSuspendDownloadPermanently()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    for(;;)
    {
        // ダウンロード待ち
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsDownloading())
        {
            break;
        }
        if (view.progress.downloaded > 0 && view.progress.total > 0)
        {
            NN_LOG("[apply] stop request server\n");
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);
            nn::os::SleepThread(nn::TimeSpan::FromSeconds(300));
        }
        if (view.progress.state != nn::ns::ApplicationDownloadState::Runnable)
        {
            NN_LOG("Download suspended as 0x%08x\n", view.progress.lastResult.GetInnerValueForDebug());
            NN_RESULT_SUCCESS;
        }

        NN_LOG("[download] %12lld / %lld\n", view.progress.downloaded, view.progress.total);

        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_RESULT_SUCCESS;
}

nn::Result OrderChecker()
{
    NN_ASSERT(nnt::GetHostArgc() == 5);
    nn::ncm::ApplicationId id1 = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    nn::ncm::ApplicationId id2 = {std::strtoull(nnt::GetHostArgv()[3], nullptr, 16)};
    nn::ncm::ApplicationId id3 = {std::strtoull(nnt::GetHostArgv()[4], nullptr, 16)};

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        nn::nim::ApplyDeltaTaskId task1, task2, task3;
        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&task1, id1));
        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&task2, id2));

        NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(task1));

        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&task3, id3));
    }
    NN_RESULT_DO(Wait(id2));
    NN_RESULT_DO(Wait(id3));

    NN_RESULT_SUCCESS;
}

// 正しく破棄しないタスクが多いので、ちゃんと破棄する
nn::Result CleanupTask()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::nim::ApplyDeltaTaskId list[16];
    int count = nn::nim::ListApplicationApplyDeltaTask(list, 16, id);
    for (int i = 0; i < count; ++i)
    {
        NN_LOG("[Cleanup] %s\n", HexToString(&(list[i].uuid.data[0]), sizeof(nn::nim::ApplyDeltaTaskId)));
        NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(list[i]));
    }

    NN_RESULT_SUCCESS;
}

nn::nifm::Request* RequestNifm()
{
    nn::nifm::RequestParameters requestParameters = { nn::nifm::RequirementPreset_InternetGeneric };
    auto request = new nn::nifm::Request(requestParameters);
    request->SubmitAndWait();
    if( request->GetRequestState() == nn::nifm::RequestState_Accepted )
    {
        return request;
    }
    else
    {
        NN_LOG("[Request failed] %d\n", static_cast<int>(request->GetRequestState()));
        delete request;
        return nullptr;
    }
}

nn::Result SuspendDownloadAndRequestUpdateApplication()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    for(;;)
    {
        // ダウンロード待ち
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsDownloading())
        {
            break;
        }
        if (view.progress.downloaded > 0 && view.progress.total > 0)
        {
            NN_LOG("[apply] stop request server\n");
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);

            auto ptr = RequestNifm();
            if (ptr)
            {
                std::unique_ptr<nn::nifm::Request> request(ptr);
                nn::ns::AsyncResult result;
                NN_RESULT_DO(nn::ns::RequestUpdateApplication(&result, id));
                result.Wait();
                NN_RESULT_DO(result.Get());
            }
            else
            {
                NN_RESULT_THROW(nn::ns::ResultInternetRequestNotAccepted());
            }
        }
        NN_LOG("[download] %12lld / %lld\n", view.progress.downloaded, view.progress.total);
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_RESULT_SUCCESS;
}

nn::Result SuspendApplyDeltaAndRequestUpdateApplication()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    NN_RESULT_DO(WaitDownload(id));

    for(;;)
    {
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
        if (!view.IsApplyingDelta())
        {
            break;
        }
        NN_LOG("[apply] %12lld / %lld\n", view.applyProgress.applied, view.applyProgress.total);

        if (view.applyProgress.applied > 0)
        {
            NN_LOG("[apply] stop request server\n");
            nn::ns::RequestServerStopper stopper;
            GetRequestServerStopper(&stopper);

            auto ptr = RequestNifm();
            if (ptr)
            {
                std::unique_ptr<nn::nifm::Request> request(ptr);
                nn::ns::AsyncResult result;
                NN_RESULT_DO(nn::ns::RequestUpdateApplication(&result, id));
                result.Wait();
                NN_RESULT_DO(result.Get());
            }
            else
            {
                NN_RESULT_THROW(nn::ns::ResultInternetRequestNotAccepted());
            }
        }
        nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
    }
    NN_RESULT_SUCCESS;
}

nn::Result FormatSdTest()
{
    // fragment のインストール。2 が NAND 側の ID、3 が SD 側の ID とする
    NN_ASSERT(nnt::GetHostArgc() == 4);
    nn::ncm::ApplicationId nand = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    nn::ncm::ApplicationId sd = {std::strtoull(nnt::GetHostArgv()[3], nullptr, 16)};

    nn::nim::ApplyDeltaTaskId nandTask, sdTask;

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&nandTask, nand));
        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&sdTask, sd, nn::ncm::StorageId::SdCard));

        NN_RESULT_DO(nn::ns::FormatSdCard());

        // NAND は取得できる
        nn::nim::ApplyDeltaTaskInfo info;
        NN_RESULT_TRY(nn::nim::GetApplyDeltaTaskInfo(&info, nandTask))
            NN_RESULT_CATCH_ALL
            {
                NN_LOG("[Failure] NAND task killed unexpectedly\n");
            }
        NN_RESULT_END_TRY

        // SD は取得できない
        auto result = nn::nim::GetApplyDeltaTaskInfo(&info, sdTask);
        if (result.IsSuccess())
        {
            NN_LOG("[Failure] SD task exists unexpectedly\n");
        }
    }

    NN_RESULT_SUCCESS;
}
nn::Result RegisterApplyDeltaTaskAndDestryWithoutCommit()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::nim::ApplyDeltaTaskId task;

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&task, id));

        nn::nim::AsyncResult async;
        NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, task));
        while(!async.TryWait())
        {
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
            nn::nim::ApplyDeltaTaskInfo info;
            NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, task));
            NN_LOG("[Progress] %lld / %lld\n", info.progress.applied, info.progress.total);
        }
        NN_RESULT_DO(async.Get());
        async.Finalize();

        // commit せずに destroy する。タスクとしては更新後のパッチはできている状態
        NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(task));
        NN_LOG("Destroyed\n");
    }
    NN_RESULT_SUCCESS;
}

nn::Result RegisterOnly()
{
    nn::ncm::ApplicationId id;
    nn::nim::ApplyDeltaTaskId task;

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&id, &task));
    }
    NN_RESULT_SUCCESS;
}

nn::Result RegisterApplyDeltaTaskAndDestryIntermediately()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::nim::ApplyDeltaTaskId task;

    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        NN_RESULT_DO(RegisterApplyPatchTaskImpl(&task, id));

        {
            nn::nim::AsyncResult async;
            NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, task));
            bool willBreak = false;
            while(!willBreak)
            {
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(100));
                nn::nim::ApplyDeltaTaskInfo info;
                NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, task));
                NN_LOG("[Progress] %lld / %lld\n", info.progress.applied, info.progress.total);
                if (info.progress.applied > 0)
                {
                    willBreak = true;
                }
            }
            async.Finalize();
        }

        // commit せずに destroy する。タスクとしては更新後のパッチはできている状態
        NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(task));
        NN_LOG("Destroyed\n");
    }
    NN_RESULT_SUCCESS;
}

nn::Result SIGLO_48725()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser;
    nn::ncm::PatchId patchId = {id.value + 0x800};
    nn::ncm::ContentMetaKey sourceKey = nn::ncm::ContentMetaKey::Make(patchId, 1 * 65536);
    nn::ncm::ContentMetaKey inOrderKeys[] =
    {
        nn::ncm::ContentMetaKey::Make(patchId, 2 * 65536),
        nn::ncm::ContentMetaKey::Make(patchId, 3 * 65536),
    };
    nn::ncm::ContentMetaKey outOfOrderKeys[] =
    {
        nn::ncm::ContentMetaKey::Make(patchId, 5 * 65536),
        nn::ncm::ContentMetaKey::Make(patchId, 3 * 65536),
    };

    {
        // 正しい順番だと作れることの確認
        nn::nim::ApplyDeltaTaskId taskId;
        nn::nim::ApplyDeltaTaskId list[16];

        // create 後に fail したりするのだが、無視してその先がとれるかを見る
        nn::nim::CreateApplyDeltaTask(&taskId, id, sourceKey, inOrderKeys, 2, storageId);
        auto count = nn::nim::ListApplicationApplyDeltaTask(list, 16, id);
        if (count != 1)
        {
            NN_LOG("task is not created unexpectedly\n");
            NN_RESULT_SUCCESS;
        }
        NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(list[0]));
    }
    {
        // 誤った順番だとエラーになる
        nn::nim::ApplyDeltaTaskId taskId;
        nn::nim::ApplyDeltaTaskId list[16];

        auto result = nn::nim::CreateApplyDeltaTask(&taskId, id, sourceKey, outOfOrderKeys, 2, storageId);
        if (result.IsSuccess())
        {
            NN_LOG("Success unexpectedly\n");
            NN_RESULT_SUCCESS;
        }

        // ここで fatal しないことを確認
        nn::nim::ListApplicationApplyDeltaTask(list, 16, id);
    }
    NN_RESULT_SUCCESS;
}
nn::Result SIGLO_47637()
{
    // view をひたすらとりつづける
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    for(;;)
    {
        nn::ns::ApplicationView view;
        NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
    }
    NN_RESULT_SUCCESS;
}

// 特定のタイミングでダウンロードタスクがキャンセルされたときに問題が起こるのでは？
// という調査用に書いたコードを一応コミットしておくだけで、実際のテストには利用しない
nn::Result NXBTS_11123()
{
    NN_ASSERT(nnt::GetHostArgc() == 4);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};
    int interval = static_cast<int>(std::strtoll(nnt::GetHostArgv()[3], nullptr, 10));
    NN_LOG("interval: %d\n", interval);
    {
        nn::ns::RequestServerStopper stopper;
        GetRequestServerStopper(&stopper);

        nn::nim::NetworkInstallTaskId task;
        auto count = nn::nim::ListApplicationNetworkInstallTask(&task, 1, id);

        while (count > 0)
        {
            nn::nim::AsyncResult async;
            NN_RESULT_DO(nn::nim::RequestNetworkInstallTaskRun(&async, task));
            nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(interval));
            nn::nim::NetworkInstallTaskInfo info;
            NN_RESULT_DO(nn::nim::GetNetworkInstallTaskInfo(&info, task));
            NN_LOG("[Progress] %lld / %lld\n", info.progress.installedSize, info.progress.totalSize);
        }
    }

    NN_RESULT_SUCCESS;
}

// 前提条件
// v1 をフルインストール && v2 を FragmentOnly でインストール
nn::Result SIGLO_51171()
{
    nn::ns::RequestServerStopper stopper;
    GetRequestServerStopper(&stopper);

    nn::ncm::ApplicationId id;
    nn::nim::ApplyDeltaTaskId taskId;
    NN_RESULT_DO(RegisterApplyPatchTaskImpl(&id, &taskId));

    int counter = 0;
    for(;;)
    {
        nn::nim::AsyncResult async;
        NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, taskId));

        // 要ウェイト時間の調整
        nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds((counter % 10) == 0 ? 1000 : 300));
        {
            nn::nim::ApplyDeltaTaskInfo info;
            NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, taskId));

            NN_LOG("[Progress] %lld / %lld\n", info.progress.applied, info.progress.total);
        }

        async.Finalize();

        {
            nn::nim::ApplyDeltaTaskInfo info;
            NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, taskId));
            if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::AllApplied)
            {
                break;
            }
            if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::Fatal)
            {
                nn::Result result = nn::util::Get(info.progress.lastResult);
                NN_LOG("[Progress] Task failed: %08x\n", result.GetInnerValueForDebug());
                NN_RESULT_THROW(result);
            }
        }
        counter++;
    }
    NN_RESULT_DO(nn::nim::CommitApplyDeltaTask(taskId));
    NN_RESULT_DO(nn::nim::DestroyApplyDeltaTask(taskId));

    NN_RESULT_SUCCESS;
}

nn::Result CleanupPlaceholders()
{
    nn::ncm::StorageId storageIds[] = {nn::ncm::StorageId::BuiltInUser, nn::ncm::StorageId::SdCard};
    for (auto storageId : storageIds)
    {
        nn::ncm::ContentStorage storage;
        auto result = nn::ncm::OpenContentStorage(&storage, storageId);
        if (result.IsFailure())
        {
            continue;
        }
        NN_RESULT_DO(storage.CleanupAllPlaceHolder());
    }
    NN_RESULT_SUCCESS;
}

nn::Result WaitApplyDeltaWithoutCommit()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::nifm::SubmitNetworkRequestAndWait();
    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_RESULT_SUCCESS;
    }
    {
        // requrest server は止める
        nn::ns::RequestServerStopper stopper;
        nn::ns::GetRequestServerStopper(&stopper);

        {
            // nim から直接ダウンロードタスク情報を取得する
            nn::nim::NetworkInstallTaskId taskId;
            auto count = nn::nim::ListApplicationNetworkInstallTask(&taskId, 1, id);
            if (count != 1)
            {
                NN_LOG("Task not found\n");
                NN_RESULT_SUCCESS;
            }
            nn::nim::AsyncResult async;
            NN_RESULT_DO(nn::nim::RequestNetworkInstallTaskRun(&async, taskId));
            for(;;)
            {
                if (async.TryWait())
                {
                    auto result = async.Get();
                    // ref: ns_InstallUtil.cpp
                    if ((result.IsSuccess()
                         || result <= nn::nim::ResultHttpConnectionCanceled()
                         || result <= nn::nim::ResultHttpConnectionRetryableTimeout()
                         || result <= nn::ncm::ResultInstallTaskCancelled()))
                    {
                        async.Finalize();
                        NN_RESULT_DO(nn::nim::RequestNetworkInstallTaskRun(&async, taskId));
                    }
                    else
                    {
                        NN_RESULT_THROW(result);
                    }
                }
                nn::nim::NetworkInstallTaskInfo info;
                NN_RESULT_DO(nn::nim::GetNetworkInstallTaskInfo(&info, taskId));
                if (info.progress.state == nn::ncm::Downloaded)
                {
                    NN_RESULT_DO(async.Get());
                    async.Finalize();

                    NN_RESULT_DO(CommitDownloadTask(id));
                    break;
                }
                else if (info.progress.state == nn::ncm::Fatal)
                {
                    nn::Result result = nn::util::Get(info.progress.lastResult);
                    NN_LOG("Task failed %08x\n", result.GetInnerValueForDebug());
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    NN_LOG("%12lld / %lld\n", info.progress.installedSize, info.progress.totalSize);
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
        {
            nn::nim::ApplyDeltaTaskId taskId;
            auto count = nn::nim::ListApplicationApplyDeltaTask(&taskId, 1, id);
            if (count != 1)
            {
                NN_LOG("Task not found\n");
                NN_RESULT_SUCCESS;
            }
            nn::nim::AsyncResult async;
            NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, taskId));
            for(;;)
            {
                nn::nim::ApplyDeltaTaskInfo info;
                NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, taskId));
                if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::AllApplied)
                {
                    NN_RESULT_DO(async.Get());
                    async.Finalize();
                    NN_LOG("Waiting commit\n");
                    break;
                }
                else if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::Fatal)
                {
                    nn::Result result = nn::util::Get(info.progress.lastResult);
                    NN_LOG("Task failed %08x\n", result.GetInnerValueForDebug());
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    NN_LOG("%12lld / %lld\n", info.progress.applied, info.progress.total);
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
        NN_RESULT_SUCCESS;
    }
}

nn::Result WaitApplyDeltaAndReportOccupiedSize()
{
    NN_ASSERT(nnt::GetHostArgc() == 3);
    nn::ncm::ApplicationId id = {std::strtoull(nnt::GetHostArgv()[2], nullptr, 16)};

    nn::nifm::SubmitNetworkRequestAndWait();
    if (!nn::nifm::IsNetworkAvailable())
    {
        NN_RESULT_SUCCESS;
    }
    {
        // requrest server は止める
        nn::ns::RequestServerStopper stopper;
        nn::ns::GetRequestServerStopper(&stopper);

        {
            // nim から直接ダウンロードタスク情報を取得する
            nn::nim::NetworkInstallTaskId taskId;
            auto count = nn::nim::ListApplicationNetworkInstallTask(&taskId, 1, id);
            if (count != 1)
            {
                NN_LOG("Task not found\n");
                NN_RESULT_SUCCESS;
            }
            nn::nim::AsyncResult async;
            NN_RESULT_DO(nn::nim::RequestNetworkInstallTaskRun(&async, taskId));
            for(;;)
            {
                if (async.TryWait())
                {
                    auto result = async.Get();
                    // ref: ns_InstallUtil.cpp
                    if ((result.IsSuccess()
                         || result <= nn::nim::ResultHttpConnectionCanceled()
                         || result <= nn::nim::ResultHttpConnectionRetryableTimeout()
                         || result <= nn::ncm::ResultInstallTaskCancelled()))
                    {
                        async.Finalize();
                        NN_RESULT_DO(nn::nim::RequestNetworkInstallTaskRun(&async, taskId));
                    }
                    else
                    {
                        NN_RESULT_THROW(result);
                    }
                }
                nn::nim::NetworkInstallTaskInfo info;
                NN_RESULT_DO(nn::nim::GetNetworkInstallTaskInfo(&info, taskId));
                if (info.progress.state == nn::ncm::Downloaded)
                {
                    NN_RESULT_DO(async.Get());
                    async.Finalize();

                    // commit 相当
                    nn::ns::ApplicationView view;
                    NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
                    break;
                }
                else if (info.progress.state == nn::ncm::Fatal)
                {
                    nn::Result result = nn::util::Get(info.progress.lastResult);
                    NN_LOG("Task failed %08x\n", result.GetInnerValueForDebug());
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    NN_LOG("%12lld / %lld\n", info.progress.installedSize, info.progress.totalSize);
                }
                nn::os::SleepThread(nn::TimeSpan::FromSeconds(1));
            }
        }
        {
            nn::nim::ApplyDeltaTaskId taskId;
            auto count = nn::nim::ListApplicationApplyDeltaTask(&taskId, 1, id);
            if (count != 1)
            {
                NN_LOG("Task not found\n");
                NN_RESULT_SUCCESS;
            }
            int64_t currentApplied = -1;
            int currentWait = 1000;
            for (;;)
            {
                nn::ns::ApplicationOccupiedSize occupied;
                NN_RESULT_DO(nn::ns::CalculateApplicationOccupiedSize(&occupied, id));
                auto f = [&occupied](nn::ncm::StorageId storageId) -> const nn::ns::ApplicationOccupiedSizeEntity*
                {
                    for (const auto& s : occupied.storage)
                    {
                        if (s.storageId == storageId)
                        {
                            return &s;
                        }
                    }
                    return nullptr;
                };
                NN_LOG("[Occupied] NAND: %lld, SD: %lld\n", f(nn::ncm::StorageId::BuiltInUser)->patchSize, f(nn::ncm::StorageId::SdCard)->patchSize);

                nn::nim::ApplyDeltaTaskInfo info;
                NN_RESULT_DO(nn::nim::GetApplyDeltaTaskInfo(&info, taskId));
                if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::AllApplied)
                {
                    // commit 相当
                    nn::ns::ApplicationView view;
                    NN_RESULT_DO(nn::ns::GetApplicationView(&view, &id, 1));
                    break;
                }
                else if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::NotEnoughSpace)
                {
                    NN_LOG("Task state is NotEnoughSpace\n");
                    NN_RESULT_SUCCESS;
                }
                else if (info.progress.state == nn::nim::ApplyDownloadedDeltaProgressState::Fatal)
                {
                    nn::Result result = nn::util::Get(info.progress.lastResult);
                    NN_LOG("Task failed %08x\n", result.GetInnerValueForDebug());
                    NN_RESULT_SUCCESS;
                }
                else
                {
                    NN_LOG("[progress] %12lld / %lld\n", info.progress.applied, info.progress.total);
                }
                if (currentApplied == info.progress.applied)
                {
                    // 進捗がなかったら、待ち時間を延ばす
                    currentWait += 1000;
                }
                else
                {
                    currentWait = 1000;
                }
                currentApplied = info.progress.applied;

                nn::nim::AsyncResult async;
                NN_RESULT_DO(nn::nim::RequestApplyDeltaTaskRun(&async, taskId));
                nn::os::SleepThread(nn::TimeSpan::FromMilliSeconds(currentWait));
                async.Finalize();
            }
        }
    }
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

nn::Result NXBTS_12466()
{
    char buffer[256];
    for (int j = 0; j < 128; ++j)
    {
    nn::ncm::StorageContentMetaKey* keys = reinterpret_cast<nn::ncm::StorageContentMetaKey*>(&(buffer[j]));
    for (int i = 0; i < 4; ++i)
    {
        nn::ncm::ContentMetaKey key = {0x10000000ul + i, 100u + i, nn::ncm::ContentMetaType::AddOnContent};
        keys[i] = {key, nn::ncm::StorageId::BuiltInUser};
    }
    int index = 0;
    for (int i = 0; i < 4; ++i)
    {
        if (keys[i].key.version == 102)
        {
            continue;
        }
        //std::memcpy(&(keys[index]), &(keys[i]), sizeof(keys[0]));
        keys[index] = keys[i];
        index++;
    }
    for (int i = 0; i < index; ++i)
    {
        NN_LOG("ID: %016llx, ver.: %d\n", keys[i].key.id, keys[i].key.version);
    }
    }
    NN_RESULT_SUCCESS;
}

// ファイルを受け取り、patch のエントリを作成する
nn::Result CreateContentMetaDatabasePatchEntry()
{
    NN_ASSERT(nnt::GetHostArgc() >= 3);

    nn::ncm::PlaceHolderId id;
    nn::ncm::ContentId     contentId;
    size_t size;
    nn::ncm::StorageId storageId = nn::ncm::StorageId::BuiltInUser;
    if (nnt::GetHostArgc() > 3 && nnt::GetHostArgv()[3] == std::string("sdcard"))
    {
        storageId = nn::ncm::StorageId::SdCard;
    }

    NN_RESULT_DO(InstallContentMetaAsPlaceHolder(&id, &contentId, &size, storageId));

    // placeholder を読み込み、InstallContentMeta を生成し、DB に push して commit する
    nn::ncm::Path path;
    nn::ncm::AutoBuffer meta;
    nn::ncm::ContentMetaDatabase db;
    nn::ncm::ContentStorage storage;
    NN_RESULT_DO(nn::ncm::OpenContentMetaDatabase(&db, storageId));
    NN_RESULT_DO(nn::ncm::OpenContentStorage(&storage, storageId));

    storage.GetPlaceHolderPath(&path, id);
    NN_RESULT_DO(nn::ncm::ReadContentMetaPath(&meta, path.string));

    // 実質利用されるのは ContentInfo のみなので、そこだけ設定
    nn::ncm::InstallContentInfo info = {};
    {
        info.info = nn::ncm::ContentInfo::Make(contentId, size, nn::ncm::ContentType::Meta, 0);
    }

    // packageReader -> installContentMeta
    nn::ncm::PackagedContentMetaReader packagedReader(meta.Get(), meta.GetSize());
    size_t installContentMetaSize = packagedReader.CalculateConvertInstallContentMetaSize();
    std::unique_ptr<nn::Bit8[]> installContentMeta(new nn::Bit8[installContentMetaSize]);
    packagedReader.ConvertToInstallContentMeta(installContentMeta.get(), installContentMetaSize, info);

    // installContentMeta -> contentMeta
    nn::ncm::InstallContentMetaReader installReader(installContentMeta.get(), installContentMetaSize);
    auto contentMetaSize = installReader.CalculateConvertSize();
    std::unique_ptr<nn::Bit8[]> contentMeta(new nn::Bit8[contentMetaSize]);
    installReader.ConvertToContentMeta(contentMeta.get(), contentMetaSize);

    // register
    NN_RESULT_DO(db.Set(installReader.GetKey(), contentMeta.get(), contentMetaSize));
    NN_RESULT_DO(db.Commit());

    NN_RESULT_DO(storage.DeletePlaceHolder(id));

    NN_RESULT_SUCCESS;
}


}

extern "C" void nnMain()
{
    NN_ASSERT(nnt::GetHostArgc() >= 2);

    g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), nn::lmem::CreationOption_NoOption);
    nn::fs::SetAllocator(Allocate, Deallocate);

    nn::oe::Initialize();

    nn::ncm::Initialize();
    NN_UTIL_SCOPE_EXIT{nn::ncm::Finalize();};

    nn::ns::Initialize();
    NN_UTIL_SCOPE_EXIT{nn::ns::Finalize();};

    nn::nim::InitializeForNetworkInstallManager();
    NN_UTIL_SCOPE_EXIT {nn::nim::FinalizeForNetworkInstallManager(); };

    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::fs::MountHostRoot());
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::nifm::Initialize());

    std::string command = nnt::GetHostArgv()[1];
    nn::Result result;
    if(command == "MountApplicationHtmlDocument")
    {
        result = MountApplicationHtmlDocument();
    }
    else if (command == "ReadDeltaMetaExtendedData")
    {
        result = ReadDeltaMetaExtendedData();
    }
    else if (command == "ReadPatchMetaExtendedData")
    {
        result = ReadPatchMetaExtendedData();
    }
    else if (command == "InstallContentMetaAsPlaceHolder")
    {
        nn::ncm::PlaceHolderId id;
        nn::ncm::ContentId     contentId;
        size_t size;
        result = InstallContentMetaAsPlaceHolder(&id, &contentId, &size);
    }
    else if (command == "ApplyDelta")
    {
        result = ApplyDelta();
    }
    else if (command == "ApplyDeltaWithRepeatedCancel")
    {
        result = ApplyDelta(true);
    }
#if 0
    else if (command == "ApplyLegacyPatchDelta")
    {
        result = ApplyLegacyPatchDelta();
    }
#endif
    else if (command == "ApplyPatchDelta")
    {
        result = ApplyPatchDelta();
    }
    else if (command == "ApplyPatchDeltaWithRepeatedCancel")
    {
        result = ApplyPatchDelta(true);
    }
    else if (command == "InstallFragmentsOnly")
    {
        result = InstallFragmentsOnly();
    }
    else if (command == "RegisterApplyPatchTask")
    {
        result = RegisterApplyPatchTask();
    }
    else if (command == "RegisterAndExpectNotEnoughSpace")
    {
        result = RegisterAndExpectNotEnoughSpace();
    }
    else if (command == "ExpectNotEnoughSpace")
    {
        result = ExpectNotEnoughSpace();
    }
    else if (command == "ResumeApplyDelta")
    {
        result = ResumeApplyDelta();
    }
    else if (command == "WaitAndCancelApplyDelta")
    {
        result = WaitAndCancelApplyDelta();
    }
    else if (command == "WaitAndSuspendApplyDelta")
    {
        result = WaitAndSuspendApplyDelta();
    }
    else if (command == "WaitAndSuspendApplyDeltaPermanently")
    {
        result = WaitAndSuspendApplyDeltaPermanently();
    }
    else if (command == "WaitAndSuspendDownloadPermanently")
    {
        result = WaitAndSuspendDownloadPermanently();
    }
    else if (command == "OrderChecker")
    {
        result = OrderChecker();
    }
    else if (command == "CleanupTask")
    {
        result = CleanupTask();
    }
    else if (command == "SuspendDownloadAndRequestUpdateApplication")
    {
        result = SuspendDownloadAndRequestUpdateApplication();
    }
    else if (command == "SuspendApplyDeltaAndRequestUpdateApplication")
    {
        result = SuspendApplyDeltaAndRequestUpdateApplication();
    }
    else if (command == "FormatSdTest")
    {
        result = FormatSdTest();
    }
    else if (command == "RegisterApplyDeltaTaskAndDestryWithoutCommit")
    {
        result = RegisterApplyDeltaTaskAndDestryWithoutCommit();
    }
    else if (command == "RegisterOnly")
    {
        result = RegisterOnly();
    }
    else if (command == "RegisterApplyDeltaTaskAndDestryIntermediately")
    {
        result = RegisterApplyDeltaTaskAndDestryIntermediately();
    }
    else if (command == "CleanupPlaceholders")
    {
        result = CleanupPlaceholders();
    }
    else if (command == "WaitApplyDeltaWithoutCommit")
    {
        result = WaitApplyDeltaWithoutCommit();
    }
    else if (command == "WaitApplyDeltaAndReportOccupiedSize")
    {
        result = WaitApplyDeltaAndReportOccupiedSize();
    }
    else if (command == "SIGLO-48725")
    {
        result = SIGLO_48725();
    }
    else if (command == "SIGLO-47637")
    {
        result = SIGLO_47637();
    }
    else if (command == "SIGLO-51171")
    {
        result = SIGLO_51171();
    }
    else if (command == "NXBTS-11123")
    {
        result = NXBTS_11123();
    }
    else if (command == "NXBTS-12466")
    {
        result = NXBTS_12466();
    }
    else if (command == "CreateContentMetaDatabasePatchEntry")
    {
        result = CreateContentMetaDatabasePatchEntry();
    }
    else
    {
        NN_LOG("[Failure] Unknown command %s\n", command.c_str());
        return;
    }

    if(result.IsFailure())
    {
        NN_LOG("[Failed] %08x\n", result.GetInnerValueForDebug());
    }
    else
    {
        NN_LOG("[Success]\n");
    }
} // NOLINT(impl/function_size)
