﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <algorithm>

#include <nn/nn_Log.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_FormatString.h>
#include <nn/lmem/lmem_ExpHeap.h>

#include <nn/fs.h>
#include <nn/nn_Assert.h>
#include <nn/crypto/crypto_Sha256Generator.h>

namespace
{
nn::Bit8 g_HeapMemory[4 << 10];
nn::lmem::HeapHandle g_HeapHandle;

void* Allocate(size_t size) NN_NOEXCEPT
{
    return nn::lmem::AllocateFromExpHeap(g_HeapHandle, size);
}

void Deallocate(void* p, size_t size) NN_NOEXCEPT
{
    NN_UNUSED(size);
    nn::lmem::FreeToExpHeap(g_HeapHandle, p);
}

const char* HexToString(const nn::Bit8* data, size_t size)
{
    // snprintf 用バッファ
    static char str[128];
    for(int i = 0; i < size && i < (sizeof(str) / 2); ++i)
    {
        nn::util::SNPrintf(&(str[i * 2]), sizeof(str) - (i * 2), "%02x", data[i]);
    }
    return str;
}

nn::Result CalculateHash(const char* path)
{
    nn::fs::FileHandle handle;
    static const int64_t BufferSize = 64 << 10;
    static nn::Bit8 buffer[BufferSize];

    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));
    NN_UTIL_SCOPE_EXIT { nn::fs::CloseFile(handle); };

    int64_t fileSize = 0;
    int64_t readSize = 0;
    NN_RESULT_DO(nn::fs::GetFileSize(&fileSize, handle));

    nn::crypto::Sha256Generator generator;
    generator.Initialize();
    while(readSize < fileSize)
    {
        size_t tryReadSize = std::min(BufferSize, fileSize - readSize);
        NN_RESULT_DO(nn::fs::ReadFile(handle, readSize, buffer, tryReadSize));
        generator.Update(buffer, tryReadSize);

        readSize += tryReadSize;
    }
    nn::Bit8 hash[nn::crypto::Sha256Generator::HashSize];
    generator.GetHash(hash, nn::crypto::Sha256Generator::HashSize);

    NN_LOG("%s %s\n", path, HexToString(hash, nn::crypto::Sha256Generator::HashSize));

    NN_RESULT_SUCCESS;
}
nn::Result Traverse(const char* path)
{
    nn::fs::DirectoryHandle handle;
    NN_RESULT_DO(nn::fs::OpenDirectory(&handle, path, nn::fs::OpenDirectoryMode_All));
    NN_UTIL_SCOPE_EXIT {nn::fs::CloseDirectory(handle); };

    for(;;)
    {
        int64_t outCount;
        nn::fs::DirectoryEntry entry;
        NN_RESULT_DO(nn::fs::ReadDirectory(&outCount, &entry, handle, 1));

        if (outCount == 0)
        {
            break;
        }
        if (entry.directoryEntryType == nn::fs::DirectoryEntryType_File)
        {
            char nextPath[128];
            nn::util::SNPrintf(nextPath, sizeof(nextPath), "%s%s", path, entry.name);
            auto result = CalculateHash(nextPath);
            if (result.IsFailure())
            {
                NN_LOG("CalculateHash failed, %s %08x\n", nextPath, result.GetInnerValueForDebug());
            }
        }
        else if (entry.directoryEntryType == nn::fs::DirectoryEntryType_Directory)
        {
            // stack なのでサイズには注意
            char nextPath[128];
            nn::util::SNPrintf(nextPath, sizeof(nextPath), "%s%s/", path, entry.name);
            auto result = Traverse(nextPath);
            if (result.IsFailure())
            {
                NN_LOG("Traverse failed, %s %08x\n", nextPath, result.GetInnerValueForDebug());
            }
        }
    }
    NN_RESULT_SUCCESS;
}

void ReadRomfs()
{
    static const size_t MountRomCacheBufferSize = 4 * 1024;
    static char s_MountRomCacheBuffer[MountRomCacheBufferSize];

    g_HeapHandle = nn::lmem::CreateExpHeap(g_HeapMemory, sizeof(g_HeapMemory), nn::lmem::CreationOption_NoOption);
    nn::fs::SetAllocator(Allocate, Deallocate);

    size_t mountRomCacheUseSize = 0;
    auto result = nn::fs::QueryMountRomCacheSize(&mountRomCacheUseSize);
    NN_ASSERT(result.IsSuccess());
    NN_ASSERT_LESS_EQUAL(mountRomCacheUseSize, MountRomCacheBufferSize);

    nn::fs::MountRom("rom", s_MountRomCacheBuffer, MountRomCacheBufferSize);
    NN_UTIL_SCOPE_EXIT { nn::fs::Unmount("rom"); };
    Traverse("rom:/");
}

}

