﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace ManyApplicationInstallTest
{
    [TestClass]
    public class InstallTest
    {
        public TestContext TestContext { get; set; }
        private static string m_TargetName = string.Empty;

        [ClassInitialize]
        public static void TestClassinitialize(TestContext context)
        {
            m_TargetName = (string)context.Properties["TargetName"];
        }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments, FileStream fsOutput = null, FileStream fsErrorOutput = null)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if(fsOutput != null && e.Data != null)
                {
                    fsOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                if (fsErrorOutput != null && e.Data != null)
                {
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes(e.Data), 0, e.Data.Length);
                    fsErrorOutput.Write(Encoding.UTF8.GetBytes("\n"), 0, 1);
                }
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }

        private void RunAndWaitProcess(Process process)
        {
            Console.WriteLine("process start.");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            Console.WriteLine("process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteMakeTestApplication(string outputPath, string xmlPath)
        {
            Console.WriteLine("ExecuteMakeTestApplication start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeTestApplication\\MakeTestApplication.exe";
                var arguments = " -o " + outputPath + " --config-xml=" + xmlPath + " -j 8 --small-code";
                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool ExecuteApplicationManagerTestTool(string outputPath, string csvPath, bool uninstallApplicationAfterExecute)
        {
            Console.WriteLine("ExecuteApplicationManagerTestTool start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var toolPath = testPath.GetTestRoot() + "\\Outputs\\NX-NXFP2-a64\\TargetTools\\ApplicationManagerTestTool\\Develop\\ApplicationManagerTestTool.nsp";

                string arguments;
                if(uninstallApplicationAfterExecute)
                {
                    arguments = " -- " + toolPath + " " + csvPath;
                }
                else
                {
                    arguments = " -- " + toolPath + " " + csvPath + " --no-uninstall";
                }

                var csvSplitPath = csvPath.Split(new char[] { '\\' });
                var txtOutput = (outputPath + "\\TestResult_" + csvSplitPath[csvSplitPath.Length - 1]).Replace(".csv", ".txt");
                var txtErrorOutput = (outputPath + "\\ErrorTestResult_" + csvSplitPath[csvSplitPath.Length - 1]).Replace(".csv", ".txt");
                var fsOutput = new FileStream(txtOutput, FileMode.Create, FileAccess.ReadWrite);
                var fsErrorOutput = new FileStream(txtErrorOutput, FileMode.Create, FileAccess.ReadWrite);
                var outputs = SetupProcess(process, filename, arguments, fsOutput, fsErrorOutput);
                RunAndWaitProcess(process);

                fsOutput.Close();
                fsErrorOutput.Close();
                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                var success = "[  PASSED  ] 1 test.";
                return outputs.Item1.ToString().Contains(success);
            }
        }

        private bool RunInstalledApplication(string applicationId, string successText)
        {
            Console.WriteLine("RunInstalledApplication start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var arguments = " -- " + applicationId;

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return outputs.Item1.ToString().Contains(successText);
            }
        }

        private bool ExecuteDevMenuCommandSystem(string exeArgs, string nspArgs, string outputPath)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(exeArgs, nspArgs, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommandSystem\\Develop\\DevMenuCommandSystem.nsp", outputPath);
        }

        private bool ExecuteDevMenuCommandCommon(string exeArgs, string nspArgs, string nspPath, string outputPath)
        {
            Console.WriteLine("ExecuteDevMenuCommandCommon start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTargetPrivate.exe";
                var arguments = $"run { nspPath } -t {m_TargetName} { exeArgs } --pattern-failure-exit \\[FAILURE\\] -- { nspArgs }";
                var txtOutput = (outputPath + "\\DevMenuCommandSystemOutput.txt");
                var txtErrorOutput = (outputPath + "\\DevMenuCommandSystemErrorOutput.txt");
                var fsOutput = new FileStream(txtOutput, FileMode.Append, FileAccess.Write);
                var fsErrorOutput = new FileStream(txtErrorOutput, FileMode.Append, FileAccess.Write);
                var outputs = SetupProcess(process, filename, " " + arguments, fsOutput, fsErrorOutput);
                RunAndWaitProcess(process);
                fsOutput.Close();
                fsErrorOutput.Close();

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool CountMakeFile(DirectoryInfo directory, int fileCount)
        {
            Console.WriteLine("All nsp file list in " + directory.FullName);
            IEnumerable<FileInfo> infos = directory.EnumerateFiles("*.nsp", SearchOption.TopDirectoryOnly);
            int number = 0;

            foreach (FileInfo info in infos)
            {
                ++number;
                Console.WriteLine(number + ". " + info.Name);
            }

            return number == fileCount;
        }

        [TestMethod]
        public void MakeTestApplicationTest()
        {
            var testPath = new TestUtility.TestPath(this.TestContext);
            var xmlPath = testPath.GetTestRoot() + "\\Tools\\Sources\\Tests\\ManyApplicationInstallTest\\TestData\\config_4096.xml";
            DirectoryInfo nspOutputDirectory = Directory.CreateDirectory(testPath.GetTestRoot() + "\\Outputs\\AnyCPU\\Tools\\ManyApplicationInstallTest\\nspOutputDirectory");

            Assert.IsTrue(ExecuteMakeTestApplication(nspOutputDirectory.FullName, xmlPath));
            Assert.IsTrue(CountMakeFile(nspOutputDirectory, 4096));

            DirectoryInfo testOutputDirectory = Directory.CreateDirectory(testPath.GetTestRoot() + "\\Outputs\\AnyCPU\\Tools\\ManyApplicationInstallTest\\testOutputDirectory");

            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "sdcard format", testOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem("--reset", "application list", testOutputDirectory.FullName));
            var csvPath_2048 = testPath.GetTestRoot() + "\\Tools\\Sources\\Tests\\ManyApplicationInstallTest\\TestData\\TestSetting_2048.csv";
            Assert.IsTrue(ExecuteApplicationManagerTestTool(testOutputDirectory.FullName, csvPath_2048, true));

            var csvPath_4096 = testPath.GetTestRoot() + "\\Tools\\Sources\\Tests\\ManyApplicationInstallTest\\TestData\\TestSetting_4096.csv";
            Assert.IsTrue(ExecuteApplicationManagerTestTool(testOutputDirectory.FullName, csvPath_4096, false));

            int[] RunApplicationIndex =
            {
                0, 1, 2, 3, 4, 5, 7, 8, 9, 10, 11, 15, 16, 17, 31, 32, 33, 63, 64, 65, 99, 100, 101, 127, 128, 129, 199, 200, 201, 255, 256, 257, 299, 300, 301, 399, 400, 401,
                499, 500, 501, 511, 512, 513, 599, 600, 601, 699, 700, 701, 799, 800, 801, 899, 900, 901, 999, 1000, 1001, 1023, 1024, 1025, 1499, 1500, 1501, 1999, 2000, 2001,
                2047, 2048, 2049, 2499, 2500, 2501, 2999, 3000, 3001, 3499, 3500, 3501, 3999, 4000, 4001, 4094, 4095,
            };

            foreach(var i in RunApplicationIndex)
            {
                var applicationId = "0x" + (0100000000010001 + i).ToString("D16");
                Assert.IsTrue(RunInstalledApplication(applicationId, "HashCheck : OK"));
            }

            nspOutputDirectory.Delete(true);
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "application uninstall --all", testOutputDirectory.FullName));
            Assert.IsTrue(ExecuteDevMenuCommandSystem(string.Empty, "ticket delete-all", testOutputDirectory.FullName));
        }
    }
}

