﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.MakeSampleSolution;
using Nintendo.MakeVisualStudioProject;

namespace Nintendo.MakeSampleSolutionTest
{
    [TestClass]
    public class SolutionConfigurationResolverTest
    {
        private readonly string[] AllBuildTypes = new[]
        {
            "Debug", "Develop", "Release"
        };

        private readonly string[] AllGenericConfigurations = new[]
        {
            "Debug", "Develop", "Release",
            "VS2013_Debug", "VS2013_Develop", "VS2013_Release",
            "VS2015_Debug", "VS2015_Develop", "VS2015_Release",
        };
        private readonly string[] AllNXConfigurations = new[]
        {
            "NX_Debug", "NX_Develop", "NX_Release",
            "NX_VS2013_Debug", "NX_VS2013_Develop", "NX_VS2013_Release",
            "NX_VS2015_Debug", "NX_VS2015_Develop", "NX_VS2015_Release",
        };

        private IEnumerable<ProjectConfiguration> GenerateProjectConfigurations(
            IEnumerable<string> platforms,
            IEnumerable<string> configurations)
        {
            foreach (var platform in platforms)
            {
                foreach (var configuration in configurations)
                {
                    yield return new ProjectConfiguration(configuration, platform);
                }
            }
        }

        private IEnumerable<SolutionConfiguration> GenerateSolutionConfigurations(
            IEnumerable<string> platforms,
            IEnumerable<string> buildTypes,
            string spec,
            string vsVersion)
        {
            foreach (var platform in platforms)
            {
                foreach (var buildType in buildTypes)
                {
                    yield return new SolutionConfiguration(buildType, platform, spec, vsVersion);
                }
            }
        }

        [TestMethod]
        public void TestWindowsPlatformMap()
        {
            var projectConfigurations = GenerateProjectConfigurations(new[] { "Win32", "x64" }, AllGenericConfigurations);

            var solutionConfiguration = new SolutionConfiguration("Debug", "Win32", "Generic", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("Debug", "Win32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Develop", "x64", "Generic", "VS2013");
            Assert.AreEqual(
                new ProjectConfiguration("VS2013_Develop", "x64"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Release", "Win32", "Generic", "VS2015");
            Assert.AreEqual(
                new ProjectConfiguration("VS2015_Release", "Win32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            //-------------------------------------------------------------------------------------
            projectConfigurations = GenerateProjectConfigurations(new[] { "Win32", "x64" }, AllNXConfigurations);

            solutionConfiguration = new SolutionConfiguration("Debug", "x64", "NX", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("NX_Debug", "x64"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Develop", "Win32", "NX", "VS2013");
            Assert.AreEqual(
                new ProjectConfiguration("NX_VS2013_Develop", "Win32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Release", "x64", "NX", "VS2015");
            Assert.AreEqual(
                new ProjectConfiguration("NX_VS2015_Release", "x64"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));
        }

        [TestMethod]
        public void TestDoNotBuildResolutionPreference()
        {
            var projectConfigurations = GenerateProjectConfigurations(new[] { "Win32" }, AllGenericConfigurations);

            // 対応するプラットフォームがない
            var solutionConfiguration = new SolutionConfiguration("Debug", "x64", "Generic", "VS2012");
            Assert.AreEqual(
                null,
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            // 対応するスペックがない
            solutionConfiguration = new SolutionConfiguration("Develop", "Win32", "NX", "VS2013");
            Assert.AreEqual(
                null,
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            //-------------------------------------------------------------------------------------
            projectConfigurations = GenerateProjectConfigurations(new[] { "Win32" }, new[] { "NX_Debug", "NX_Release" });

            // 対応するビルドタイプがない
            solutionConfiguration = new SolutionConfiguration("Develop", "Win32", "NX", "VS2012");
            Assert.AreEqual(
                null,
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            // 対応する Visual Studio バージョンがない
            solutionConfiguration = new SolutionConfiguration("Debug", "Win32", "NX", "VS2013");
            Assert.AreEqual(
                null,
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Release", "Win32", "NX", "VS2015");
            Assert.AreEqual(
                null,
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));
        }

        [TestMethod]
        public void TestNonWindowsPlatformMap()
        {
            var projectConfigurations = GenerateProjectConfigurations(new[] { "NX32" }, AllNXConfigurations);

            var solutionConfiguration = new SolutionConfiguration("Debug", "NX32", "NX", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("NX_Debug", "NX32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            // Windows プラットフォームでない場合は Visual Studio 2012 の構成にマップされる
            solutionConfiguration = new SolutionConfiguration("Develop", "NX32", "NX", "VS2013");
            Assert.AreEqual(
                new ProjectConfiguration("NX_Develop", "NX32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));

            solutionConfiguration = new SolutionConfiguration("Release", "NX32", "NX", "VS2015");
            Assert.AreEqual(
                new ProjectConfiguration("NX_Release", "NX32"),
                SolutionConfigurationResolver.Resolve(solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.DoNotBuild));
        }

        [TestMethod]
        public void TestCustomMapResolutionPreference()
        {
            var projectConfigurations = GenerateProjectConfigurations(new[] { "Win32" }, new[] { "Debug", "Develop", "Release" });
            var resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Debug", "x64", "Debug", "Win32"),
                new SolutionConfigurationResolver.MapRule("Develop", "x64", "Develop", "Win32"),
                new SolutionConfigurationResolver.MapRule("Release", "x64", "Release", "Win32"),
            };

            var solutionConfiguration = new SolutionConfiguration("Debug", "x64", "Generic", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("Debug", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            solutionConfiguration = new SolutionConfiguration("Develop", "x64", "Generic", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("Develop", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            solutionConfiguration = new SolutionConfiguration("Release", "x64", "Generic", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("Release", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));
        }

        [TestMethod]
        public void TestCustomMapResolutionPreferenceException()
        {
            var projectConfigurations = GenerateProjectConfigurations(new[] { "Win32" }, new[] { "NX_Debug", "NX_Release" });
            var resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Debug", "x64", "NX_Debug", "Win32"),
                new SolutionConfigurationResolver.MapRule("Release", "x64", "NX_Release", "Win32"),
            };

            // 対応するソリューションプラットフォームが解決ルールにない
            var solutionConfiguration = new SolutionConfiguration("Debug", "NX32", "NX", "VS2012");
            AssertUtil.ExpectException<ErrorException>(() =>
            {
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules);
            });

            // 対応するソリューション構成が解決ルールにない
            solutionConfiguration = new SolutionConfiguration("Develop", "x64", "NX", "VS2015");
            AssertUtil.ExpectException<ErrorException>(() =>
            {
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules);
            });

            //-------------------------------------------------------------------------------------
            resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Debug", "NX64", "NX_Debug", "x64"),
                new SolutionConfigurationResolver.MapRule("Release", "NX64", "NX_Release", "x64"),
            };

            // 解決先のプロジェクトプラットフォームがない
            solutionConfiguration = new SolutionConfiguration("Debug", "NX64", "NX", "VS2012");
            AssertUtil.ExpectException<ErrorException>(() =>
            {
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules);
            });

            //-------------------------------------------------------------------------------------
            resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Debug", "x64", "NX_VS2013_Debug", "Win32"),
                new SolutionConfigurationResolver.MapRule("Release", "x64", "NX_VS2013_Release", "Win32"),
            };

            // 解決先のプロジェクト構成がない
            solutionConfiguration = new SolutionConfiguration("Release", "x64", "NX", "VS2013");
            AssertUtil.ExpectException<ErrorException>(() =>
            {
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules);
            });
        }

        [TestMethod]
        public void TestCustomMapResolutionPreferenceWildCard()
        {
            const string ANY = SolutionConfigurationResolver.MapRule.ANY;

            var projectConfigurations = GenerateProjectConfigurations(new[] { "Win32", "x64" }, AllNXConfigurations);
            var resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule(ANY, "NX32", "NX_VS2013_Release", "Win32"),
            };

            var solutionConfiguration = new SolutionConfiguration("Debug", "NX32", "NX", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("NX_VS2013_Release", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            //-------------------------------------------------------------------------------------
            resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule(ANY, "NX32", ANY, "Win32"),
                new SolutionConfigurationResolver.MapRule(ANY, "NX64", ANY, "x64"),
            };

            solutionConfiguration = new SolutionConfiguration("Debug", "NX32", "NX", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("NX_Debug", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            solutionConfiguration = new SolutionConfiguration("Develop", "NX64", "NX", "VS2013");
            Assert.AreEqual(
                new ProjectConfiguration("NX_VS2013_Develop", "x64"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            solutionConfiguration = new SolutionConfiguration("Release", "NX32", "NX", "VS2015");
            Assert.AreEqual(
                new ProjectConfiguration("NX_VS2015_Release", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            //-------------------------------------------------------------------------------------
            projectConfigurations = GenerateProjectConfigurations(new[] { "Win32", "x64" }, new[] { "Debug", "Release" });
            resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Develop", ANY, "Release", "Win32"),
            };

            solutionConfiguration = new SolutionConfiguration("Develop", "x64", "Generic", "VS2012");
            Assert.AreEqual(
                new ProjectConfiguration("Release", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            //-------------------------------------------------------------------------------------
            resolutionRules = new[]
            {
                new SolutionConfigurationResolver.MapRule("Develop", ANY, "Release", ANY),
            };

            solutionConfiguration = new SolutionConfiguration("Develop", "Win32", "Generic", "VS2013");
            Assert.AreEqual(
                new ProjectConfiguration("Release", "Win32"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));

            solutionConfiguration = new SolutionConfiguration("Develop", "x64", "Generic", "VS2015");
            Assert.AreEqual(
                new ProjectConfiguration("Release", "x64"),
                SolutionConfigurationResolver.Resolve(
                    solutionConfiguration, projectConfigurations, SolutionConfigurationResolutionPreference.CustomMap, resolutionRules));
        }
    }
}
