﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Security.Cryptography;

namespace MakeNrrTest
{
    [TestClass]
    public class BinaryTest : TestBase
    {
        byte[] ReadFile(FileManager fd)
        {
            using (FileStream fs = new FileStream(fd.FilePath, FileMode.Open, FileAccess.Read))
            {
                BinaryReader br = new BinaryReader(fs);
                using (MemoryStream ms = new MemoryStream())
                {
                    fs.CopyTo(ms);
                    return ms.ToArray();
                }
            }
        }

        void CheckSignatureArea(byte[] contents, int offset)
        {
            // ApplicationIdMask が正しい
            UInt64 applicationIdMask = BitConverter.ToUInt64(contents, NrrCertificationFormat.ApplicationMaskOffset + offset);
            Assert.AreEqual(applicationIdMask, 0ul);

            // ApplicationIdPattern が正しい
            UInt64 applicationIdPattern = BitConverter.ToUInt64(contents, NrrCertificationFormat.ApplicationIdPatternOffset + offset);
            Assert.AreEqual(applicationIdPattern, 0ul);

            byte[] correctSign = new byte[0x100];
            byte[] sign = new byte[0x100];
            Array.Copy(contents, NrrCertificationFormat.SignOffset, sign, 0, 0x100);

            // 証明書の署名領域が正しい
            Assert.IsTrue(sign.SequenceEqual(correctSign));

            // NRR ファイルの署名領域が正しい
            byte[] nrrSign = new byte[0x100];
            Array.Copy(contents, NrrCertificationFormat.PublicKeyOffset + offset, nrrSign, 0, nrrSign.Length);
            Assert.IsTrue(nrrSign.SequenceEqual(correctSign));
        }

        static int CompareHash(byte[] a, byte[] b)
        {
            Assert.IsTrue(a.Length == b.Length);
            for (int i = 0; i < a.Length; i++)
            {
                int result = a[i] - b[i];
                if (result == 0)
                {
                    continue;
                }
                else
                {
                    return result;
                }
            }
            return 0;
        }

        [TestMethod]
        public void CheckBinary()
        {
            string[] inputFiles = new string[]
            {
                GetTestNroFilePath(TestNroFile1),
                GetTestNroFilePath(TestNroFile2)
            };

            FileManager nrr = SuccessExecute(inputFiles);
            byte[] contents = ReadFile(nrr);

            // Signature が正しい
            uint signature = BitConverter.ToUInt32(contents, NrrFormat.SignatureOffset);
            Assert.AreEqual(signature, NrrFormat.Signature);

            // 証明書領域が正しい
            // 署名領域が正しい
            CheckSignatureArea(contents, NrrFormat.CertOffset);

            // アプリケーションID が正しい
            UInt64 applicationId = BitConverter.ToUInt64(contents, NrrFormat.ApplicationIdOffset);
            Assert.AreEqual(applicationId, 0ul);


            // サイズが正しい
            UInt32 size = BitConverter.ToUInt32(contents, NrrFormat.SizeOffset);
            Assert.AreEqual(size, 0x390u);

            // ハッシュ値領域のオフセットアドレスが正しい
            UInt32 hashOffset = BitConverter.ToUInt32(contents, NrrFormat.HashAddressOffset);
            Assert.AreEqual(hashOffset, 0x350u);

            // 登録されているハッシュ値の個数が正しい
            UInt32 numHash = BitConverter.ToUInt32(contents, NrrFormat.NumHashOffset);
            Assert.AreEqual(numHash, 2u);

            List<byte[]> hashList = new List<byte[]>();

            // 登録されているハッシュ値が正しい
            for (int i = 0; i < inputFiles.Length; i++)
            {
                var f = inputFiles[i];
                byte[] data;
                using (FileStream fs = new FileStream(f, FileMode.Open, FileAccess.Read))
                {
                    using (MemoryStream ms = new MemoryStream())
                    {
                        fs.CopyTo(ms);
                        data = ms.ToArray();
                        SHA256Managed sha256 = new SHA256Managed();
                        byte[] hash = sha256.ComputeHash(data);
                        hashList.Add(hash);
                    }
                }
            }
            hashList.Sort(CompareHash);
            foreach (var item in hashList.Select((hash, i) => new { hash, i }))
            {
                byte[] data = new byte[NrrFormat.HashSize];
                Array.Copy(contents, hashOffset + item.i * NrrFormat.HashSize, data, 0, NrrFormat.HashSize);
                Assert.IsTrue(item.hash.SequenceEqual(data));
            }
        }
    }
}
