﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nnt.h>
#include <gtest/gtest.h>
#include <cstdarg>
#include <memory>
#include <vector>
#include <string>
#include <algorithm>

#include <nn/nn_Result.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_Log.h>
#include <nn/nn_Assert.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_IntrusiveList.h>
#include <nn/cstd/cstd_CStdArg.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_TFormatString.h>

#include "utilTool_CommandInterface.h"
#include "utilTool_CommandLog.h"
#include "utilTool_CommandFramework.h"

TEST(TestArgument, MakeFatImageTest)
{
    bool isSparseImage;
    bool isVerbose;
    std::string outputPath;
    std::string positionalArgument;
    std::vector<std::string> variableArguments;

    SingleCommandInterface command;
    command.SetName("TestCommand");
    command.SetSummary("Do something.");
    command.AddKeywordArgument(&outputPath, 'o', "output", "Output generated data.", false);
    command.AddPositionalArgument(&positionalArgument, 0, "PositionalArgument", "Set positional argument.", true);
    command.SetVariableArguments(&variableArguments, "VariableArguments", "Set many arguments.", true);
    command.AddFlagArgument(&isSparseImage, "sparse-image", "Make sparse image.");
    command.AddFlagArgument(&isVerbose, 'v', "verbose", "Output many logs.");

    EXPECT_TRUE(command.TryParseVa(4, "someargument", "someargument", "--sparse-image", "--verbose"));

    // 名前付きフラグを取れる
    EXPECT_TRUE(isSparseImage);
    EXPECT_TRUE(isVerbose);

    EXPECT_TRUE(command.TryParseVa(2, "someargument", "someargument"));

    // 2回目も適切に処理される
    EXPECT_FALSE(isSparseImage);
    EXPECT_FALSE(isVerbose);

    // 2回指定してはいけない
    EXPECT_FALSE(command.TryParseVa(5, "someargument", "someargument", "--sparse-image", "--verbose", "--verbose"));
    EXPECT_FALSE(command.TryParseVa(5, "someargument", "someargument", "--sparse-image", "--verbose", "-v"));

    // 名前付き引数を取れる
    EXPECT_TRUE(command.TryParseVa(3, "exe", "-o", "outputPath"));
    EXPECT_EQ("outputPath", outputPath);

    // 位置引数を取れる
    EXPECT_TRUE(command.TryParseVa(2, "-v", "positionalArgument"));
    EXPECT_EQ("positionalArgument", positionalArgument);

    // 位置引数が無い
    EXPECT_FALSE(command.TryParseVa(1, "-v"));

    // HelpText を取得できる
    NN_LOG("%s\n", MakePlainTextCommandDocument(command.GetDocument()).c_str());
}

TEST(TestCommandLog, MakeFatImageTest)
{
    DefaultCommandLogObserver defaultObserver;
    defaultObserver.SetPositionDisplay(true);
    defaultObserver.SetLevel(CommandLogLevel_Verbose);
    RegisterCommandLogObserver(defaultObserver);
    NN_UTIL_SCOPE_EXIT{ UnregisterCommandLogObserver(defaultObserver); };

    NN_UTILTOOL_LOG_FORCE("Force");
    NN_UTILTOOL_LOG_ERROR("Error");
    NN_UTILTOOL_LOG_WARNING("Warning");
    NN_UTILTOOL_LOG_PROGRESS("1/10");
    NN_UTILTOOL_LOG_DEBUG("Debug");
}

class SomeCommand
{
public:
    const char* Name()
    {
        return "SomeCommand";
    }

    const char* Summary()
    {
        return "Do something.";
    }

    const char* Description()
    {
        return
            "description 1.\n"
            "description 2.\n"
            "description 3.";
    }

    void Arguments(SingleCommandInterface &interface)
    {
        interface.AddPositionalArgument(&m_OutputPath, 0, "OutputPath", "Output this file.", true);
        interface.AddPositionalArgument(&m_InputPath, 1, "InputPath", "Output this file.", true);
    }

    int Main()
    {
        NN_UTILTOOL_LOG_VERBOSE("Do something from %s to %s.", m_InputPath.c_str(), m_OutputPath.c_str());
        return 0;
    }

private:
    std::string m_OutputPath;
    std::string m_InputPath;
};


TEST(TestCommandFramework, Basic)
{
    {
        int argc = 2;
        const char* argv[] = { "exe", "--help" };
        int statusCode;

        statusCode = BasicCommandFramework<SomeCommand>::Run(argc, argv);

        EXPECT_EQ(0, statusCode);
    }

    {
    int argc = 4;
    const char* argv[] = { "exe", "TestOutput", "TestInput", "--verbose" };
    int statusCode;

    statusCode = BasicCommandFramework<SomeCommand>::Run(argc, argv);

    EXPECT_EQ(0, statusCode);
}
}
