﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using TestUtility;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace MakeDescTest
{
    [TestClass]
    public class TestReadDescFile : TestBase
    {
        [TestMethod]
        public void TestDescTag()
        {
            // Desc ファイルを読み込める
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 想定されていないタグを受け取らない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Meta");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                desc.AddLine(elem.GetXml());
                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, null, null);
            }
        }

        [TestMethod]
        public void TestProgramIdMin()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);

                XmlElement idTag2 = new XmlElement("ProgramIdMax");
                idTag2.InnerText = "0x2";
                elem.AddChild(idTag2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0x0";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0xFFFFFFFFFFFFFFFF";
                elem.AddChild(idTag);

                XmlElement idTag2 = new XmlElement("ProgramIdMax");
                idTag2.InnerText = "0xFFFFFFFFFFFFFFFF";
                elem.AddChild(idTag2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 10 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "1";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Desc/ProgramIdMin"), desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);
                XmlElement idTag2 = new XmlElement("ProgramIdMin");
                idTag.InnerText = "0x2";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "ProgramIdMin"), desc, null, null);
            }
        }

        [TestMethod]
        public void TestProgramIdMax()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0x0";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0xFFFFFFFFFFFFFFFF";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 10 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "1";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Desc/ProgramIdMax"), desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0x1";
                elem.AddChild(idTag);
                XmlElement idTag2 = new XmlElement("ProgramIdMax");
                idTag.InnerText = "0x2";
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "ProgramIdMax"), desc, null, null);
            }
        }

        [TestMethod]
        public void TestFsAccessControlDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("FsAccessControlDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }
        }

        [TestMethod]
        public void TestSrvAccessControlDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // Entry を読み込める
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // Entry 以外を受け付けない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entries");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "Entries"), desc, null, null);
            }

            // 重複を許可する
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                srv.AddChild(entry);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // IsServer に True を指定できる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "TRUE";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // IsServer に False を指定できる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "False";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // True/False 以外の文字列を受け取らない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "Success";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Entry/IsServer"), desc, null, null);
            }

            // Entry が宣言されている場合、IsServer が指定されていないとエラーになる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/IsServer"), desc, null, null);
            }

            // IsServer を重複して宣言出来ない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement isServer2 = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "False";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "IsServer"), desc, null, null);
            }

            // サービス名を指定できる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 長さ 0 のサービス名を許容しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = string.Empty;

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/Name"), desc, null, null);
            }

            // 8文字の文字列を許容する
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "12345678";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 9 文字以上の長さの文字列を許容しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "123456789";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidStringLength, "Entry/Name", 9), desc, null, null);
            }

            // Entry が指定されている場合、 Name が指定されていないとエラーになる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/Name"), desc, null, null);
            }

            // Name を重複して宣言出来ない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlDescriptor");
                elem.AddChild(srv);

                XmlElement entry = new XmlElement("Entry");
                srv.AddChild(entry);

                XmlElement isServer = new XmlElement("IsServer");
                entry.AddChild(isServer);
                isServer.InnerText = "True";

                XmlElement name = new XmlElement("Name");
                entry.AddChild(name);
                name.InnerText = "Test";

                XmlElement name2 = new XmlElement("Name");
                entry.AddChild(name2);
                name2.InnerText = "Name";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "Name"), desc, null, null);
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void TestKernelCapabilityDescriptor()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement idTag = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag);
                XmlElement idTag2 = new XmlElement("KernelCapabilityDescriptor");
                elem.AddChild(idTag2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "KernelCapabilityDescriptor"), desc, null, null);
            }

            // ThreadInfo を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeThreadInfoElement("0", "0", "0", "0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeThreadInfoElement("63", "63", "255", "255"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement("-1", "0", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", 6),
                    TestDataForKc.MakeThreadInfoElement("64", "0", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement(null, "0", "0", "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", "-1", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", 6),
                    TestDataForKc.MakeThreadInfoElement("0", "64", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", null, "0", "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "-1", "0"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "256", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", null, "0"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "-1"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "256"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", null));

                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("0", "1", "0", "0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("0", "0", "1", "0"));
            }

            // EnableSystemCalls を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableSystemCall("CreateThread"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    TestDataForKc.MakeEnableSystemCall("SleepThread"),
                });
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", "CreateThread"),
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    }
                );
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundSvc, "InvalidSvc"),
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("InvalidSvc"),
                });
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundSvc, "1"),
                    TestDataForKc.MakeEnableSystemCall("1"));
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x30000", "0x1000", "RW", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x40000", "0x1000", "Rw", "Static"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x50000", "0x3000", "Ro", "Static"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x30000", "0x2000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x40000", "0x2000", "RW", "Static"),
                    TestDataForKc.MakeMemoryMap("0x50000", "0x2000", "Ro", "Static"),
                });
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x0", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMemoryMap("0xFFFFFF000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "MemoryMap/BeginAddress"),
                    TestDataForKc.MakeMemoryMap("10000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35),
                    TestDataForKc.MakeMemoryMap("0x1000000000", "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/BeginAddress"),
                    TestDataForKc.MakeMemoryMap(null, "0x3000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_Invalid0, "MemoryMap/Size"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x0", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeMemoryMap("0x10000", "0x100000000", "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Size"),
                    TestDataForKc.MakeMemoryMap("0x10000", null, "Ro", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Permission", "RO", "RW"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "RX", "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Permission"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", null, "Io"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Type", "Io", "Static"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Normal"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Type"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", null));
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("10"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("1022"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeEnableInterrupt("1023"));
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("1023"),
                    TestDataForKc.MakeEnableInterrupt("23"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                SuccessTestKernelCapabilityDescriptor(new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("1023"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("0x10"));
                /* TORIAEZU: 一旦無効に
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_DuplicateDefinition, "EnableInterrupts", "45"),
                    new XmlElement[] {
                        TestDataForKc.MakeEnableInterrupt("45"),
                        TestDataForKc.MakeEnableInterrupt("45"),
                    }
                );
                */
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_Invalid0, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("0"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    TestDataForKc.MakeEnableInterrupt("-1"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfRange, "EnableInterrupts", 1024),
                    TestDataForKc.MakeEnableInterrupt("1024"));
            }

            // MiscParams を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("1"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscParams("7"));

                FailTestKernelCapabilityDescriptor(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", 8),
                    TestDataForKc.MakeMiscParams("8"));
                FailTestKernelCapabilityDescriptor(
                    null,
                    TestDataForKc.MakeMiscParams("-1"));
            }

            // HandleTableSize を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("256"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeHandleTableSize("1023"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidValue_Bigger, "KernelCapabilityDescriptor/HandleTableSize", 1023),
                    TestDataForKc.MakeHandleTableSize("1024"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityDescriptor/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("-1"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityDescriptor/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("0x1"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("True", "True"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("True", null));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags(null, "True"));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags("FALSE", null));
                SuccessTestKernelCapabilityDescriptor(TestDataForKc.MakeMiscFlags(null, "FALSE"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/EnableDebug"),
                    TestDataForKc.MakeMiscFlags("Success", "FALSE"));
                FailTestKernelCapabilityDescriptor(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/ForceDebug"),
                    TestDataForKc.MakeMiscFlags("True", "Fail"));
            }
        }

        [TestMethod]
        public void TestDefaultFsControlData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement fs = new XmlElement("FsAccessControlData");
                defaultElem.AddChild(fs);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement fs = new XmlElement("FsAccessControlData");
                elem.AddChild(fs);
                XmlElement fs2 = new XmlElement("FsAccessControlData");
                elem.AddChild(fs2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "FsAccessControlData"),  desc, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultSrvControlData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlData");
                defaultElem.AddChild(srv);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement srv = new XmlElement("SrvAccessControlData");
                elem.AddChild(srv);
                XmlElement srv2 = new XmlElement("SrvAccessControlData");
                elem.AddChild(srv2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "SrvAccessControlData"),  desc, null, null);
            }

            // エントリを登録できる
            SuccessTestSrvAccessControlDataWithSamSad(TestDataForSrv.MakeSrvEntry("Test", "False"));


            // エントリを複数個登録できる
            SuccessTestSrvAccessControlDataWithSamSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("Test", "False"),
                TestDataForSrv.MakeSrvEntry("Server", "True"),
            });

            // IsServer に True を指定できる
            SuccessTestSrvAccessControlDataWithSamSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("True", "True"),
                TestDataForSrv.MakeSrvEntry("true", "true"),
                TestDataForSrv.MakeSrvEntry("TRUE", "TRUE"),
                TestDataForSrv.MakeSrvEntry("tRUE", "tRUE"),
            });

            // IsServer に False を指定できる
            SuccessTestSrvAccessControlDataWithSamSad(new XmlElement[] {
                TestDataForSrv.MakeSrvEntry("False", "False"),
                TestDataForSrv.MakeSrvEntry("false", "false"),
                TestDataForSrv.MakeSrvEntry("FALSE", "FALSE"),
                TestDataForSrv.MakeSrvEntry("FALSe", "FALSe"),
            });

            XmlElement success = TestDataForSrv.MakeSrvEntry("Test", "true");

            // True/False 以外の文字列を受け取らない
            FailTestSrvAccessControlData(
                string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", "Failure"));

            // Entry が宣言されている場合、IsServer が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/IsServer"),
                success,
                TestDataForSrv.MakeSrvEntry("Test", null));

            // IsServer を重複して宣言出来ない
            {
                XmlFile desc = new XmlFile();

                // Descriptor 用
                XmlElement descriptor = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    descriptor.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    descriptor.AddChild(name);
                    name.InnerText = "Test";
                }

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement isServer2 = new XmlElement("IsServer");
                    data.AddChild(isServer2);
                    isServer2.InnerText = "False";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";
                }

                MakeDescFileForSrvAccessControlData(ref desc, new XmlElement[] { descriptor } , new XmlElement[] { data });
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "IsServer"), desc, null, null);
            }

            // サービス名を指定できる
            SuccessTestSrvAccessControlDataWithSamSad(TestDataForSrv.MakeSrvEntry("Test", "True"));

            // 長さ 0 のサービス名を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry("", "True"));

            // 8文字の文字列を許容する
            SuccessTestSrvAccessControlDataWithSamSad(TestDataForSrv.MakeSrvEntry("12345678", "True"));

            // 9 文字以上の長さの文字列を許容しない
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_InvalidStringLength, "Entry/Name", 9),
                success,
                TestDataForSrv.MakeSrvEntry("123456789", "True"));

            // Entry が指定されている場合、 Name が指定されていないとエラーになる
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Entry/Name"),
                success,
                TestDataForSrv.MakeSrvEntry(null, "True"));

            // Name を重複して宣言出来ない
            {
                XmlFile desc = new XmlFile();

                // Descriptor 用
                XmlElement descriptor = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    descriptor.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    descriptor.AddChild(name);
                    name.InnerText = "Test";
                }

                // Default 用
                XmlElement data = new XmlElement("Entry");
                {
                    XmlElement isServer = new XmlElement("IsServer");
                    data.AddChild(isServer);
                    isServer.InnerText = "True";

                    XmlElement name = new XmlElement("Name");
                    data.AddChild(name);
                    name.InnerText = "Test";

                    XmlElement name2 = new XmlElement("Name");
                    data.AddChild(name2);
                    name.InnerText = "Name";
                }

                MakeDescFileForSrvAccessControlData(ref desc, new XmlElement[] { descriptor } , new XmlElement[] { data });
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "Name"), desc, null, null);
            }
        }

        [TestMethod]
        public void DescriptorTestForSrvControlData()
        {
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Clie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            SuccessTestSrvAccessControlData(
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Cl*ie*", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Cl*ient", "False") }
                );

            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_DefaultNotPermitted, "Client", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "True") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
            FailTestSrvAccessControlData(
                string.Format(MakeDesc.Properties.Resources.Message_DefaultNotPermitted, "Server", "SrvAccessControlDescriptor"),
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Ser*", "True"), TestDataForSrv.MakeSrvEntry("Client", "False") },
                new XmlElement[] { TestDataForSrv.MakeSrvEntry("Server", "False"), TestDataForSrv.MakeSrvEntry("Client", "False") }
                );
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void TestDefaultKernelCapabilityData()
        {
            // 正常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kc = new XmlElement("KernelCapabilityData");
                defaultElem.AddChild(kc);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement kc = new XmlElement("KernelCapabilityData");
                defaultElem.AddChild(kc);
                XmlElement kc2 = new XmlElement("KernelCapabilityData");
                defaultElem.AddChild(kc2);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "KernelCapabilityData"), desc, null, null);
            }

            // ThreadInfo を読み込める
            {
                XmlElement success = TestDataForKc.MakeThreadInfoElement("63", "0", "0", "255");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("0", "0", "0", "0"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeThreadInfoElement("63", "63", "255", "255"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("-1", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "LowestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("64", "0", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/LowestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement(null, "0", "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "-1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "HighestPriority", 6),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "64", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", null, "0", "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "-1", "0"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "256", "0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", null, "0"));

                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "-1"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", "256"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_NotFound, "ThreadInfo/MaxCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "0", null));

                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_LessThan, "LowestPriority", "HighestPriority"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "1", "0", "0"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_LessThan, "MaxCoreNumber", "MinCoreNumber"),
                    success,
                    TestDataForKc.MakeThreadInfoElement("0", "0", "1", "0"));
            }


            // EnableSystemCalls を読み込める
            {
                XmlElement[] success = TestDataForKc.MakeEnableSystemCalls(new string[] { "CreateThread", "SleepThread" });
                SuccessTestKernelCapabilityData(success, new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") });
                SuccessTestKernelCapabilityData(success, new XmlElement[] {
                    TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    TestDataForKc.MakeEnableSystemCall("SleepThread"),
                });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_DuplicateDefinition, "EnableSystemCalls", "CreateThread"),
                    success,
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                    }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundSvc, "InvalidSvc"),
                    success,
                    new XmlElement[] {
                        TestDataForKc.MakeEnableSystemCall("CreateThread"),
                        TestDataForKc.MakeEnableSystemCall("InvalidSvc"),
                });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundSvc, "1"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("1") });
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x30000", "0x1000", "RW", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x40000", "0x1000", "Rw", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x50000", "0x3000", "Ro", "Static"));
                SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x20000", "0x1000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x30000", "0x2000", "RO", "Io"),
                    TestDataForKc.MakeMemoryMap("0x40000", "0x2000", "RW", "Static"),
                    TestDataForKc.MakeMemoryMap("0x50000", "0x2000", "Ro", "Static"),
                });
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0x3000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMemoryMap("0xFFFFFF000", "0x3000", "Ro", "Io"));

                XmlElement[] success = new XmlElement[] {
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Static"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Rw", "Io"),
                    TestDataForKc.MakeMemoryMap("0x0", "0xFFFFF000", "Ro", "Io"),
                };
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("10000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_ValueShouldBeUnsigned, "MemoryMap/BeginAddress", 35),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x1000000000", "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/BeginAddress"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap(null, "0x3000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_Invalid0, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x0", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_InvalidAlign, "MemoryMap/Size", 0x1000),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x1", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    null,
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x100000000", "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Size"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", null, "Ro", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Permission", "RO", "RW"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "RX", "Io") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Permission"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", null, "Io") });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString_2, "MemoryMap/Type", "Io", "Static"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Normal") });
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "MemoryMap/Type"),
                    success,
                    new XmlElement[] { TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", null) });
            }

            // EnableInterrupts を読み込める
            {
                XmlElement success = TestDataForKc.MakeEnableInterrupt("1023");
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeEnableInterrupt("10"));
                SuccessTestKernelCapabilityData(success, TestDataForKc.MakeEnableInterrupt("1022"));
                SuccessTestKernelCapabilityData(new XmlElement[] { success }, new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("23"),
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                SuccessTestKernelCapabilityData(new XmlElement[] { success }, new XmlElement[] {
                    TestDataForKc.MakeEnableInterrupt("45"),
                });
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0x10"));
                /* TORIAEZU: 一旦無効に
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_DuplicateDefinition, "EnableInterrupts", "45"),
                    new XmlElement[] { success },
                    new XmlElement[] {
                        TestDataForKc.MakeEnableInterrupt("45"),
                        TestDataForKc.MakeEnableInterrupt("45"),
                    }
                );
                */
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_Invalid0, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("0"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "EnableInterrupts"),
                    success,
                    TestDataForKc.MakeEnableInterrupt("-1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfRange, "EnableInterrupts", 1023),
                    success,
                    TestDataForKc.MakeEnableInterrupt("1023"));
            }

            // MiscParams を読み込める
            {
                XmlElement success = TestDataForKc.MakeMiscParams("1");
                SuccessTestKernelCapabilityDataWithSameKc(success);
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscParams("7"));

                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfRange, "MiscParams/ProgramType", 8),
                    success,
                    TestDataForKc.MakeMiscParams("8"));
                FailTestKernelCapabilityData(
                    null,
                    success,
                    TestDataForKc.MakeMiscParams("-1"));
            }

            // HandleTableSize を読み込める
            {
                XmlElement success = TestDataForKc.MakeHandleTableSize("1023");
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeHandleTableSize("256"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeHandleTableSize("1023"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidValue_Bigger, "KernelCapabilityData/HandleTableSize", 1023),
                    success,
                    TestDataForKc.MakeHandleTableSize("1024"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("-1"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "KernelCapabilityData/HandleTableSize"),
                    success,
                    TestDataForKc.MakeHandleTableSize("0x1"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags("True", "True"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags("True", null));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags(null, "True"));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags("FALSE", null));
                SuccessTestKernelCapabilityDataWithSameKc(TestDataForKc.MakeMiscFlags(null, "FALSE"));

                XmlElement success = TestDataForKc.MakeMiscFlags("True", "True");
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/EnableDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("Success", "FALSE"));
                FailTestKernelCapabilityData(
                    string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "MiscFlags/ForceDebug"),
                    success,
                    TestDataForKc.MakeMiscFlags("True", "Fail"));
            }
        }

        [TestMethod, TestCategory("MiddleTest")]
        public void DescriptorTestForKernelCapability()
        {
            // ThreadInfo
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeThreadInfoElement("63", "0", "0", "4"), TestDataForKc.MakeThreadInfoElement("59", "22", "1", "3"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "ThreadInfo/LowestPriority"),
                    TestDataForKc.MakeThreadInfoElement("1", "0", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "ThreadInfo/HighestPriority"),
                    TestDataForKc.MakeThreadInfoElement("2", "1", "0", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MinCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "0", "1"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "ThreadInfo/MaxCoreNumber"),
                    TestDataForKc.MakeThreadInfoElement("2", "0", "1", "1"), TestDataForKc.MakeThreadInfoElement("2", "0", "1", "2"));
            }

            // EnableSystemCalls
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread"), TestDataForKc.MakeEnableSystemCall("SleepThread") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread") }
                    );

                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapabilityWith, "EnableSystemCalls", "SetHeapSize"),
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("CreateThread"), TestDataForKc.MakeEnableSystemCall("SleepThread") },
                    new XmlElement[] { TestDataForKc.MakeEnableSystemCall("SetHeapSize") }
                    );
            }

            // MapMemory を読み込める
            {
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Ro", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                SuccessTestKernelCapabilityData(
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Ro", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x2000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Ro", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x2000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x4000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x4000", "Rw", "Io"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapabilityWithMemoryMap, "KernelCapabilityData", "MemoryMap", 0x10000, 0x1000, "RW", "Io"),
                    TestDataForKc.MakeMemoryMap("0x10000", "0x3000", "Rw", "Io"), TestDataForKc.MakeMemoryMap("0x10000", "0x1000", "Rw", "Io"));
            }

            // EnableInterrupts を読み込める
            {
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8") }
                );
                SuccessTestKernelCapabilityData(
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("1023") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapabilityWith, "EnableInterrupts",  8),
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("9") },
                    new XmlElement[] { TestDataForKc.MakeEnableInterrupt("4"), TestDataForKc.MakeEnableInterrupt("8"), TestDataForKc.MakeEnableInterrupt("9") }
                );

            }

            // MiscParams を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("0"), TestDataForKc.MakeMiscParams("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("1"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscParams("7"), TestDataForKc.MakeMiscParams("7"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "MiscFlags/ProgramType"),
                    TestDataForKc.MakeMiscParams("1"), TestDataForKc.MakeMiscParams("0"));
            }

            // HandleTableSize を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("0"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("10"));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "KernelCapabilityData/HandleTableSize"),
                    TestDataForKc.MakeHandleTableSize("256"), TestDataForKc.MakeHandleTableSize("512"));
            }

            // MiscFlags を読み込める
            {
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("True", "True"), TestDataForKc.MakeMiscFlags("True", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "True"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("False", "False"));
                SuccessTestKernelCapabilityData(TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags(null, null));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "MiscFlags/EnableDebug"),
                    TestDataForKc.MakeMiscFlags("False", "True"), TestDataForKc.MakeMiscFlags("True", null));
                FailTestKernelCapabilityData(
                    string.Format(MakeDesc.Properties.Resources.Message_OutOfCapability, "MiscFlags/ForceDebug"),
                    TestDataForKc.MakeMiscFlags("True", "False"), TestDataForKc.MakeMiscFlags("True", "True"));
            }
        }

        [TestMethod]
        public void TestDefaultIs64BitInstruction()
        {
            const string Is64BitInstructionName = "Is64BitInstruction";
            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "TRUE";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, Is64BitInstructionName), desc, null, null);
            }

            // True という文字列が受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "TRUE";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // False という文字列が受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "false";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // True/False 以外の文字列を受け取らない ( True/False の小文字、大文字の違いは気にしない )
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(Is64BitInstructionName);

                XmlElement item = new XmlElement(Is64BitInstructionName);
                defaultElem.AddChild(item);
                item.InnerText = "falure";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidBoolString, "Default/" + Is64BitInstructionName), desc, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(Is64BitInstructionName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_NotFound, "Default/" + Is64BitInstructionName), desc, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultProcessAddressSpace()
        {
            const string ProcessAddressSpaceName = "ProcessAddressSpace";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace64Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, "ProcessAddressSpace"), desc, null, null);
            }

            // AddressSpace32Bit を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace32Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // AddressSpace64BitOld を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace64BitOld";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // AddressSpace32BitNoReserved を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace32BitNoReserved";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // AddressSpace64Bit を受け付ける
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace64Bit";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 異常値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                XmlElement item = new XmlElement(ProcessAddressSpaceName);
                defaultElem.AddChild(item);
                item.InnerText = "AddressSpace";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidArgumentString, "Default/" + ProcessAddressSpaceName, item.InnerText), desc, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(ProcessAddressSpaceName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Default/" + ProcessAddressSpaceName), desc, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadPriority()
        {
            const string MainThreadPriorityName = "MainThreadPriority";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, MainThreadPriorityName), desc, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "0x18";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadPriorityName), desc, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "63";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値 - 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "-1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadPriorityName), desc, null, null);
            }

            // 最大値 + 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                XmlElement item = new XmlElement(MainThreadPriorityName);
                defaultElem.AddChild(item);
                item.InnerText = "64";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_OutOfRange, "Default/" + MainThreadPriorityName, "64"), desc, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadPriorityName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Default/" + MainThreadPriorityName), desc, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadCoreNumber()
        {
            const string MainThreadCoreNumberName = "MainThreadCoreNumber";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "24";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, MainThreadCoreNumberName), desc, null, null);
            }

            // 10 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 16 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadCoreNumberName), desc, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "255";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値 - 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "-1";

                desc.AddLine(elem.GetXml());
                desc.WriteData();

                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidDecimalNumberString, "Default/" + MainThreadCoreNumberName), desc, null, null);
            }

            // 最大値 + 1
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                XmlElement item = new XmlElement(MainThreadCoreNumberName);
                defaultElem.AddChild(item);
                item.InnerText = "256";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadCoreNumberName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_NotFoundElement, "Default/" + MainThreadCoreNumberName), desc, null, null);
            }
        }

        [TestMethod]
        public void TestDefaultMainThreadStackSize()
        {
            const string MainThreadStackSizeName = "MainThreadStackSize";

            // 重複を許可しない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x4000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(MakeDesc.Properties.Resources.Message_UnknownElement, MainThreadStackSizeName), desc, null, null);
            }

            // 10 進数の数値を受け取れない
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "4096";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidHexadecimalNumberString, "Default/" + MainThreadStackSizeName), desc, null, null);
            }

            // 16 進数の数値を受け取れる
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最大値
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0xFFFFF000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                SuccessExecute(desc, null, null);
            }

            // 最小値 - 0x1000
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x0";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_Invalid0, "Default/" + MainThreadStackSizeName), desc, null, null);
            }

            // 最大値 + 0x1000
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x100000000";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(null, desc, null, null);
            }

            // 0x1000 にアライメントされていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                XmlElement item = new XmlElement(MainThreadStackSizeName);
                defaultElem.AddChild(item);
                item.InnerText = "0x1001";

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_InvalidAlignment, "Default/" + MainThreadStackSizeName, 0x1000), desc, null, null);
            }

            // 指定されていないとエラー
            {
                XmlFile desc = new XmlFile();
                XmlElement elem = new XmlElement("Desc");
                XmlElement defaultElem = CreateDefaultParam();
                elem.AddChild(defaultElem);
                defaultElem.RemoveChild(MainThreadStackSizeName);

                desc.AddLine(elem.GetXml());
                desc.WriteData();
                FailExecute(string.Format(DescModel.Properties.Resources.Message_NotFound, "Default/" + MainThreadStackSizeName), desc, null, null);
            }
        }

        /* KernelCapability 用のユーティリティ */

        void MakeDescFileForKernelCapabilityData(ref XmlFile outputDesc, XmlElement[] descs, XmlElement[] metas)
        {
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            descElem.AddChild(defaultElem);

            XmlElement kcd = new XmlElement("KernelCapabilityDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    kcd.AddChild(elem);
                }
            }
            descElem.AddChild(kcd);

            XmlElement kc = new XmlElement("KernelCapabilityData");
            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    kc.AddChild(elem);
                }
            }
            defaultElem.AddChild(kc);

            outputDesc.AddLine(descElem.GetXml());
            outputDesc.WriteData();
        }

        void SuccessTestKernelCapabilityData(XmlElement[] descs, XmlElement[] metas)
        {
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });
            XmlFile descFile = new XmlFile();
            MakeDescFileForKernelCapabilityData(ref descFile, descs, metas);
            SuccessExecute(descFile, new string[] { baseId }, null);
        }

        void FailTestKernelCapabilityData(string errMsg, XmlElement[] descs, XmlElement[] metas)
        {
            string baseId = Path.Combine(new string[] { GetSvcHeaderDirPath(), BaseIdHeader });
            XmlFile descFile = new XmlFile();
            MakeDescFileForKernelCapabilityData(ref descFile, descs, metas);
            FailExecute(errMsg, descFile, new string[] { baseId }, null);
        }

        // ラッパー
        void SuccessTestKernelCapabilityDescriptor(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, null);
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement[] elems)
        {
            FailTestKernelCapabilityData(errMsg, elems, null);
        }
        void SuccessTestKernelCapabilityDescriptor(XmlElement elem)
        {
            SuccessTestKernelCapabilityDescriptor(new XmlElement[] { elem });
        }
        void FailTestKernelCapabilityDescriptor(string errMsg, XmlElement elem)
        {
            FailTestKernelCapabilityDescriptor(errMsg, new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement elem)
        {
            SuccessTestKernelCapabilityDataWithSameKc(new XmlElement[] { elem });
        }
        void SuccessTestKernelCapabilityDataWithSameKc(XmlElement[] elems)
        {
            SuccessTestKernelCapabilityData(elems, elems);
        }
        void SuccessTestKernelCapabilityData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestKernelCapabilityData(string errMsg, XmlElement desc, XmlElement meta)
        {
            FailTestKernelCapabilityData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta });
        }

        /* SrviceAccessControl 用のユーティリティ */

        void MakeDescFileForSrvAccessControlData(ref XmlFile outputDesc, XmlElement[] descs, XmlElement[] metas)
        {
            XmlElement descElem = new XmlElement("Desc");
            XmlElement defaultElem = CreateDefaultParam();
            descElem.AddChild(defaultElem);

            XmlElement sad = new XmlElement("SrvAccessControlDescriptor");
            if (descs != null)
            {
                foreach (var elem in descs)
                {
                    sad.AddChild(elem);
                }
            }
            descElem.AddChild(sad);

            XmlElement sac = new XmlElement("SrvAccessControlData");
            if (metas != null)
            {
                foreach (var elem in metas)
                {
                    sac.AddChild(elem);
                }
            }

            defaultElem.AddChild(sac);

            outputDesc.AddLine(descElem.GetXml());
            outputDesc.WriteData();
        }

        void SuccessTestSrvAccessControlData(XmlElement[] descs, XmlElement[] metas)
        {
            XmlFile desc = new XmlFile();
            MakeDescFileForSrvAccessControlData(ref desc, descs, metas);
            SuccessExecute(desc, null, null);
        }

        void FailTestSrvAccessControlData(string errMsg, XmlElement[] descs, XmlElement[] metas)
        {
            XmlFile desc = new XmlFile();
            MakeDescFileForSrvAccessControlData(ref desc, descs, metas);
            FailExecute(errMsg, desc, null, null);
        }

        // ラッパー
        void SuccessTestSrvAccessControlData(XmlElement desc, XmlElement meta)
        {
            SuccessTestKernelCapabilityData(new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void FailTestSrvAccessControlData(string errMsg, XmlElement desc, XmlElement meta)
        {
            FailTestSrvAccessControlData(errMsg, new XmlElement[] { desc }, new XmlElement[] { meta });
        }
        void SuccessTestSrvAccessControlDataWithSamSad(XmlElement elem)
        {
            SuccessTestSrvAccessControlData(new XmlElement[] { elem }, new XmlElement[] { elem });
        }
        void SuccessTestSrvAccessControlDataWithSamSad(XmlElement[] elems)
        {
            SuccessTestSrvAccessControlData(elems, elems);
        }
        void FailTestSrvAccessControlDataWithSameSad(string errMsg, XmlElement elem)
        {
            FailTestSrvAccessControlData(errMsg, new XmlElement[] { elem }, new XmlElement[] { elem });
        }
        void FailTestSrvAccessControlDataWithSamSad(string errMsg, XmlElement[] elems)
        {
            FailTestSrvAccessControlData(errMsg, elems, elems);
        }
    }
}
