﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
namespace MakeDataDocumentTest
{
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using System.IO;
    using System.Runtime.Serialization.Json;
    using System.Collections.Generic;
    using System.Text;
    using System.Linq;
    using System.Diagnostics;
    using MakeDataDocument;

    /// <summary>
    /// MakeDataDocument のテストを配置します。
    /// </summary>
    [TestClass]
    public class UnitTest
    {
        public TestContext TestContext { get; set; }

        public string GetTestDataRoot()
        {
            return Path.Combine(new TestPath(TestContext).GetToolTestResourceDirectory(), "DataDocumentTools");
        }

        /// <summary>
        /// ドキュメント定義ファイルの読み込みをテストします。
        /// </summary>
        [TestMethod]
        public void TestDataDocumentDefinitionFile()
        {
            var testfile = Path.Combine(GetTestDataRoot(), "SIGLO-SDK_build_packages_tests_data_document_definition.json");

            var definition = DataDocumentDefinition.MakeFromJsonFile(testfile);

            Assert.AreEqual("Test", definition.Name);
            Assert.AreEqual(@"./Source", definition.SourceDirectory);
            Assert.AreEqual(@"./Output", definition.OutputDirectory);
            CollectionAssert.AreEqual(
                new Dictionary<string, string> {
                    { "path", "^処理時間/ルールの実行/ExecuteProgram$" }
                },
                definition.Filter);
            Assert.AreEqual("name", definition.DataSampleDefinition.Key);
            Assert.AreEqual("time", definition.DataSampleDefinition.X);
            Assert.AreEqual("msec", definition.DataSampleDefinition.Y);
        }

        /// <summary>
        /// ドキュメントデータをテキスト形式の JSON データで出力するテストです。
        /// </summary>
        [TestMethod]
        public void TestGenerateVisualizableDataAsJsonText()
        {
            var generator = new DataDocumentGenerator(new DataDocumentDefinition
            {
                Name = "Test",
                Filter = new Dictionary<string, string> {
                    { "path", "^処理時間/ルールの実行/ExecuteProgram$" }
                },
                SourceDirectory = GetTestDataRoot(),
                OutputDirectory = new TestPath(this.TestContext).GetOutputRoot(),
                DataSampleDefinition = new DataSampleDefinition
                {
                    Key = "name",
                    X = "time",
                    Y = "msec"
                }
            });

            var generatedText = generator.GenerateVisualizableDataAsJsonText();
            StringAssert.Contains(generatedText, @"{""key"":""cl.exe"",""values"":[{""x"":""2014-04-09 17:15:44"",");
        }

        /// <summary>
        /// ドキュメントデータを JSON ファイルに出力するテストです。
        /// </summary>
        [TestMethod]
        public void TestGenerateVisualizableDataAsJsonFile()
        {
            var testpath = new TestPath(this.TestContext);

            var definition = new DataDocumentDefinition
            {
                Name = "Test",
                Filter = new Dictionary<string, string> {
                    { "path", "^処理時間/ルールの実行/ExecuteProgram$" }
                },
                SourceDirectory = GetTestDataRoot(),
                OutputDirectory = testpath.GetOutputRoot(),
                DataSampleDefinition = new DataSampleDefinition
                {
                    Key = "name",
                    X = "time",
                    Y = "msec"
                }
            };
            var generator = new DataDocumentGenerator(definition);

            generator.GenerateVisualizableDataAsJsonFile();

            var jsonFilePath = Path.Combine(testpath.GetOutputRoot(), "Test.json");
            var jsonText = File.ReadAllText(jsonFilePath);

            StringAssert.Contains(jsonText, @"{""key"":""cl.exe"",""values"":[{""x"":""2014-04-09 17:15:44"",");
        }

        /// <summary>
        /// ドキュメントデータをクラスオブジェクトとして取得するテストです。
        /// </summary>
        [TestMethod]
        public void TestGenerateVisualizableData()
        {
            var generator = new DataDocumentGenerator(new DataDocumentDefinition
            {
                Name = "Test",
                Filter = new Dictionary<string, string> {
                    { "name", "^[~a-z]*$" }
                },
                DataSampleDefinition = new DataSampleDefinition
                {
                    Key = "name",
                    X = "time",
                    Y = "msec"
                }
            });

            var sourceData = new List<Dictionary<string, object>>
            {
                new Dictionary<string, object> {
                    { "name", "aaaa" },
                    { "time", "3" },
                    { "msec", "132" }
                },
                new Dictionary<string, object> {
                    { "name", "bbbb" },
                    { "time", "4" },
                    { "msec", "333" }
                },
                new Dictionary<string, object> {
                    { "name", "cccc" },
                    { "msec", "132" }
                },
                new Dictionary<string, object> {
                    { "name", "1234" },
                    { "time", "4" },
                    { "msec", "333" }
                }
            };

            var generated = generator.GenerateVisualizableData(sourceData);

            Assert.AreEqual(2, generated.Count);
            Assert.AreEqual("aaaa", generated[0].Key);
            Assert.AreEqual(1, generated[0].Values.Count);
            Assert.AreEqual("3", generated[0].Values[0]["x"]);
            Assert.AreEqual("bbbb", generated[1].Key);
        }

        /// <summary>
        /// 特定ディレクトリ下のすべてのデータファイルをイテレーションするテストです。
        /// </summary>
        [TestMethod]
        public void TestEnumerateDataletsFromFiles()
        {
            foreach (var datalet in DataletUtility.EnumerateDataletsFromFiles(DataletUtility.EnumerateDataFiles(GetTestDataRoot())))
            {
            }
        }

        /// <summary>
        /// データのマージに失敗し、例外を出力することを確認するテストです。
        /// </summary>
        [TestMethod]
        [ExpectedException(typeof(InvalidDataException))]
        public void TestConflictInformation()
        {
            DataletUtility.MergeDataletAndMeta(
                new Dictionary<string, object>
                {
                    { "a", "b" }
                },
                new Dictionary<string, object>
                {
                    { "a", "c" }
                });
        }

        /// <summary>
        /// データのマージに成功することを確認するテストです。
        /// </summary>
        [TestMethod]
        public void TestMergeDataletAndMeta()
        {
            CollectionAssert.AreEquivalent(
                new Dictionary<string, object>
                {
                    { "a", "b" },
                    { "c", "d" }
                },
                DataletUtility.MergeDataletAndMeta(
                    new Dictionary<string, object>
                    {
                        { "a", "b" }
                    },
                    new Dictionary<string, object>
                    {
                        { "c", "d" }
                    }));
        }

        /// <summary>
        /// メタファイルの読み込みを確認するテストです。
        /// </summary>
        [TestMethod]
        public void TestMetaFile()
        {
            var testfile = Path.Combine(GetTestDataRoot(), "SIGLO-SDK_build_packages_tests_202.log.data.json");

            Assert.AreEqual("SIGLO-SDK_build_packages_tests_202.log.meta.json", Path.GetFileName(DataletUtility.GetMetaFileName(testfile)));

            var metaInfo = DataletUtility.GetMetaInformation(testfile);
            Assert.AreEqual("refs/heads/feature/SIGLO-1100_modifying_nw4f_sln", metaInfo["branch"]);
        }

        /// <summary>
        /// テストデータからの Datalet の読み込みテスト
        /// ここでは全てのファイルをエラー無く読めることのみ確認し、中身のテストは行いません。
        /// </summary>
        [TestMethod]
        public void TestReadDataletContainerFromTestData()
        {
            foreach (string dataletPath in DataletUtility.EnumerateDataFiles(GetTestDataRoot()))
            {
                using (var reader = new StreamReader(dataletPath))
                {
                    DataletUtility.MakeDataletContainerFromJsonText(reader.ReadToEnd());
                }
            }
        }

        /// <summary>
        /// Datalet の読み込みテスト
        /// </summary>
        [TestMethod]
        public void TestReadDataletContainer()
        {
            var result = DataletUtility.MakeDataletContainerFromJsonText(@"[{""name"":""aaaa"", ""msec"":324},{""name"":""bbbb"", ""flag"":true, ""float"":12.05}]");

            Assert.AreEqual(result.Count, 2);
            CollectionAssert.AreEqual(
                    new Dictionary<string, object> {
                        { "name", "aaaa" },
                        { "msec", 324 },
                    }, result[0]);
            CollectionAssert.AreEqual(
                    new Dictionary<string, object> {
                        { "name", "bbbb" },
                        { "flag", true },
                        { "float", 12.05m },
                    },
                    result[1]);
        }
    }
}
