﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore;
using LECore.Manipulator;
using LECore.Save_Load;
using LECore.Structures;
using LECore.Structures.Core;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.IO;
using System.Linq;
using System.Reflection;
using System.Text.RegularExpressions;
using System.Xml.Serialization;

namespace LECoreTest
{
    using FileFmt = LECore.Structures.SerializableObject.Lan;

    /// <summary>
    /// テストでよく使用する機能を提供するクラスです
    /// </summary>
    public class TestUtility
    {
        #region テスト用データ

        /// <summary>
        /// テスト用データクラスです
        /// </summary>
        public class TestData
        {
            public enum DataKind
            {
                Position,
                Size,
                Scale,
                Rotate,
                Transparency,
                Visibility,
                WindowFrameSize,
                RGBAColor,
                FloatColor,
                AnmInt,
                AnmIntVec2,
                AnmFloat,
                AnmFloatVec2,
                AnmFloatVec3,
            }

            public static Dictionary<DataKind, object> TestDataTable = new Dictionary<DataKind, object>()
            {
                { DataKind.Position,          new FVec3(99.9f, 99.9f, 99.9f)         },
                { DataKind.Size,              new FVec3(99.9f, 99.9f, 99.9f)         },
                { DataKind.Scale,             new FVec3(99.9f, 99.9f, 99.9f)         },
                { DataKind.Rotate,            new FVec3(99.9f, 99.9f, 99.9f)         },
                { DataKind.Transparency,      (Byte)99                               },
                { DataKind.Visibility,        true                                   },
                { DataKind.WindowFrameSize,   new FVec4(99.9f, 99.9f, 99.9f, 99.9f)  },
                { DataKind.RGBAColor,         new RGBAColor(99, 99, 99, 99)          },
                { DataKind.FloatColor,         new FloatColor(99f/255f, 99f/255f, 99f/255f, 99f/255f) },
                { DataKind.AnmInt,            99                                     },
                { DataKind.AnmIntVec2,        new IVec2(99, 99)                      },
                { DataKind.AnmFloat,          99.9f                                  },
                { DataKind.AnmFloatVec2,      new FVec2(99.9f, 99.9f)                },
                { DataKind.AnmFloatVec3,      new FVec3(99.9f, 99.9f, 99.9f)         },
            };

            public static object GetTestData(DataKind kind)
            {
                return TestDataTable[kind];
            }
        }

        #endregion

        #region フィールド

        static SceneManipulator _sceneMnp = new SceneManipulator();
        static SubSceneManipulator _subSceneMnp = new SubSceneManipulator();
        static PaneManipulator _paneMnp = new PaneManipulator();
        static PictureManipulator _pictureMnp = new PictureManipulator();
        static WindowManipulator _windowMnp = new WindowManipulator();
        static TextBoxManipulator _textBoxMnp = new TextBoxManipulator();
        static RevMaterialManipulator _matMnp = new RevMaterialManipulator();
        static AnimFrameSectionSetManipulator _animFrameSectionSetMnp = new AnimFrameSectionSetManipulator();
        static UserDataHolderManipulator _userDataHolderMnp = new UserDataHolderManipulator();
        static AnmCurveManipulator _animCurveMnp = new AnmCurveManipulator();
        static TextureMgrManipulator _textureMgrMnp = new TextureMgrManipulator();
        static PartsLayoutManipulator _partsLayoutMnp = new PartsLayoutManipulator();
        static AnmKeyFrameManipulator _anmKeyFrameMnp = new AnmKeyFrameManipulator();
        static GroupMgrManipulator _groupMgrMnp = new GroupMgrManipulator();

        #endregion

        #region プロパティ

        /// <summary>
        /// シーン
        /// </summary>
        public static IScene Scene
        {
            get
            {
                return LayoutEditorCore.Scene;
            }
        }

        /// <summary>
        /// シーンマニピュレータ
        /// </summary>
        static SceneManipulator SceneMnp
        {
            get
            {
                _sceneMnp.BindTarget(Scene);
                return _sceneMnp;
            }
        }

        /// <summary>
        /// サブシーンマニピュレータ
        /// </summary>
        static SubSceneManipulator SubSceneMnp
        {
            get
            {
                return _subSceneMnp;
            }
        }

        /// <summary>
        /// ペインマニピュレータ
        /// </summary>
        static PaneManipulator PaneMnp
        {
            get
            {
                return _paneMnp;
            }
        }

        /// <summary>
        /// ピクチャマニピュレータ
        /// </summary>
        static PictureManipulator PictureMnp
        {
            get
            {
                return _pictureMnp;
            }
        }

        /// <summary>
        /// ウインドウマニピュレータ
        /// </summary>
        static WindowManipulator WindowMnp
        {
            get
            {
                return _windowMnp;
            }
        }

        /// <summary>
        /// ウインドウマニピュレータ
        /// </summary>
        static TextBoxManipulator TextBoxMnp
        {
            get
            {
                return _textBoxMnp;
            }
        }

        /// <summary>
        /// マテリアルマニピュレータ
        /// </summary>
        static RevMaterialManipulator MatMnp
        {
            get
            {
                return _matMnp;
            }
        }

        /// <summary>
        /// 区間タグマニピュレータ
        /// </summary>
        static AnimFrameSectionSetManipulator AnimFrameSectionSetMnp
        {
            get
            {
                return _animFrameSectionSetMnp;
            }
        }

        /// <summary>
        /// ユーザーデータホルダーマニピュレータ
        /// </summary>
        static UserDataHolderManipulator UserDataHolderMnp
        {
            get
            {
                return _userDataHolderMnp;
            }
        }

        /// <summary>
        /// アニメーションカーブマニピュレータ
        /// </summary>
        static AnmCurveManipulator AnimCurveMnp
        {
            get
            {
                return _animCurveMnp;
            }
        }

        /// <summary>
        /// テクスチャマネージャマニピュレータ
        /// </summary>
        static TextureMgrManipulator TextureMgrMnp
        {
            get
            {
                return _textureMgrMnp;
            }
        }

        /// <summary>
        /// 部品レイアウトマニピュレータ
        /// </summary>
        static PartsLayoutManipulator PartsLayoutMnp
        {
            get
            {
                return _partsLayoutMnp;
            }
        }

        /// <summary>
        /// 部品レイアウトマニピュレータ
        /// </summary>
        static AnmKeyFrameManipulator AnmKeyFrameMnp
        {
            get
            {
                return _anmKeyFrameMnp;
            }
        }

        /// <summary>
        /// グループマネージャマニピュレータ
        /// </summary>
        static GroupMgrManipulator GroupMgrMnp
        {
            get
            {
                return _groupMgrMnp;
            }
        }

        #endregion

        #region 初期化

        /// <summary>
        /// テスト実行前に必要な処理を行ないます
        /// </summary>
        public static void InitializeTest(MethodBase method)
        {
            TestUtility.WriteLog("Start", method.ToString());

            // 部品ルートを設定
            TestUtility.SetPartsRoot();

            // テンポラリルートを初期化
            TestUtility.InitializeTempRoot();

            // テンポラリを削除
            TestUtility.ClearTempDirectory();

            // TileSettingParamsを初期化（暫定）
            TileSettingParams.TileOptimize = "performance";
            TileSettingParams.TileSizeThreshold = 75;
        }

        /// <summary>
        /// テスト実行後に必要な処理を行ないます
        /// </summary>
        public static void FinalizeTest(MethodBase method)
        {
            // テンポラリを削除
            TestUtility.ClearTempDirectory();

            TestUtility.WriteLog("End", method.ToString());
        }

        /// <summary>
        /// 部品ルートを設定します
        /// </summary>
        public static void SetPartsRoot()
        {
            string path = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts");
            SetPartsRoot(path);
        }
        public static void SetPartsRoot(string path)
        {
            // 部品ルートの設定
            SceneMnp.RefreshPartsSubScenes(path);
        }

        /// <summary>
        /// テンポラリルートを初期化します
        /// </summary>
        static public void InitializeTempRoot()
        {
            LECore.Util.TemporaryFileUtil.Initialize();
        }

        #endregion

        #region レイアウト操作

        /// <summary>
        /// レイアウトファイルを読み込みます
        /// </summary>
        public static ISubScene LoadLayout(string fileName)
        {
            string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, fileName);
            return LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;
        }

        /// <summary>
        /// レイアウトファイルを読み込みます
        /// フルパスをダイレクトに指定します
        /// </summary>
        public static ISubScene LoadLayoutFullPath(string filePath)
        {
            return LayoutEditorCore.LoadLayoutFile(filePath, LayoutEditorCore.LoadOption.TryToOpenRlan).ISubScene;
        }

        /// <summary>
        /// レイアウトファイルを保存します
        /// テンポラリフォルダに保存されます
        /// </summary>
        public static string SaveLayout(ISubScene subScene, string fileName)
        {
            string outPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output");
            string filePath = Path.Combine(outPath, fileName);

            return SaveLayoutFullPath(subScene, filePath);
        }

        /// <summary>
        /// レイアウトファイルを保存します
        /// フルパスをダイレクトに指定します
        /// </summary>
        public static string SaveLayoutFullPath(ISubScene subScene, string filePath)
        {
            ExportOption exportOption = new ExportOption()
            {
                UseBaseValue = true,
                Frame = GlobalTime.Inst.Time,
            };
            LayoutEditorCore.BeforeExportEventArgs prepareOption = new LayoutEditorCore.BeforeExportEventArgs(filePath, exportOption);

            LayoutEditorCore.PrepareExport(prepareOption);
            LayoutEditorCore.ExportToFileAll(subScene, filePath, exportOption);

            return filePath;
        }

        /// <summary>
        /// レイアウトを作成します
        /// </summary>
        public static ISubScene CreateLayout()
        {
            return (Scene as Scene).AddSubScene();
        }

        /// <summary>
        /// レイアウトを閉じます
        /// </summary>
        public static void CloseLayout(ISubScene subScene)
        {
            SceneMnp.RemoveSubScene(subScene);
        }

        /// <summary>
        /// 部品レイアウトを読み込みます
        /// </summary>
        public static ISubScene LoadPartsLayout(string fileName)
        {
            LayoutEditorCore.Scene.LoadAndCachePartsSubSceneIfNeeded(fileName);
            return LayoutEditorCore.Scene.FindPartsSubSceneByFileName(fileName).SubScene;
        }

        #endregion

        #region ペイン追加

        /// <summary>
        /// レイアウトにヌルペインを追加します
        /// </summary>
        public static IPane AddNullPane(ISubScene subScene, string paneName)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddNullPane(paneName);
        }

        /// <summary>
        /// レイアウトにピクチャペインを追加します
        /// </summary>
        public static IPane AddPicturePane(ISubScene subScene, string paneName, string textureName)
        {
            SubSceneMnp.BindTarget(subScene);
            if (textureName != null)
            {
                return SubSceneMnp.AddPicturePane(paneName, textureName, false);
            }
            else
            {
                return SubSceneMnp.AddPicturePaneWithoutTexture(paneName);
            }
        }

        /// <summary>
        /// レイアウトにテキストボックスペインを追加します
        /// </summary>
        public static IPane AddTextBoxPane(ISubScene subScene, string paneName, ILEFont font, string text)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddTextBoxPane(paneName, font, text);
        }

        /// <summary>
        /// レイアウトにウインドウペインを追加します
        /// </summary>
        public static IPane AddWindowPane(ISubScene subScene, string paneName)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddWindowPane(paneName);
        }

        /// <summary>
        /// レイアウトに境界ペインを追加します
        /// </summary>
        public static IPane AddBoudingPane(ISubScene subScene, string paneName)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddBoudingPane(paneName);
        }

        /// <summary>
        /// レイアウトにキャプチャペインを追加します
        /// </summary>
        public static IPane AddCapturePane(ISubScene subScene, string paneName)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddCapturePane(paneName);
        }

        /// <summary>
        /// レイアウトに部品ペインを追加します
        /// </summary>
        public static IPane AddPartsPane(ISubScene subScene, string paneName, string partsLayoutName, ISubScene partsSubScene)
        {
            SubSceneMnp.BindTarget(subScene);
            return SubSceneMnp.AddPartsPane(paneName, partsLayoutName, partsSubScene);
        }

        /// <summary>
        /// レイアウトからペインを削除します
        /// </summary>
        public static void RemovePane(ISubScene subScene, IPane pane)
        {
            SubSceneMnp.BindTarget(subScene);
            SubSceneMnp.SelectPanesByPaneRef(pane);
            SubSceneMnp.DeleteSelectedPanes();
        }

        /// <summary>
        /// 派生部品を作成します
        /// </summary>
        public static ISubScene CreateDerivativeParts(string fileName)
        {
            string filePath = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts") + "\\" + fileName;

            LayoutEditorCore.LoadOption option = LayoutEditorCore.LoadOption.TryToOpenRlan | LayoutEditorCore.LoadOption.LoadAsDerivativeBase;
            return LayoutEditorCore.LoadLayoutFileAsNewFile(filePath, option).ISubScene;
        }

        #endregion ペイン追加

        #region ペイン操作

        /// <summary>
        /// ペインにテストパラメータを自動的に設定します
        /// </summary>
        public static void ModifyPaneTestParameters(IPane pane)
        {
            ModifyPosition(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Position));

            // キャプチャペインの入力が制限されているパラメータには設定しない。
            if (pane.ICapture == null)
            {
                ModifySize(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Size));
                ModifyScale(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Scale));
                ModifyRotate(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Rotate));
                ModifyTransparency(pane, (byte)TestData.GetTestData(TestData.DataKind.Transparency));
                ModifyVisibility(pane, (bool)TestData.GetTestData(TestData.DataKind.Visibility));
            }

            ModifyExtUserData(pane, "user1", "user1", UserDataKind.AnmInt, TestData.GetTestData(TestData.DataKind.AnmInt));
            ModifyExtUserData(pane, "user2", "user2", UserDataKind.AnmIntVec2, TestData.GetTestData(TestData.DataKind.AnmIntVec2));
            ModifyExtUserData(pane, "rgba", "rgba", UserDataKind.AnmByteRGBA4, TestData.GetTestData(TestData.DataKind.RGBAColor));
            ModifyExtUserData(pane, "float", "float", UserDataKind.AnmFloat, TestData.GetTestData(TestData.DataKind.AnmFloat));
            ModifyExtUserData(pane, "float2", "float2", UserDataKind.AnmFloatVec2, TestData.GetTestData(TestData.DataKind.AnmFloatVec2));
            ModifyExtUserData(pane, "float3", "float3", UserDataKind.AnmFloatVec3, TestData.GetTestData(TestData.DataKind.AnmFloatVec3));

            if (pane.IPicture != null)
            {
                ModifyPictureVertexColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                ModifyMaterialColor(pane, (FloatColor)TestData.GetTestData(TestData.DataKind.FloatColor));
            }
            else if (pane.ITextBox != null)
            {
                ModifyTextBoxTopBottomColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                ModifyTextBoxShadowColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                ModifyMaterialColor(pane, (FloatColor)TestData.GetTestData(TestData.DataKind.FloatColor));
            }
            else if (pane.ILEWindow != null)
            {
                ModifyWindowFrameSize(pane, (FVec4)TestData.GetTestData(TestData.DataKind.WindowFrameSize));
                ModifyWindowVertexColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                ModifyMaterialColor(pane, (FloatColor)TestData.GetTestData(TestData.DataKind.FloatColor));
            }
        }

        /// <summary>
        /// ペインの位置を変更します
        /// </summary>
        public static void ModifyPosition(IPane pane, FVec3 trans)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Trans = trans;
        }

        /// <summary>
        /// ペインのサイズを変更します
        /// </summary>
        public static void ModifySize(IPane pane, FVec3 size)
        {
            // 部品レイアウトはサイズを変更できない
            if (pane.IPartsLayout != null) return;

            PaneMnp.BindTarget(pane);
            PaneMnp.Size = size;
        }

        /// <summary>
        /// ペインのスケールを変更します
        /// </summary>
        public static void ModifyScale(IPane pane, FVec3 scale)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Scale = scale;
        }

        /// <summary>
        /// ペインの回転を変更します
        /// </summary>
        public static void ModifyRotate(IPane pane, FVec3 rot)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.RotAng = rot;
        }

        /// <summary>
        /// ペインの透明度を変更します
        /// </summary>
        public static void ModifyTransparency(IPane pane, byte trans)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Transparency = trans;
        }

        /// <summary>
        /// ペインの表示・非表示を変更します
        /// </summary>
        public static void ModifyVisibility(IPane pane, bool visible)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Visible = visible;
        }

        /// <summary>
        /// ウインドウペインのフレームサイズを変更します
        /// </summary>
        public static void ModifyWindowFrameSize(IPane pane, FVec4 frameSize)
        {
            ModifyWindowFrameSize(pane, frameSize.X, frameSize.Y, frameSize.Z, frameSize.W);
        }
        public static void ModifyWindowFrameSize(IPane pane, float top, float bottom, float left, float right)
        {
            ILEWindow window = pane.ILEWindow;

            Debug.Assert(window != null);

            WindowMnp.BindTarget(window);
            WindowMnp.SetFrameSize(top, bottom, left, right);
        }

        /// <summary>
        /// ピクチャペインの頂点カラーを変更します
        /// </summary>
        public static void ModifyPictureVertexColor(IPane pane, RGBAColor color)
        {
            IPicture picture = pane.IPicture;

            Debug.Assert(picture != null);

            PictureMnp.BindTarget(picture);
            PictureMnp.VtxColor_LT = color;
            PictureMnp.VtxColor_LB = color;
            PictureMnp.VtxColor_RT = color;
            PictureMnp.VtxColor_RB = color;
        }

        /// <summary>
        /// テキストボックスペインの上下端カラーを変更します
        /// </summary>
        public static void ModifyTextBoxTopBottomColor(IPane pane, RGBAColor color)
        {
            ITextBox textBox = pane.ITextBox;

            Debug.Assert(textBox != null);

            TextBoxMnp.BindTarget(textBox);
            TextBoxMnp.TopColor = color;
            TextBoxMnp.BottomColor = color;
        }

        /// <summary>
        /// テキストボックスペインの影カラーを変更します
        /// </summary>
        public static void ModifyTextBoxShadowColor(IPane pane, RGBAColor color)
        {
            ITextBox textBox = pane.ITextBox;

            Debug.Assert(textBox != null);

            TextBoxMnp.BindTarget(textBox);
            TextBoxMnp.ShadowBlackColor = new RGBAColor(color.R, color.G, color.B, textBox.ShadowBlackColor.A); // 影黒カラーはアルファなし
            TextBoxMnp.ShadowWhiteColor = color;
        }

        /// <summary>
        /// テキストボックスペインのフォントを差し替えます
        /// </summary>
        public static void ModifyTextBoxFont(IPane pane, string fontPath)
        {
            ITextBox textBox = pane.ITextBox;

            Debug.Assert(textBox != null);

            _textBoxMnp.BindTarget(textBox);
            _textBoxMnp.SetLEFontByFileName(System.IO.Path.GetFileName(fontPath), fontPath);
        }

        /// <summary>
        /// ウインドウペインの頂点カラーを変更します
        /// </summary>
        public static void ModifyWindowVertexColor(IPane pane, RGBAColor color)
        {
            ILEWindow window = pane.ILEWindow;

            Debug.Assert(window != null);

            WindowMnp.BindTarget(window);
            WindowMnp.VtxCol_LT = color;
            WindowMnp.VtxCol_LB = color;
            WindowMnp.VtxCol_RT = color;
            WindowMnp.VtxCol_RB = color;
        }

        public static void ModifyMaterialColor(IPane pane, FloatColor color)
        {
            IRevHWMaterial[] materials = PaneHelper.GetRevHWMatFromPane(pane);
            Debug.Assert(materials != null);

            // [C]と[LT]以外は共有設定のためカラーの指定不可
            foreach (IRevHWMaterial material in materials)
            {
                if ((material as RevHWMaterial).Description == "" ||
                    (material as RevHWMaterial).Description == "C" ||
                    (material as RevHWMaterial).Description == "LT" )
                {
                    // ByteConvert はアニメーションしないなど特殊なので、ここでは変更されないようにする
                    var color2 = new FloatColor(color);
                    color2.ByteConvert = material.IMaterial.BlackColor.ByteConvert;

                    MatMnp.BindTarget(material);
                    if (pane.ITextBox != null)
                    {
                        // テキストボックスペインの黒カラーはアルファなし
                        MatMnp.BlackColor = new FloatColor(color2.R, color2.G, color2.B, pane.ITextBox.IMaterial.BlackColor.A, color2.ByteConvert);
                        MatMnp.WhiteColor = new FloatColor(color2);
                    }
                    else
                    {
                        MatMnp.BlackColor = new FloatColor(color2);
                        MatMnp.WhiteColor = new FloatColor(color2);
                    }
                }
            }
        }

        /*
        /// <summary>
        /// インダイレクトスケールを変更します
        /// </summary>
        public static void ModifyIndirectScale(IPane pane, FVec2 uv)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// インダイレクト回転を変更します
        /// </summary>
        public static void ModifyIndirectRotate(IPane pane, float rot)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// アルファテストを変更します
        /// </summary>
        public static void ModifyAlphaTest(IPane pane, float alpha)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// LowLevelMaterialを変更します
        /// </summary>
        public static void ModifyLowLevelMaterial(IPane pane, RGBAColor color)
        {
            // TODO:
            //
            //
        }
        */

        /// <summary>
        /// 拡張ユーザー情報を追加します
        /// </summary>
        public static void AddExtUserData(IPane pane, string name, UserDataKind kind, object val)
        {
            UserDataHolderMnp.BindTarget(pane.IUserDataHolder);
            UserDataHolderMnp.AddUserDataElement(name, kind, val);
        }

        /// <summary>
        /// 拡張ユーザー情報を変更します
        /// </summary>
        public static void ModifyExtUserData(IPane pane, string name, string newName, UserDataKind kind, object val)
        {
            Assert.IsTrue(UserDataElementHelper.CheckParamatersValid(newName, kind, val));
            UserDataHolderMnp.BindTarget(pane.IUserDataHolder);
            UserDataHolderMnp.SetUserDataPamaters(name, newName, kind, val);
        }

        /// <summary>
        /// 拡張ユーザー情報を削除します
        /// </summary>
        public static void RemoveExtUserData(IPane pane, string name)
        {
            UserDataHolderMnp.BindTarget(pane.IUserDataHolder);
            UserDataHolderMnp.RemoveUserDataElement(name);
        }

        /// <summary>
        /// ペインをロックします
        /// </summary>
        public static void LockPane(IPane pane)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Locked = true;
        }

        /// <summary>
        /// ペインのロックを解除します
        /// </summary>
        public static void UnLockPane(IPane pane)
        {
            PaneMnp.BindTarget(pane);
            PaneMnp.Locked = false;
        }

        /// <summary>
        /// 部品レイアウトをリロードします
        /// </summary>
        public static void ReloadPartsLayout(IPane pane)
        {
            PartsLayoutManipulator.ReloadPartsPane(pane);
        }

        #endregion

        #region 区間タグ操作

        /// <summary>
        /// 区間タグを追加します
        /// </summary>
        public static IAnimFrameSection AddAnimFrameSection(ISubScene subScene, string name, int startFrame, int endFrame, string comment = "")
        {
            AnimFrameSectionSetMnp.BindTarget(subScene.IAnimFrameSectionSet);
            IAnimFrameSection section = AnimFrameSectionSetMnp.AddNewFrameSection(name, comment, startFrame, endFrame, subScene);

            return section;
        }

        /// <summary>
        /// 区間タグを編集します
        /// </summary>
        public static void EditAnimFrameSection(ISubScene subScene, IAnimFrameSection section, string name, int startFrame, int endFrame)
        {
            AnimFrameSectionSetMnp.BindTarget(subScene.IAnimFrameSectionSet);
            AnimFrameSectionSetMnp.SetTagBasic(section, name, "", startFrame, endFrame);
        }

        /// <summary>
        /// 区間タグを削除します
        /// </summary>
        public static void RemoveAnimFrameSection(ISubScene subScene, IAnimFrameSection section)
        {
            AnimFrameSectionSetMnp.BindTarget(subScene.IAnimFrameSectionSet);
            AnimFrameSectionSetMnp.Remove(section, subScene);
        }

        /// <summary>
        /// 区間タグを切り替えます
        /// </summary>
        public static IAnimFrameSection ModifyAnimFrameSectionFromName(ISubScene subScene, string tagName)
        {
            IAnimFrameSection section = subScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == tagName);

            ModifyAnimFrameSection(subScene, section);

            return section;
        }
        public static void ModifyAnimFrameSection(ISubScene subScene, IAnimFrameSection section)
        {
            AnimFrameSectionSetMnp.BindTarget(subScene.IAnimFrameSectionSet);
            AnimFrameSectionSetMnp.SetTargetAnimFrameSection(section);

            return;
        }

        /// <summary>
        /// 分割/結合モードを変更します
        /// </summary>
        public static bool ChangeAnimEditMode(ISubScene subScene)
        {
            SubSceneMnp.BindTarget(subScene);
            SubSceneMnp.AnmConvert();

            return subScene.IsAnimEditSeparateMode();
        }

        /// <summary>
        /// 区間タグをエクスポートします
        /// </summary>
        public static void ExportAnimSection(ISubScene subScene, string fileName)
        {
            string filePath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output") + "\\" + fileName;

            RlanConverter rlanConverter = new RlanConverter(LECore.LayoutEditorCore.MsgReporter);
            FileFmt.AnimTag[] animTagSet = rlanConverter.SetTagDataToFileFmt(subScene.IAnimFrameSectionSet.IAnimFrameSectionSet);

            StreamWriter writer = null;
            XmlSerializer xmlSerializer = null;

            xmlSerializer = XmlSerializerCache.GetXmlSerializer(typeof(FileFmt.AnimTag[]));
            writer = new StreamWriter(filePath, false);
            xmlSerializer.Serialize(writer, animTagSet);

            writer.Close();
        }

        /// <summary>
        /// 区間タグをインポートします
        /// </summary>
        public static void ImportAnimSection(ISubScene subScene, string fileName)
        {
            string filePath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output") + "\\" + fileName;

            RlanConverter rlanConverter = new RlanConverter(LECore.LayoutEditorCore.MsgReporter);
            XmlSerializer serializer = XmlSerializerCache.GetXmlSerializer(typeof(FileFmt.AnimTag[]));

            using (FileStream fs = new FileStream(filePath, FileMode.Open, FileAccess.Read))
            {
                FileFmt.AnimTag[] animTagSet = XmlSerializerCache.Deserialize(serializer, fs) as FileFmt.AnimTag[];
                List<string> errorTagMessage = new List<string>();

                rlanConverter.AddFrameSection(
                    subScene, animTagSet, errorTagMessage,
                    AnimFrameSectionSetMnp.AddNewFrameSection,
                    (IAnimFrameSection frameSection, ILEGroup[] groups, bool descendingBind) => {
                        AnimFrameSectionSetMnp.SetTagTargetGroup(frameSection, groups, descendingBind);
                    },
                    SubSceneMnp.AddAnimSection);
            }
        }

        /// <summary>
        /// 区間をフィッティングします
        /// </summary>
        public static void FittingSection(ISubScene subScene)
        {
            SubSceneMnp.BindTarget(subScene);
            SubSceneMnp.FittingSection();
        }

        #endregion

        #region タイムスライダー

        /// <summary>
        /// タイムスライダーの時間を変更します
        /// </summary>
        public static void SetTime(int time)
        {
            GlobalTime.Inst.SetTimeForcibly(time);
        }

        #endregion

        #region コピー＆ペースト

        /// <summary>
        /// ペインを複製します
        /// </summary>
        public static IPane DuplicatePane(ISubScene subScene, IPane srcPane)
        {
            SubSceneMnp.BindTarget(subScene);
            SubSceneMnp.SelectPanesByPaneRef(srcPane);
            SubSceneMnp.CopySelectedPanes();
            SubSceneMnp.PastePanes(true, null);

            string paneName;
            string no = srcPane.PaneName.Substring(srcPane.PaneName.Length - 2, 2);
            if (Regex.IsMatch(no, "[0-9][0-9]"))
            {
                string baseName = srcPane.PaneName.Substring(0, srcPane.PaneName.Length - 2);

                int newNo = int.Parse(no) + 1;
                paneName = baseName + String.Format("{0:D2}", newNo);
            }
            else
            {
                paneName = srcPane.PaneName + "_00";
            }

            return subScene.FindPaneByName(paneName); ;
        }

        /// <summary>
        /// ペインパラメータを貼り付けます
        /// </summary>
        public static void PastePaneParameter(IPane srcPane, IPane dstPane)
        {
            PaneParamaterCopyOption option = new PaneParamaterCopyOption();
            option.EnableOption(ParamaterKind.All);

            PastePaneParameter(srcPane, dstPane, option);
        }
        public static void PastePaneParameter(IPane srcPane, IPane dstPane, PaneParamaterCopyOption option)
        {
            PaneParamaterPaster.PasteParamaters(option, dstPane, srcPane);
        }

        #endregion

        #region グループ操作

        /// <summary>
        /// ペインからグループを作成します
        /// </summary>
        public static ILEGroup CreateGroup(IPane pane, string name)
        {
            GroupMgrMnp.BindTarget(pane.OwnerSubScene.ILEGroupMgr);

            GroupMgrMnp.BeginEditGroup();
            ILEGroup group = GroupMgrMnp.MakeNewGroupFromPaneSet(new IPane[1] { pane }, name);
            GroupMgrMnp.EndEditGroup();

            return group;
        }

        /// <summary>
        /// グループを削除します
        /// </summary>
        public static void RemoveGroup(ISubScene subScene, ILEGroup group)
        {
            GroupMgrMnp.BindTarget(subScene.ILEGroupMgr);

            GroupMgrMnp.BeginEditGroup();
            GroupMgrMnp.RemoveGroup(group);
            GroupMgrMnp.EndEditGroup();
        }

        /// <summary>
        /// グループにペインを追加します
        /// </summary>
        public static void AddPaneToGroup(ISubScene subScene, ILEGroup group, IPane pane)
        {
            GroupMgrMnp.BindTarget(subScene.ILEGroupMgr);

            GroupMgrMnp.BeginEditGroup();
            GroupMgrMnp.AddMember(group, pane);
            GroupMgrMnp.EndEditGroup();
        }

        /// <summary>
        /// グループからペインを削除します
        /// </summary>
        public static void RemovePaneFromGroup(ISubScene subScene, ILEGroup group, IPane pane)
        {
            GroupMgrMnp.BindTarget(subScene.ILEGroupMgr);

            GroupMgrMnp.BeginEditGroup();
            GroupMgrMnp.RemoveMember(group, pane);
            GroupMgrMnp.EndEditGroup();
        }

        #endregion

        #region カーブ操作

        /// <summary>
        /// アトリビュートにアニメーションキーを追加します
        /// </summary>
        public static IAnmKeyFrame AddAnimKey(IAnmAttribute attr, int time, object value)
        {
            AnimCurveMnp.BindTarget(attr.ICurrentAnimationCurve);
            AnimCurveMnp.MakeKeyFrame(time, value);

            return attr.ICurrentAnimationCurve.FindKeyExistAt(time);
        }

        /// <summary>
        /// アニメーションキーを編集します
        /// </summary>
        public static void ModifyAnimKey(IAnmKeyFrame key, int time, float value)
        {
            AnmKeyFrameMnp.BindTarget(key);
            AnmKeyFrameMnp.Modify(time, value);
        }

        /// <summary>
        /// 拡張ユーザーデータのアトリビュートにアニメーションキーを追加します
        /// </summary>
        public static IAnmKeyFrame[] AddExtUserDataAnimKey(IUserDataElement elem, int time, object value)
        {
            var attr = elem.IAnmAttribute;
            List<IAnmKeyFrame> keyList = new List<IAnmKeyFrame>();

            if ((attr as AnmAttribute).CanGetOrSetValue() &&
                attr.NumSubAttribute == 0)
            {
                keyList.Add(AddAnimKey(attr, time, value));
            }
            else
            {
                var array = UserDataElementHelper.ConvertAnmValueToArray(elem.UserDataKind, value) as Array;
                for (int i = 0; i < attr.NumSubAttribute; i++)
                {
                    keyList.Add(AddAnimKey(attr.FindSubAttributeByIdx(i), time, array.GetValue(i)));
                }
            }

            return keyList.ToArray();
        }

        /// <summary>
        /// アトリビュートからアニメーションキーを削除します
        /// </summary>
        public static void RemoveAnimKey(IAnmKeyFrame key)
        {
            AnimCurveMnp.BindTarget(key.OwnerIAnmCurve);
            AnimCurveMnp.RemoveKeyFrame(key, true);
        }

        /// <summary>
        /// カーブをスナップします
        /// </summary>
        public static void SnapCurve(ISubScene subScene, IAnmAttribute attr)
        {
            SubSceneMnp.BindTarget(subScene);
            SubSceneMnp.SnapBeforeKey(new IAnmCurve[1] { attr.ICurrentAnimationCurve });
            SubSceneMnp.SnapAfterKey(new IAnmCurve[1] { attr.ICurrentAnimationCurve });
        }

        #endregion

        #region フォント

        /// <summary>
        /// フォントを読み込みます
        /// </summary>
        public static ILEFont LoadFontFullPath(ISubScene subScene, string fontPath)
        {
            return subScene.ILEFontManager.CreateILEFont(Path.GetFileName(fontPath), fontPath, null);
        }
        public static ILEFont LoadFont(ISubScene subScene, string fileName)
        {
            string fontPath = LECore.AppConstants.NwUnitTestDataRoot + "\\Font\\" + fileName;

            return LoadFontFullPath(subScene, fontPath);
        }

        #endregion

        #region テクスチャ

        /// <summary>
        /// テクスチャを読み込みます
        /// </summary>
        public static void RegistTexture(ISubScene subScene, string fileName)
        {
            string texturePath = LECore.AppConstants.NwUnitTestDataRoot + "\\Texture\\" + fileName;

            TextureMgrMnp.BindTarget(subScene.ITextureMgr);
            TextureMgrMnp.RegisterITextureImageSetFromFile(new string[1] { texturePath });
        }

        #endregion

        #region 比較

        /// <summary>
        /// ペインが一致していることを確認する。
        /// </summary>
        public static void AreEqualsPaneParameters(IPane paneA, IPane paneB)
        {
            ComparePaneParametersImpl_(paneA, paneB, (valA, valB) => Assert.AreEqual(valA, valB));
        }

        /// <summary>
        /// ペインが異なっていることを確認する。
        /// </summary>
        public static void AreNotEqualsPaneParameters(IPane paneA, IPane paneB)
        {
            ComparePaneParametersImpl_(paneA, paneB, (valA, valB) => Assert.AreNotEqual(valA, valB));
        }

        /// <summary>
        /// 比較します。
        /// </summary>
        private static void CompareEqualsPaneParameters_(IAnmAttribute attrA, IAnmAttribute attrB, Action<object, object> assertionAction)
        {
            if (attrA.Name == "CharTransformScale" ||
                attrA.Name == "CharTransformTrans" ||
                attrA.Name == "CharTransformRotate")
            {
                // 暫定：文字毎変換で計算されるパラメータは、原因は不明だが値が更新されないことがあるため、比較に失敗するので除外します
                // 　　　実際のLayoutEditorではこれらのアトリビュートの値は使われていないので動作に問題はありません
                // 　　　SIGLO-79784で文字毎変換のアトリビュートの扱いについて検討します
                return;
            }

            if ((attrA as AnmAttribute).CanGetOrSetValue() &&
                (attrB as AnmAttribute).CanGetOrSetValue() &&
                attrA.NumSubAttribute == 0 &&
                attrB.NumSubAttribute == 0 &&
                attrA.CheckActiveRecursive() &&
                attrB.CheckActiveRecursive())
            {
                object valA = null;
                object valB = null;

                if (attrA is AnmAttribute && attrB is AnmAttribute)
                {
                    (attrA as AnmAttribute).GetValue(out valA);
                    (attrB as AnmAttribute).GetValue(out valB);

                    WriteLog(GetAttributeTreeString(attrA));
                    assertionAction(valA, valB);
                }
            }

            // サブアトリビュートを処理する
            Debug.Assert(attrA.NumSubAttribute == attrB.NumSubAttribute);
            for (int i = 0; i < attrA.NumSubAttribute; i++)
            {
                CompareEqualsPaneParameters_(attrA.FindSubAttributeByIdx(i), attrB.FindSubAttributeByIdx(i), assertionAction);
            }
        }

        /// <summary>
        /// 比較します。
        /// </summary>
        private static void ComparePaneParametersImpl_(IPane paneA, IPane paneB, Action<object, object> assertionAction)
        {
            foreach (ParamaterKind kind in ParamaterKindHelper.AnimationKindSet)
            {
                var attrSetA = paneA.GetAnimationTargetAttributeSet(kind);
                var attrSetB = paneB.GetAnimationTargetAttributeSet(kind);

                Debug.Assert(attrSetA.Length == attrSetB.Length);
                for (int i = 0; i < attrSetA.Length; i++)
                {
                    CompareEqualsPaneParameters_(attrSetA[i], attrSetB[i], assertionAction);
                }
            }
        }

        /*
         * 比較はメンテナンスコストを考えて、アトリビュートの全比較形式で実装します。
         * パラメータ単位の比較は必要に応じてコメントアウトを解除して使用してください。
         */
        /*
        /// <summary>
        /// ペインを比較します
        /// </summary>
        public static void AreEqualsPaneParameters(IPane paneA, IPane paneB)
        {
            IsEqualPosition(paneA, paneB);
            IsEqualSize(paneA, paneB);
            IsEqualScale(paneA, paneB);
            IsEqualRotate(paneA, paneB);
            IsEqualTransparency(paneA, paneB);
            IsEqualVisibility(paneA, paneB);

            if (paneA.IPicture != null)
            {
                Debug.Assert(paneB.IPicture != null);

                IsEqualPictureVertexColor(paneA, paneB);
                IsEqualMaterialColor(paneA, paneB);
            }
            else if (paneA.ITextBox != null)
            {
                Debug.Assert(paneB.ITextBox != null);

                IsEqualTextBoxTopBottomColor(paneA, paneB);
                IsEqualTextBoxShadowColor(paneA, paneB);
                IsEqualMaterialColor(paneA, paneB);
            }
            else if (paneA.ILEWindow != null)
            {
                Debug.Assert(paneB.ILEWindow != null);

                IsEqualWindowFrameSize(paneA, paneB);
                IsEqualWindowVertexColor(paneA, paneB);
                IsEqualMaterialColor(paneA, paneB);
            }
        }
        public static void AreEqualsPaneTestParameters(IPane pane)
        {
            IsEqualPosition(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Position));
            IsEqualSize(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Size));
            IsEqualScale(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Scale));
            IsEqualRotate(pane, (FVec3)TestData.GetTestData(TestData.DataKind.Rotate));
            IsEqualTransparency(pane, (byte)TestData.GetTestData(TestData.DataKind.Transparency));
            IsEqualVisibility(pane, (bool)TestData.GetTestData(TestData.DataKind.Visibility));

            if (pane.IPicture != null)
            {
                IsEqualPictureVertexColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                IsEqualMaterialColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
            }
            else if (pane.ITextBox != null)
            {
                IsEqualTextBoxTopBottomColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                IsEqualTextBoxShadowColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                IsEqualMaterialColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
            }
            else if (pane.ILEWindow != null)
            {
                IsEqualWindowFrameSize(pane, (FVec4)TestData.GetTestData(TestData.DataKind.WindowFrameSize));
                IsEqualWindowVertexColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
                IsEqualMaterialColor(pane, (RGBAColor)TestData.GetTestData(TestData.DataKind.RGBAColor));
            }
        }
        */

        /// <summary>
        /// 位置を比較します
        /// </summary>
        public static void IsEqualPosition(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.Trans, paneB.Trans);
        }
        public static void IsEqualPosition(IPane pane, FVec3 pos)
        {
            Assert.AreEqual(pane.Trans, pos);
        }

        /*
        /// <summary>
        /// サイズを比較します
        /// </summary>
        public static void IsEqualSize(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.Size, paneB.Size);
        }
        public static void IsEqualSize(IPane pane, FVec3 size)
        {
            Assert.AreEqual(pane.Size, size);
        }

        /// <summary>
        /// スケールを比較します
        /// </summary>
        public static void IsEqualScale(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.Scale, paneB.Scale);
        }
        public static void IsEqualScale(IPane pane, FVec3 scale)
        {
            Assert.AreEqual(pane.Scale, scale);
        }

        /// <summary>
        /// 回転を比較します
        /// </summary>
        public static void IsEqualRotate(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.RotAng, paneB.RotAng);
        }
        public static void IsEqualRotate(IPane pane, FVec3 rotate)
        {
            Assert.AreEqual(pane.RotAng, rotate);
        }

        /// <summary>
        /// ペインの透明度を比較します
        /// </summary>
        public static void IsEqualTransparency(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.Transparency, paneB.Transparency);
        }
        public static void IsEqualTransparency(IPane pane, byte trasparency)
        {
            Assert.AreEqual(pane.Transparency, trasparency);
        }

        /// <summary>
        /// ペインの表示・非表示を比較します
        /// </summary>
        public static void IsEqualVisibility(IPane paneA, IPane paneB)
        {
            Assert.AreEqual(paneA.Visible, paneB.Visible);
        }
        public static void IsEqualVisibility(IPane pane, bool visible)
        {
            Assert.AreEqual(pane.Visible, visible);
        }

        /// <summary>
        /// ウインドウペインのフレームサイズを比較します
        /// </summary>
        public static void IsEqualWindowFrameSize(IPane paneA, IPane paneB)
        {
            Debug.Assert(paneA.ILEWindow != null);
            Debug.Assert(paneB.ILEWindow != null);

            Assert.AreEqual(paneA.ILEWindow.FrameSize.Top, paneB.ILEWindow.FrameSize.Top);
            Assert.AreEqual(paneA.ILEWindow.FrameSize.Bottom, paneB.ILEWindow.FrameSize.Bottom);
            Assert.AreEqual(paneA.ILEWindow.FrameSize.Left, paneB.ILEWindow.FrameSize.Left);
            Assert.AreEqual(paneA.ILEWindow.FrameSize.Right, paneB.ILEWindow.FrameSize.Right);
        }
        public static void IsEqualWindowFrameSize(IPane pane, FVec4 frameSize)
        {
            IsEqualWindowFrameSize(pane, frameSize.X, frameSize.Y, frameSize.Z, frameSize.W);
        }
        public static void IsEqualWindowFrameSize(IPane pane, float top, float bottom, float left, float right)
        {
            Debug.Assert(pane.ILEWindow != null);

            Assert.AreEqual(pane.ILEWindow.FrameSize.Top, top);
            Assert.AreEqual(pane.ILEWindow.FrameSize.Bottom, bottom);
            Assert.AreEqual(pane.ILEWindow.FrameSize.Left, left);
            Assert.AreEqual(pane.ILEWindow.FrameSize.Right, right);
        }

        /// <summary>
        /// ピクチャペインの頂点カラーを比較します
        /// </summary>
        public static void IsEqualPictureVertexColor(IPane paneA, IPane paneB)
        {
            Debug.Assert(paneA.IPicture != null);
            Debug.Assert(paneB.IPicture != null);

            Assert.AreEqual(paneA.IPicture.VtxCol_LT, paneB.IPicture.VtxCol_LT);
            Assert.AreEqual(paneA.IPicture.VtxCol_LB, paneB.IPicture.VtxCol_LB);
            Assert.AreEqual(paneA.IPicture.VtxCol_RT, paneB.IPicture.VtxCol_RT);
            Assert.AreEqual(paneA.IPicture.VtxCol_RB, paneB.IPicture.VtxCol_RB);
        }
        public static void IsEqualPictureVertexColor(IPane pane, RGBAColor color)
        {
            Debug.Assert(pane.IPicture != null);

            Assert.AreEqual(pane.IPicture.VtxCol_LT, color);
            Assert.AreEqual(pane.IPicture.VtxCol_LB, color);
            Assert.AreEqual(pane.IPicture.VtxCol_RT, color);
            Assert.AreEqual(pane.IPicture.VtxCol_RB, color);
        }

        /// <summary>
        /// テキストボックスペインの上下端カラーを比較します
        /// </summary>
        public static void IsEqualTextBoxTopBottomColor(IPane paneA, IPane paneB)
        {
            Debug.Assert(paneA.ITextBox != null);
            Debug.Assert(paneB.ITextBox != null);

            Assert.AreEqual(paneA.ITextBox.TopColor, paneB.ITextBox.TopColor);
            Assert.AreEqual(paneA.ITextBox.BottomColor, paneB.ITextBox.BottomColor);
        }
        public static void IsEqualTextBoxTopBottomColor(IPane pane, RGBAColor color)
        {
            Debug.Assert(pane.ITextBox != null);

            Assert.AreEqual(pane.ITextBox.TopColor, color);
            Assert.AreEqual(pane.ITextBox.BottomColor, color);
        }

        /// <summary>
        /// テキストボックスペインの影カラーを比較します
        /// </summary>
        public static void IsEqualTextBoxShadowColor(IPane paneA, IPane paneB)
        {
            Debug.Assert(paneA.ITextBox != null);
            Debug.Assert(paneB.ITextBox != null);

            Assert.AreEqual(paneA.ITextBox.ShadowBlackColor, paneB.ITextBox.ShadowBlackColor);
            Assert.AreEqual(paneA.ITextBox.ShadowWhiteColor, paneB.ITextBox.ShadowWhiteColor);
        }
        public static void IsEqualTextBoxShadowColor(IPane pane, RGBAColor color)
        {
            Debug.Assert(pane.ITextBox != null);

            Assert.AreEqual(pane.ITextBox.ShadowBlackColor, color);
            Assert.AreEqual(pane.ITextBox.ShadowWhiteColor, color);
        }

        /// <summary>
        /// ウインドウペインの頂点カラーを比較します
        /// </summary>
        public static void IsEqualWindowVertexColor(IPane paneA, IPane paneB)
        {
            Debug.Assert(paneA.ILEWindow != null);
            Debug.Assert(paneB.ILEWindow != null);

            Assert.AreEqual(paneA.ILEWindow.VtxCol_LT, paneB.ILEWindow.VtxCol_LT);
            Assert.AreEqual(paneA.ILEWindow.VtxCol_LB, paneB.ILEWindow.VtxCol_LB);
            Assert.AreEqual(paneA.ILEWindow.VtxCol_RT, paneB.ILEWindow.VtxCol_RT);
            Assert.AreEqual(paneA.ILEWindow.VtxCol_RB, paneB.ILEWindow.VtxCol_RB);
        }
        public static void IsEqualWindowVertexColor(IPane pane, RGBAColor color)
        {
            Debug.Assert(pane.ILEWindow != null);

            Assert.AreEqual(pane.ILEWindow.VtxCol_LT, color);
            Assert.AreEqual(pane.ILEWindow.VtxCol_LB, color);
            Assert.AreEqual(pane.ILEWindow.VtxCol_RT, color);
            Assert.AreEqual(pane.ILEWindow.VtxCol_RB, color);
        }

        /// <summary>
        /// マテリアルカラーを比較します
        /// </summary>
        public static void IsEqualMaterialColor(IPane paneA, IPane paneB)
        {
            IRevHWMaterial[] materialsA = PaneHelper.GetRevHWMatFromPane(paneA);
            IRevHWMaterial[] materialsB = PaneHelper.GetRevHWMatFromPane(paneB);
            Debug.Assert(materialsA != null);
            Debug.Assert(materialsB != null);
            Debug.Assert(materialsA.Length == materialsB.Length);

            for (int i = 0; i < materialsA.Length; i++)
            {
                Assert.AreEqual(materialsA[i].IMaterial.BlackColor.R, materialsB[i].IMaterial.BlackColor.R);
                Assert.AreEqual(materialsA[i].IMaterial.BlackColor.G, materialsB[i].IMaterial.BlackColor.G);
                Assert.AreEqual(materialsA[i].IMaterial.BlackColor.B, materialsB[i].IMaterial.BlackColor.B);
                Assert.AreEqual(materialsA[i].IMaterial.BlackColor.A, materialsB[i].IMaterial.BlackColor.A);
                Assert.AreEqual(materialsA[i].IMaterial.WhiteColor.R, materialsB[i].IMaterial.WhiteColor.R);
                Assert.AreEqual(materialsA[i].IMaterial.WhiteColor.G, materialsB[i].IMaterial.WhiteColor.G);
                Assert.AreEqual(materialsA[i].IMaterial.WhiteColor.B, materialsB[i].IMaterial.WhiteColor.B);
                Assert.AreEqual(materialsA[i].IMaterial.WhiteColor.A, materialsB[i].IMaterial.WhiteColor.A);
            }
        }
        */
        public static void IsEqualMaterialColor(IPane pane, RGBAColor color)
        {
            IsEqualMaterialColor(pane, ConvertRGBAColorToFloatColor(color));
        }
        public static void IsEqualMaterialColor(IPane pane, FloatColor color)
        {
            IRevHWMaterial[] materials = PaneHelper.GetRevHWMatFromPane(pane);
            Debug.Assert(materials != null);

            foreach (IRevHWMaterial material in materials)
            {
                // [C]と[LT]以外は共有設定のためカラーの指定不可
                if ((material as RevHWMaterial).Description == "" ||
                    (material as RevHWMaterial).Description == "C" ||
                    (material as RevHWMaterial).Description == "LT")
                {
                    Assert.AreEqual(material.IMaterial.BlackColor.R, color.R);
                    Assert.AreEqual(material.IMaterial.BlackColor.G, color.G);
                    Assert.AreEqual(material.IMaterial.BlackColor.B, color.B);
                    Assert.AreEqual(material.IMaterial.BlackColor.A, color.A);
                    Assert.AreEqual(material.IMaterial.WhiteColor.R, color.R);
                    Assert.AreEqual(material.IMaterial.WhiteColor.G, color.G);
                    Assert.AreEqual(material.IMaterial.WhiteColor.B, color.B);
                    Assert.AreEqual(material.IMaterial.WhiteColor.A, color.A);
                }
            }
        }

        /*
        /// <summary>
        /// インダイレクトスケールを比較します
        /// </summary>
        public static void IsEqualIndirectScale(IPane paneA, IPane paneB)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// インダイレクト回転を比較します
        /// </summary>
        public static void IsEqualIndirectRotate(IPane paneA, IPane paneB)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// アルファテストを比較します
        /// </summary>
        public static void IsEqualAlphaTest(IPane paneA, IPane paneB)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// LowLevelMaterialを比較します
        /// </summary>
        public static void IsEqualLowLevelMaterial(IPane paneA, IPane paneB)
        {
            // TODO:
            //
            //
        }

        /// <summary>
        /// 拡張ユーザー情報を比較します
        /// </summary>
        public static void IsEqualExtUserData(IPane paneA, IPane paneB)
        {
            // TODO:
            //
            //
        }
        */

        #endregion

        #region ログ

        /// <summary>
        /// ログを出力します
        /// </summary>
        public static void WriteLog(string msg)
        {
            Trace.WriteLine(msg);
        }

        /// <summary>
        /// ログを出力します
        /// </summary>
        public static void WriteLog(object obj)
        {
            Trace.WriteLine(obj);
        }

        /// <summary>
        /// ログを出力します
        /// </summary>
        public static void WriteLog(string msg, string category)
        {
            Trace.WriteLine(msg, category);
        }

        /// <summary>
        /// ログを出力します
        /// </summary>
        public static void WriteLog(object obj, string category)
        {
            Trace.WriteLine(obj, category);
        }

        #endregion

        #region その他

        /// <summary>
        /// テンポラリフォルダを削除します
        /// </summary>
        public static void ClearTempDirectory()
        {
            string outPath = Path.Combine(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location), "Output");

            // 一時フォルダが残っていれば消す
            if (Directory.Exists(outPath))
            {
                Directory.Delete(outPath, true);
            }

            // Delete直後にCreate出来ないケースがあるようなので一定時間待たせる
            System.Threading.Thread.Sleep(10);
            Directory.CreateDirectory(outPath);
        }

        /// <summary>
        /// ファイルを複製します
        /// </summary>
        public static string CopyFile(string srcFileName, string dstFileName)
        {
            string dir = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts");
            string src = dir + "\\" + srcFileName;
            string dst = dir + "\\" + dstFileName;

            Debug.Assert(File.Exists(src)) ;

            File.Copy(src, dst);

            return dst;
        }

        /// <summary>
        /// ファイルを削除します
        /// </summary>
        public static string DeleteFile(string fileName)
        {
            string dir = Path.Combine(AppConstants.NwUnitTestDataRoot, "parts");
            string src = dir + "\\" + fileName;

            File.Delete(src);

            return src;
        }

        /// <summary>
        /// RGBAColorをFloatColorに変換します
        /// </summary>
        public static FloatColor ConvertRGBAColorToFloatColor(RGBAColor color)
        {
            float r = color.R / 255.0f;
            float g = color.G / 255.0f;
            float b = color.B / 255.0f;
            float a = color.A / 255.0f;

            return new FloatColor(r, g, b, a);
        }

        /// <summary>
        /// アトリビュートツリーを文字列で取得します
        /// </summary>
        internal static string GetAttributeTreeString(IAnmAttribute baseAttr)
        {
            string str = String.Empty;
            List<string> nameList = new List<string>();

            IAnmAttribute attr = baseAttr;
            while (true)
            {
                nameList.Add(attr.Name);
                if (attr.OwnerNode == null)
                {
                    nameList.RemoveAt(nameList.Count-1);
                    if (attr is Pane)
                    {
                        nameList.Add((attr as Pane).PaneName);
                    }

                    break;
                }
                attr = attr.OwnerNode;
            }

            nameList.Reverse();
            foreach (string name in nameList)
            {
                str += name + ".";
            }

            return str.Substring(0, str.Length - 1);
        }

        /// <summary>
        /// レイアウト内のアトリビュートを列挙
        /// </summary>
        public static IEnumerable<IAnmAttribute> EnumerateAttributes(ISubScene subScene)
        {
            return from pane in subScene.IPaneArray
                   from kind in ParamaterKindHelper.AnimationKindSet
                   from root in pane.GetAnimationTargetAttributeSet(kind)
                   from node in EnumerateAttributes(root)
                   select node;
        }

        /// <summary>
        /// root 以下のアトリビュートを root も含めて全て再帰的に列挙
        /// </summary>
        public static IEnumerable<IAnmAttribute> EnumerateAttributes(IAnmAttribute root)
        {
            return Enumerable.Repeat(root, 1).Concat(
                Enumerable.Range(0, root.NumSubAttribute).SelectMany(x => EnumerateAttributes(root.FindSubAttributeByIdx(x))));
        }
        #endregion
    }
}
