﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore.Structures;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System;
using System.Diagnostics;
using System.Linq;
using System.Reflection;

namespace LECoreTest
{
    /// <summary>
    /// レイアウトに関するテスト項目
    ///
    /// ※※※※※※※※※※※※※※※※※※※※※※※※※
    /// TODO: 詳細マテリアルのパラメータ確認
    ///        *インダイレクト
    ///        *アルファテスト
    ///        *LowLevelMaterial
    ///       TestUtilityに該当の処理を追加すること
    /// ※※※※※※※※※※※※※※※※※※※※※※※※※
    ///
    /// </summary>
    [TestClass]
    public class LayoutTest
    {
        #region レイアウトの「新規作成」、「閉じる」が行なえる

        /// <summary>
        /// レイアウトの「新規作成」、「閉じる」が行なえる
        /// </summary>
        [TestMethod]
        public void LayoutTest_CreateAndClose()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // シーンにサブシーンが追加されたか確認する
                Assert.AreEqual(1, TestUtility.Scene.ISubSceneSet.Length);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);

                // シーンからサブシーンが削除されたか確認する
                Assert.AreEqual(0, TestUtility.Scene.ISubSceneSet.Length);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region レイアウトの「読み込み」、「編集」、「保存」ができる 「再読み込み」ができる

        /// <summary>
        /// レイアウトの「読み込み」、「編集」、「保存」ができる
        /// 「再読み込み」ができる
        /// </summary>
        [TestMethod]
        public void LayoutTest_ReadEditSaveAndReload()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // アニメーションなしレイアウトの読み込み(結合モード)　時間＝0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPane.flyt");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // 正しく再読み込みされていることを確認する
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーションなしレイアウトの読み込み(結合モード)　時間≠0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPane.flyt");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 時間を進める
                    TestUtility.SetTime(100);

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // 正しく再読み込みされていることを確認する
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーション付きレイアウトの読み込み(結合モード)　時間＝0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 一旦レイアウトを閉じる
                    TestUtility.CloseLayout(layout);

                    // 元ファイルと保存したファイルを開く
                    layout = TestUtility.LoadLayout("allPaneAnim.flyt");
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // 正しく再読み込みされていることを確認する
                    // allPaneAnim.flytは全アトリビュートにカーブが設定されているので、SetAttributeValueAllで設定した値は保存時に失われる
                    // そのため、allPaneAnim.flytとLayoutTest_ReadEditSaveAndReload.flytはアトリビュート値が同じになるはず
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーション付きレイアウトの読み込み(結合モード)　時間≠0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim.flyt");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 時間を進める
                    TestUtility.SetTime(100);

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 一旦レイアウトを閉じる
                    TestUtility.CloseLayout(layout);

                    // 元ファイルと保存したファイルを開く
                    layout = TestUtility.LoadLayout("allPaneAnim.flyt");
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // 正しく再読み込みされていることを確認する
                    // allPaneAnim.flytは全アトリビュートにカーブが設定されているので、SetAttributeValueAllで設定した値は保存時に失われる
                    // そのため、allPaneAnim.flytとLayoutTest_ReadEditSaveAndReload.flytはアトリビュート値が同じになるはず
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーションなしレイアウトの読み込み(分割モード)　時間＝0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPane_separated.flyt");

                    // 区間を切り替える
                    TestUtility.ModifyAnimFrameSectionFromName(layout, "Tag_00");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // テストデータを開きなおして初期状態に戻す
                    TestUtility.CloseLayout(layout);
                    layout = TestUtility.LoadLayout("allPane_separated.flyt");

                    // 正しく再読み込みされていることを確認する
                    // 分割モードの場合、現在値は破棄されbaseValueで保存される
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーションなしレイアウトの読み込み(分割モード)　時間≠0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPane_separated.flyt");

                    // 区間を切り替える
                    TestUtility.ModifyAnimFrameSectionFromName(layout, "Tag_00");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 時間を進める
                    TestUtility.SetTime(100);

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // テストデータを開きなおして初期状態に戻す
                    TestUtility.CloseLayout(layout);
                    layout = TestUtility.LoadLayout("allPane_separated.flyt");

                    // 正しく再読み込みされていることを確認する
                    // 分割モードの場合、現在値は破棄されbaseValueで保存される
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーション付きレイアウトの読み込み(分割モード)　時間＝0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    // 区間を切り替える
                    TestUtility.ModifyAnimFrameSectionFromName(layout, "Tag_00");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // テストデータを開きなおして初期状態に戻す
                    TestUtility.CloseLayout(layout);
                    layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    // 正しく再読み込みされていることを確認する
                    // 分割モードの場合、現在値は破棄されbaseValueで保存される
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }

                // アニメーション付きレイアウトの読み込み(分割モード)　時間≠0
                {
                    // テストデータを読み込む
                    ISubScene layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    // 区間を切り替える
                    TestUtility.ModifyAnimFrameSectionFromName(layout, "Tag_00");

                    // ペインを操作する
                    foreach (IPane pane in layout.IPaneArray)
                    {
                        TestUtility.ModifyPaneTestParameters(pane);
                    }

                    // 時間を進める
                    TestUtility.SetTime(100);

                    // 保存する
                    string saveFilePath = TestUtility.SaveLayout(layout, "LayoutTest_ReadEditSaveAndReload.flyt");

                    // 保存したファイルを開く
                    ISubScene layout2 = TestUtility.LoadLayoutFullPath(saveFilePath);

                    // テストデータを開きなおして初期状態に戻す
                    TestUtility.CloseLayout(layout);
                    layout = TestUtility.LoadLayout("allPaneAnim_separated.flyt");

                    // 正しく再読み込みされていることを確認する
                    // 分割モードの場合、現在値は破棄されbaseValueで保存される
                    Debug.Assert(layout.IPaneArray.Count() == layout2.IPaneArray.Count());
                    for (int i = 0; i < layout.IPaneArray.Count(); i++)
                    {
                        TestUtility.AreEqualsPaneParameters(layout.IPaneArray.ElementAt(i), layout2.IPaneArray.ElementAt(i));
                    }

                    // レイアウトを閉じる
                    TestUtility.CloseLayout(layout);
                    TestUtility.CloseLayout(layout2);
                }
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion
    }
}
