﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore;
using LECore.Structures;
using LECore.Structures.Core;
using LECore.Structures.Core.Command;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Linq;
using System.Reflection;

namespace LECoreTest
{
    /// <summary>
    /// アニメーション区間タグに関するテスト項目
    /// </summary>
    [TestClass]
    public class AnimSectionTest
    {
        #region 区間タグの「作成」、「編集」、「削除」ができる

        /// <summary>
        /// 区間タグの「作成」、「編集」、「削除」ができる
        /// </summary>
        [TestMethod]
        public void AnimSectionTest_AddEditAndRemove()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);
                IAnimFrameSection sectionB = TestUtility.AddAnimFrameSection(newSubScene, "Tag_01", 1, 101);
                IAnimFrameSection sectionC = TestUtility.AddAnimFrameSection(newSubScene, "Tag_02", 2, 102);

                // 区間が作成されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 3);

                // 区間を編集する
                TestUtility.EditAnimFrameSection(newSubScene, sectionA, "Tag_00_Edit", 10, 1000);
                TestUtility.EditAnimFrameSection(newSubScene, sectionB, "Tag_01_Edit", 11, 1001);
                TestUtility.EditAnimFrameSection(newSubScene, sectionC, "Tag_02_Edit", 12, 1002);

                // 編集されたことを確認する
                IAnimFrameSection section;
                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_00_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 10);
                Assert.AreEqual(section.EndFrame, 1000);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_01_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 11);
                Assert.AreEqual(section.EndFrame, 1001);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_02_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 12);
                Assert.AreEqual(section.EndFrame, 1002);

                // 区間を削除する
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionA);
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionB);
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionC);

                // 区間が削除されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 0);

                // Undoする
                var cmdQueue = (LayoutEditorCore.Scene.CurrentISubScene.ISubSceneCmdQueue as SubSceneCmdQueue);
                cmdQueue.Undo();
                cmdQueue.Undo();
                cmdQueue.Undo();

                // 区間が復元されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 3);

                // Undoする
                cmdQueue.Undo();
                cmdQueue.Undo();
                cmdQueue.Undo();

                // 編集されたことを確認する
                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_00");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 0);
                Assert.AreEqual(section.EndFrame, 100);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_01");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 1);
                Assert.AreEqual(section.EndFrame, 101);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_02");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 2);
                Assert.AreEqual(section.EndFrame, 102);

                // Undoする
                cmdQueue.Undo();
                cmdQueue.Undo();
                cmdQueue.Undo();

                // 区間が削除されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 0);

                // Redoする
                cmdQueue.Redo();
                cmdQueue.Redo();
                cmdQueue.Redo();

                // 区間が復元されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 3);

                // Redoする
                cmdQueue.Redo();
                cmdQueue.Redo();
                cmdQueue.Redo();

                // 編集されたことを確認する
                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_00_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 10);
                Assert.AreEqual(section.EndFrame, 1000);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_01_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 11);
                Assert.AreEqual(section.EndFrame, 1001);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_02_Edit");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 12);
                Assert.AreEqual(section.EndFrame, 1002);

                // Redoする
                cmdQueue.Redo();
                cmdQueue.Redo();
                cmdQueue.Redo();

                // 区間が削除されたことを確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 0);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 区間タグの「エクスポート」、「インポート」ができる

        /// <summary>
        /// 区間タグの「エクスポート」、「インポート」ができる
        /// </summary>
        [TestMethod]
        public void AnimSectionTest_ImportAndExport()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);
                IAnimFrameSection sectionB = TestUtility.AddAnimFrameSection(newSubScene, "Tag_01", 1, 101);
                IAnimFrameSection sectionC = TestUtility.AddAnimFrameSection(newSubScene, "Tag_02", 2, 102);

                // エクスポートする
                string fileName = "TagFile";
                TestUtility.ExportAnimSection(newSubScene, fileName);

                // 区間を削除する
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionA);
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionB);
                TestUtility.RemoveAnimFrameSection(newSubScene, sectionC);

                // インポートする
                TestUtility.ImportAnimSection(newSubScene, fileName);

                // 区間が読み込まれたか確認する
                Assert.AreEqual(newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.Count(), 3);

                IAnimFrameSection section;
                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_00");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 0);
                Assert.AreEqual(section.EndFrame, 100);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_01");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 1);
                Assert.AreEqual(section.EndFrame, 101);

                section = newSubScene.IAnimFrameSectionSet.IAnimFrameSectionSet.FirstOrDefault(s => s.Name == "Tag_02");
                Assert.AreNotEqual(section, null);
                Assert.AreEqual(section.StartFrame, 2);
                Assert.AreEqual(section.EndFrame, 102);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 分割モードで区間タグごとにアニメーションを持てる

        /// <summary>
        /// 分割モードで区間タグごとにアニメーションを持てる
        /// </summary>
        [TestMethod]
        public void AnimSectionTest_HasSeparatedAnimation()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加
                IPane nullPane = TestUtility.AddNullPane(newSubScene, "NullPane");
                float defaultX = nullPane.Trans.X;

                // アトリビュートを取得
                IAnmAttribute posXAttr = nullPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);
                IAnimFrameSection sectionB = TestUtility.AddAnimFrameSection(newSubScene, "Tag_01", 1, 101);
                IAnimFrameSection sectionC = TestUtility.AddAnimFrameSection(newSubScene, "Tag_02", 2, 102);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 10, 10);

                // 区間を変更
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_01");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 20, 20);

                // 区間を変更
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_02");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 30, 30);

                // 区間を未選択状態にする
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, null);

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 0);
                Assert.AreEqual(nullPane.Trans.X, defaultX);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 10);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_01");

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 20);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_02");

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 30);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 分割モード時、フィッティングが行える

        /// <summary>
        /// 分割モード時、フィッティングが行える
        /// </summary>
        [TestMethod]
        public void AnimSectionTest_Fitting()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加
                IPane nullPane = TestUtility.AddNullPane(newSubScene, "NullPane");
                float defaultX = nullPane.Trans.X;

                // アトリビュートを取得
                IAnmAttribute posXAttr = nullPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);
                IAnimFrameSection sectionB = TestUtility.AddAnimFrameSection(newSubScene, "Tag_01", 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 10, 0);
                TestUtility.AddAnimKey(posXAttr, 50, 0);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_01");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 20, 0);
                TestUtility.AddAnimKey(posXAttr, 70, 0);

                // フィッティングする
                TestUtility.FittingSection(newSubScene);

                // 比較する
                Assert.AreEqual(sectionA.StartFrame, 0);
                Assert.AreEqual(sectionA.EndFrame, 100); // 選択中の区間だけがフィッティングされる
                Assert.AreEqual(sectionB.StartFrame, 0);
                Assert.AreEqual(sectionB.EndFrame, 70);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 150, 0);

                // 区間を選択
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_01");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 170, 0);

                // フィッティングする
                TestUtility.FittingSection(newSubScene);

                // 比較する
                Assert.AreEqual(sectionA.StartFrame, 0);
                Assert.AreEqual(sectionA.EndFrame, 100); // 選択中の区間だけがフィッティングされる
                Assert.AreEqual(sectionB.StartFrame, 0);
                Assert.AreEqual(sectionB.EndFrame, 170);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion
    }
}
