﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using LECore.Structures;
using LECore.Structures.Core;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using System.Reflection;

namespace LECoreTest
{
    /// <summary>
    /// アニメーションカーブに関するテスト項目
    /// </summary>
    [TestClass]
    public class AnimCurveTest
    {
        #region アニメーションキーの「追加」、「編集」、「削除」ができる

        /// <summary>
        /// アニメーションキーの「追加」、「編集」、「削除」ができる
        /// </summary>
        [TestMethod]
        public void AnimCurveTest_AddEditAndRemove()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // 結合モード
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加
                IPane nullPane = TestUtility.AddNullPane(newSubScene, "NullPane");
                float defaultX = nullPane.Trans.X;

                // アトリビュートを取得
                IAnmAttribute posXAttr = nullPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

                // キーを打つ
                IAnmKeyFrame key = TestUtility.AddAnimKey(posXAttr, 10, 10);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 10);

                // キーを変更する
                TestUtility.ModifyAnimKey(key, 20, 20);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 20);

                // キーを削除する
                TestUtility.RemoveAnimKey(key);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 0);
                Assert.AreEqual(nullPane.Trans.X, 20); // アニメーションキーが無い場合再評価しても値は変更されない

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 分割モード
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加
                IPane nullPane = TestUtility.AddNullPane(newSubScene, "NullPane");
                float defaultX = nullPane.Trans.X;

                // アトリビュートを取得
                IAnmAttribute posXAttr = nullPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間を変更する
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // キーを打つ
                IAnmKeyFrame key = TestUtility.AddAnimKey(posXAttr, 10, 10);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 10);

                // キーを変更する
                TestUtility.ModifyAnimKey(key, 20, 20);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);
                Assert.AreEqual(nullPane.Trans.X, 20);

                // キーを削除する
                TestUtility.RemoveAnimKey(key);
                TestUtility.SetTime(GlobalTime.Inst.Time); // 現在時間で再評価する

                // 比較
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 0);
                Assert.AreEqual(nullPane.Trans.X, 20); // アニメーションキーが無い場合再評価しても値は変更されない

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion

        #region 前後区間へのスナップが行える

        /// <summary>
        /// 前後区間へのスナップが行える
        /// </summary>
        [TestMethod]
        public void AnimCurveTest_Snap()
        {
            // 準備
            {
                TestUtility.InitializeTest(MethodBase.GetCurrentMethod());
            }

            // テスト
            {
                // レイアウトを新規作成
                ISubScene newSubScene = TestUtility.CreateLayout();

                // ヌルペインを追加
                IPane nullPane = TestUtility.AddNullPane(newSubScene, "NullPane");
                float defaultX = nullPane.Trans.X;

                // アトリビュートを取得
                IAnmAttribute posXAttr = nullPane.IPaneAttribute.TransIAttr.FindSubAttributeByIdx(0);

                // 区間を作成する
                IAnimFrameSection sectionA = TestUtility.AddAnimFrameSection(newSubScene, "Tag_00", 0, 100);
                IAnimFrameSection sectionB = TestUtility.AddAnimFrameSection(newSubScene, "Tag_01", 0, 100);
                IAnimFrameSection sectionC = TestUtility.AddAnimFrameSection(newSubScene, "Tag_02", 0, 100);

                // 分割モードに変更する
                TestUtility.ChangeAnimEditMode(newSubScene);

                // 区間を変更する
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_00");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 0, 10);
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);

                // 区間を変更する
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_02");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 0, 20);
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 1);

                // 区間を変更する
                TestUtility.ModifyAnimFrameSectionFromName(newSubScene, "Tag_01");

                // キーを打つ
                TestUtility.AddAnimKey(posXAttr, 10, 30);
                TestUtility.AddAnimKey(posXAttr, 20, 40);
                Assert.AreEqual(posXAttr.ICurrentAnimationCurve.NumKeyFrame, 2);

                // スナップする
                TestUtility.SnapCurve(newSubScene, posXAttr);

                // 比較
                TestUtility.SetTime(10);
                Assert.AreEqual(nullPane.Trans.X, 10);
                TestUtility.SetTime(20);
                Assert.AreEqual(nullPane.Trans.X, 20);

                // レイアウトを閉じる
                TestUtility.CloseLayout(newSubScene);
            }

            // 後処理
            {
                TestUtility.FinalizeTest(MethodBase.GetCurrentMethod());
            }
        }

        #endregion
    }
}
