﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Diagnostics;
using System.Text;
using Microsoft.VisualStudio.TestTools.UnitTesting;

namespace InstallTest
{
    [TestClass]
    public class ExecutionTest
    {
        public TestContext TestContext { get; set; }
        private static string m_TargetName = string.Empty;

        [ClassInitialize]
        public static void TestClassinitialize(TestContext context)
        {
            m_TargetName = (string)context.Properties["TargetName"];
        }

        private bool ExecuteDevMenuCommand(string args)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommand\\Release\\DevMenuCommand.nsp");
        }

        private bool ExecuteDevMenuCommandSystem(string args)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandCommon(args, testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommandSystem\\Release\\DevMenuCommandSystem.nsp");
        }
        private bool ExecuteDevMenuCommandToLaunchApplicationAndExpectOutput(string id, string expectOutput, int timeoutSeconds)
        {
            var testPath = new TestUtility.TestPath(this.TestContext);

            return ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon(id, expectOutput, timeoutSeconds,
                       testPath.GetSigloRoot() + "\\Programs\\Eris\\Outputs\\NX-NXFP2-a64\\TargetTools\\DevMenuCommand\\Release\\DevMenuCommand.nsp");
        }

        private Tuple<StringBuilder, StringBuilder> SetupProcess(Process process, string filename, string arguments)
        {
            var standardOutput = new StringBuilder();
            var standardError = new StringBuilder();

            process.StartInfo.FileName = filename;
            process.StartInfo.Arguments = arguments;
            process.StartInfo.CreateNoWindow = true;
            process.StartInfo.UseShellExecute = false;
            process.StartInfo.RedirectStandardOutput = true;
            process.StartInfo.RedirectStandardError = true;
            process.StartInfo.StandardOutputEncoding = System.Text.Encoding.UTF8;
            process.OutputDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardOutput.AppendLine(e.Data);
            };
            process.ErrorDataReceived += (object sender, DataReceivedEventArgs e) =>
            {
                standardError.AppendLine(e.Data);
            };

            return new Tuple<StringBuilder, StringBuilder>(standardOutput, standardError);
        }
        private void RunAndWaitProcess(Process process)
        {
            Console.WriteLine("process start.");
            process.Start();
            process.BeginOutputReadLine();
            process.BeginErrorReadLine();
            Console.WriteLine("process wait for exit.");
            process.WaitForExit();
        }

        private bool ExecuteDevMenuCommandCommon(string args, string ncaPath)
        {
            Console.WriteLine("ExecuteDevMenuCommandCommon start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var arguments = " -t " + m_TargetName +
                                " " + ncaPath +
                                " --pattern-failure-exit \\[FAILURE\\] " +
                                " -v " +
                                "-- " +
                                args;

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }
        private bool ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon(string id, string expectOutput, int timeoutSeconds, string ncaPath)
        {
            Console.WriteLine("ExecuteDevMenuCommandToLaunchApplicationAndExpectOutputCommon start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTargetPrivate.exe";
                var arguments = "run " +
                                " " + ncaPath +
                                " -t " + m_TargetName +
                                " --suppress-polling-process " +
                                " --pattern-success-exit " +
                                expectOutput +
                                " --pattern-failure-exit \\[FAILURE\\] " +
                                " --failure-timeout " +
                                string.Format("{0}", timeoutSeconds) +
                                " -v " +
                                " -- " +
                                " application " +
                                " launch " +
                                id;

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool IsApplicationInstalled(string id, bool withReset = false)
        {
            Console.WriteLine("IsApplicationInstalled start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var ncaPath = testPath.GetSigloRoot()
                    + @"\Programs\Eris\Outputs\NX-NXFP2-a64\TargetTools\DevMenuCommandSystem\Release\DevMenuCommandSystem.nsp";

                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\RunOnTarget.exe";
                var resetOption = withReset ? "--reset" : string.Empty;
                var arguments = " -t " + m_TargetName +
                                " " + ncaPath +
                                " " + resetOption + " " +
                                " --pattern-failure-exit \\[FAILURE\\] " +
                                " -v " +
                                " -- " +
                                " application " +
                                " list ";

                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                if (process.ExitCode != 0)
                {
                    throw new Exception("IsApplicationInstalled Exception: RunOnTarget Failure");
                }

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                if (outputs.Item1.ToString().Contains(id))
                {
                    return true;
                }

                return false;
            }
        }

        private bool ExecuteControlTarget(string args)
        {
            Console.WriteLine($"ControlTarget {args} start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\ControlTarget.exe";
                var arguments = args;
                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        private bool ExecuteMakeTestApplication(string args)
        {
            Console.WriteLine($"MakeTestApplication {args} start.");

            using (var process = new Process())
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                var filename = testPath.GetSigloRoot() + "\\Tools\\CommandLineTools\\MakeTestApplication\\MakeTestApplication.exe";
                var arguments = args;
                var outputs = SetupProcess(process, filename, arguments);
                RunAndWaitProcess(process);

                Console.WriteLine(string.Format("Standard Output: {0}", outputs.Item1.ToString()));
                Console.WriteLine(string.Format("Standard Error: {0}", outputs.Item2.ToString()));

                return process.ExitCode == 0;
            }
        }

        [TestMethod]
        public void TestInstallAndRunExternalKeyApplication()
        {
            string applicationId = "0x0100000000010000";
            string applicationFileName = "TestApplication";
            string applicationNspDirectory = System.IO.Directory.GetCurrentDirectory() + "\\" + "TestApplicationNsp";
            string applicationNspPath = applicationNspDirectory + "\\" + applicationFileName + ".nsp";

            System.IO.Directory.CreateDirectory(applicationNspDirectory);
            Assert.IsTrue(ExecuteMakeTestApplication("-o " + applicationNspDirectory + " --id " + applicationId + " --output-file-name " + applicationFileName + " --ticket --small-code"));
            Assert.IsTrue(ExecuteDevMenuCommand("application install " + applicationNspPath));
            Assert.IsTrue(ExecuteDevMenuCommand("application launch " + applicationId));
            Assert.IsTrue(ExecuteDevMenuCommand("application uninstall " + applicationId));
            System.IO.Directory.Delete(applicationNspDirectory, true);
        }
    }
}
