﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using System.Threading.Tasks;

using Nintendo.HtcTools.Htclow;

namespace Nintendo.HtcTools.Htclow
{
    [TestClass]
    public class ProtocolTest
    {
        private const int ModuleId = 0;
        private const int ChannelId = 0;

        private Channel m_Channel = new Channel() { ModuleId = ModuleId, ChannelId = ChannelId };

        #region Version Test

        [TestMethod]
        public void DefaultVersion()
        {
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                service.SendPacket(factory.MakeSynPacket(m_Channel));

                htclowChannel.Connect();

                var expected = factory.MakeSynAckPacket(m_Channel);
                var actual = service.ReceivePacket();

                Assert.AreEqual(expected, actual);
            }
        }

        [TestMethod]
        public void HigherVersion()
        {
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                var packet = factory.MakeSynPacket(m_Channel);
                packet.Version = PacketFactory.MaxVersion + 1;

                service.SendPacket(packet);

                htclowChannel.Connect();

                var expected = factory.MakeSynAckPacket(m_Channel);
                var actual = service.ReceivePacket();

                Assert.AreEqual(expected, actual);
            }
        }

        [TestMethod]
        public void LowerVersion()
        {
            var factory = new PacketFactory()
            {
                Version = 0,
            };

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                var packet = factory.MakeSynPacket(m_Channel);

                service.SendPacket(packet);

                htclowChannel.Connect();

                var expected = factory.MakeSynAckPacket(m_Channel);
                var actual = service.ReceivePacket();

                Assert.AreEqual(expected, actual);
            }
        }

        #endregion

        #region Protocol Violation Test

        [TestMethod]
        [ExpectedException(typeof(HtclowException))]
        public void IncorrectProtocolId()
        {
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                var packet = factory.MakeSynPacket(m_Channel);

                // Incorrect protocol
                packet.GetBytes()[0] = 0xff;
                packet.GetBytes()[1] = 0xff;

                service.SendPacket(packet);

                htclowChannel.Connect();
            }
        }

        [TestMethod]
        [ExpectedException(typeof(HtclowException))]
        public void IncorrectBodySize()
        {
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId);

                // 1 バイトのボディを持つ Data パケットを書き換えて 1 バイトのボディを持つ不正な Syn パケットを作る
                var packet = factory.MakeDataPacket(m_Channel, new byte[] { 0x01 });
                packet.PacketType = PacketType.Syn;
                packet.SequenceId = 0;

                service.SendPacket(packet);

                htclowChannel.Connect();
            }
        }

        #endregion

    }
}
