﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using Microsoft.VisualStudio.TestTools.UnitTesting;

using System.Threading.Tasks;

using Nintendo.HtcTools.Htclow;

namespace Nintendo.HtcTools.Htclow
{
    [TestClass]
    public class BoundaryTest
    {
        private const int ModuleId = 0;
        private const int ChannelId = 0;

        private Channel m_Channel = new Channel() { ModuleId = ModuleId, ChannelId = ChannelId };

        #region 送受信サイズの境界テスト

        /// <summary>
        /// 0 バイトの Send() を混ぜても正しく送信できることをテストする
        /// </summary>
        [TestMethod]
        public void ZeroByteSend()
        {
            const int DataSize1 = 1;
            const int DataSize2 = 16;
            const int DataSize3 = Packet.MaxBodySize;

            // 送信データ作成
            var factory = new PacketFactory();

            var data0 = new byte[0];
            var data1 = DataUtil.MakeRandom(DataSize1);
            var data2 = DataUtil.MakeRandom(DataSize2);
            var data3 = DataUtil.MakeRandom(DataSize3);

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                using (var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId))
                {
                    var util = new HandshakeUtil(factory, service);
                    util.DoHandshakeByHost(htclowChannel, m_Channel);

                    // データの送信
                    htclowChannel.Send(data0);
                    htclowChannel.Send(data1);
                    htclowChannel.Send(data0);
                    htclowChannel.Send(data2);
                    htclowChannel.Send(data0);
                    htclowChannel.Send(data0);
                    htclowChannel.Send(data0);
                    htclowChannel.Send(data3);

                    // データの受信
                    var expect1 = factory.MakeDataPacket(m_Channel, data1, 1);
                    var actual1 = service.ReceivePacket();
                    Assert.AreEqual(expect1, actual1);

                    var expect2 = factory.MakeDataPacket(m_Channel, data2, 2);
                    var actual2 = service.ReceivePacket();
                    Assert.AreEqual(expect2, actual2);

                    var expect3 = factory.MakeDataPacket(m_Channel, data3, 3);
                    var actual3 = service.ReceivePacket();
                    Assert.AreEqual(expect3, actual3);
                }
            }
        }

        /// <summary>
        /// ホスト側で 0 バイトの Receive() を混ぜても、正しい受信ができることをテストする
        /// </summary>
        [TestMethod]
        public void ZeroByteReceive()
        {
            const int DataSize1 = 1;
            const int DataSize2 = 16;
            const int DataSize3 = Packet.MaxBodySize;

            // 送信データ作成
            var factory = new PacketFactory();

            var data0 = new byte[0];
            var data1 = DataUtil.MakeRandom(DataSize1);
            var data2 = DataUtil.MakeRandom(DataSize2);
            var data3 = DataUtil.MakeRandom(DataSize3);

            var dataPackets = new Packet[]
            {
                factory.MakeDataPacket(m_Channel, data1),
                factory.MakeDataPacket(m_Channel, data2),
                factory.MakeDataPacket(m_Channel, data3),
            };

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                using (var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId))
                {
                    var util = new HandshakeUtil(factory, service);
                    util.DoHandshakeByHost(htclowChannel, m_Channel);

                    // ターゲット側データの送信
                    foreach (var packet in dataPackets)
                    {
                        service.SendPacket(packet);
                    }

                    // ターゲット側 ack の受信
                    for (long i = 0; i < dataPackets.Length; i++)
                    {
                        var expected = factory.MakeDataAckPacket(m_Channel, i + 1);
                        var actual = service.ReceivePacket();

                        Assert.AreEqual(expected, actual);
                    }

                    // ホスト側データの受信
                    byte[] buffer0;

                    buffer0 = htclowChannel.Receive(0);
                    CollectionAssert.AreEqual(data0, buffer0);

                    var buffer1 = htclowChannel.Receive(DataSize1);
                    CollectionAssert.AreEqual(data1, buffer1);

                    buffer0 = htclowChannel.Receive(0);
                    CollectionAssert.AreEqual(data0, buffer0);

                    var buffer2 = htclowChannel.Receive(DataSize2);
                    CollectionAssert.AreEqual(data2, buffer2);

                    buffer0 = htclowChannel.Receive(0);
                    CollectionAssert.AreEqual(data0, buffer0);

                    buffer0 = htclowChannel.Receive(0);
                    CollectionAssert.AreEqual(data0, buffer0);

                    buffer0 = htclowChannel.Receive(0);
                    CollectionAssert.AreEqual(data0, buffer0);

                    var buffer3 = htclowChannel.Receive(DataSize3);
                    CollectionAssert.AreEqual(data3, buffer3);
                }
            }
        }

        /// <summary>
        /// 対向が 0 バイトのデータパケットを混ぜて送信しても、
        /// 正しく受信できることをテストする
        /// </summary>
        [TestMethod]
        public void ReceiveEmptyDataPacket()
        {
            const int DataSize1 = 1;
            const int DataSize2 = 16;
            const int DataSize3 = Packet.MaxBodySize;

            // 送信データ作成
            var factory = new PacketFactory();

            var data0 = new byte[0];
            var data1 = DataUtil.MakeRandom(DataSize1);
            var data2 = DataUtil.MakeRandom(DataSize2);
            var data3 = DataUtil.MakeRandom(DataSize3);

            var dataPackets = new Packet[]
            {
                factory.MakeDataPacket(m_Channel, data0),
                factory.MakeDataPacket(m_Channel, data1),
                factory.MakeDataPacket(m_Channel, data0),
                factory.MakeDataPacket(m_Channel, data2),
                factory.MakeDataPacket(m_Channel, data0),
                factory.MakeDataPacket(m_Channel, data0),
                factory.MakeDataPacket(m_Channel, data0),
                factory.MakeDataPacket(m_Channel, data3),
            };

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                using (var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId))
                {
                    var util = new HandshakeUtil(factory, service);
                    util.DoHandshakeByHost(htclowChannel, m_Channel);

                    // ターゲット側データの送信
                    foreach (var packet in dataPackets)
                    {
                        service.SendPacket(packet);
                    }

                    // ターゲット側 ack の受信
                    for (long i = 0; i < dataPackets.Length; i++)
                    {
                        var expected = factory.MakeDataAckPacket(m_Channel, i + 1);
                        var actual = service.ReceivePacket();

                        Assert.AreEqual(expected, actual);
                    }

                    // ホスト側データの受信
                    var buffer1 = htclowChannel.Receive(DataSize1);
                    CollectionAssert.AreEqual(data1, buffer1);

                    var buffer2 = htclowChannel.Receive(DataSize2);
                    CollectionAssert.AreEqual(data2, buffer2);

                    var buffer3 = htclowChannel.Receive(DataSize3);
                    CollectionAssert.AreEqual(data3, buffer3);
                }
            }
        }
        #endregion

        #region パケット境界のテスト

        /// <summary>
        /// 複数のパケットを1回の Receive() で取得できることをテストする
        /// </summary>
        [TestMethod]
        public void OneReceiveForMultiPacket()
        {
            var dataSet = new int[][] {
                new int[] { 1, 2, 3, 4, 5, 6, 7, 8 },
                new int[] { Packet.MaxBodySize, Packet.MaxBodySize, Packet.MaxBodySize, 1 },
            };

            // 送信データ作成
            int sequenceId = 1;
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                using (var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId))
                {
                    var util = new HandshakeUtil(factory, service);
                    util.DoHandshakeByHost(htclowChannel, m_Channel);

                    foreach (var sizeList in dataSet)
                    {
                        // 送信データ準備
                        var expected = DataUtil.MakeRandom(sizeList.Sum());

                        // 複数回に分けて送信
                        int offset = 0;
                        foreach (var size in sizeList)
                        {
                            var data = new byte[size];
                            System.Buffer.BlockCopy(expected, offset, data, 0, size);

                            var packet = factory.MakeDataPacket(m_Channel, data, sequenceId);
                            service.SendPacket(packet);
                            sequenceId++;

                            offset += size;
                        }

                        // 1回で受信
                        var actual = htclowChannel.Receive(sizeList.Sum());

                        // 比較
                        CollectionAssert.AreEqual(expected, actual);
                    }
                }
            }
        }

        /// <summary>
        /// 1つのパケットを複数回の Receive() で取得できることをテストする
        /// </summary>
        [TestMethod]
        public void MultiReceiveForOnePacket()
        {
            var dataSet = new int[][] {
                new int[] { 1, 2, 3, 4, 5, 6, 7, 8 },
                new int[] { Packet.MaxBodySize - 1, 1 },
            };

            // 送信データ作成
            int sequenceId = 1;
            var factory = new PacketFactory();

            using (var service = new HtclowLibraryTestService())
            {
                var htclowClient = service.HtclowClient;
                using (var htclowChannel = htclowClient.OpenChannel(ModuleId, ChannelId))
                {
                    var util = new HandshakeUtil(factory, service);
                    util.DoHandshakeByHost(htclowChannel, m_Channel);

                    foreach (var sizeList in dataSet)
                    {
                        // 送信データ準備
                        var data = DataUtil.MakeRandom(sizeList.Sum());

                        // 1回のパケット送信
                        var packet = factory.MakeDataPacket(m_Channel, data, sequenceId);
                        service.SendPacket(packet);
                        sequenceId++;

                        // 複数回に分けて受信 & 比較
                        int offset = 0;
                        foreach (var size in sizeList)
                        {
                            var expected = new byte[size];
                            System.Buffer.BlockCopy(data, offset, expected, 0, size);

                            var actual = htclowChannel.Receive(size);

                            CollectionAssert.AreEqual(expected, actual);

                            offset += size;
                        }
                    }
                }
            }
        }

        #endregion
    }
}
