﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;

using Microsoft.VisualStudio.TestTools.UnitTesting;

using System.IO;
using System.Text;

using Nintendo.HtcTools.HtcfsLibrary;
using Nintendo.HtcTools.HtcToolsTestCommonLibrary;

namespace Nintendo.HtcTools.HtcfsLibraryTest
{
    internal struct CompoundResult
    {
        public HtcfsResult HtcfsResult { get; }
        public int NativeResult { get; }

        public CompoundResult(HtcfsResult htcfsResult, int nativeResult)
        {
            HtcfsResult = htcfsResult;
            NativeResult = nativeResult;
        }

        public bool IsSuccess()
        {
            return HtcfsResult == HtcfsResult.Success && NativeResult == 0;
        }
    }

    /// <summary>
    /// テスト用に HtcfsService や NetworkEmulationStream を提供するサービス
    /// </summary>
    internal class HtcfsLibraryTestService : IDisposable
    {
        private class ChannelHolderForTest
        {
            private readonly NetworkEmulationStream stream;

            public ChannelHolderForTest(NetworkEmulationStream stream)
            {
                this.stream = stream;
            }

            public void Connect()
            {
            }

            public byte[] Receive(int count)
            {
                return stream.ReadFromTest(count);
            }

            public int Receive(byte[] buffer, int offset, int count)
            {
                var data = stream.ReadFromTest(count);
                if (data.Length != count)
                {
                    throw new NotImplementedException();
                }

                Buffer.BlockCopy(data, 0, buffer, offset, count);
                return count;
            }

            public Header ReceiveHeader()
            {
                return new Header(Receive(Header.Size));
            }

            public int ReceiveInt32()
            {
                return BitConverter.ToInt32(stream.ReadFromTest(sizeof(int)), 0);
            }

            public long ReceiveInt64()
            {
                return BitConverter.ToInt64(stream.ReadFromTest(sizeof(long)), 0);
            }

            public void Send(byte[] buffer)
            {
                stream.WriteFromTest(buffer);
            }

            public void Send(byte[] buffer, int offset, int count)
            {
                if (offset != 0 || count != buffer.Length)
                {
                    throw new NotImplementedException();
                }
                Send(buffer);
            }

            public void Close()
            {
                stream.Close();
            }
        }

        public string TestRootPath { get; private set; }

        private readonly TestHeaderFactory factory = new TestHeaderFactory();
        private readonly NetworkEmulationStream stream = new NetworkEmulationStream();

        private readonly ChannelHolderForTest channel;
        private readonly HtcfsService htcfsService;

        private const short InvalidVersion = -1;
        private const int InvalidHandle = -1;
        private const int InvalidCount = -1;
        private const long InvalidFileSize = -1;

        public HtcfsLibraryTestService()
        {
            TestRootPath = $"{Path.GetTempPath()}/Nintendo/HtcfsLibraryTest";

            var info = new DirectoryInfo(TestRootPath);
            if (info.Exists)
            {
                Directory.Delete(TestRootPath, true);
            }
            Directory.CreateDirectory(TestRootPath);

            channel = new ChannelHolderForTest(stream);
            htcfsService = new HtcfsService(stream);
        }

        #region Dispose
        private bool disposed = false;

        public void Dispose()
        {
            if (disposed)
            {
                return;
            }

            htcfsService.Dispose();
            stream.Close();

            disposed = true;
            GC.SuppressFinalize(this);
        }
        #endregion

        public HtcfsResult GetMaxProtocolVersion(out short version)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.GetMaxProtocolVersion, 0);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.GetMaxProtocolVersion, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = (HtcfsResult)header.Param0;

            if (result == HtcfsResult.Success)
            {
                version = (short)header.Param1;
            }
            else
            {
                version = InvalidVersion;
            }

            return result;
        }

        public HtcfsResult SetProtocolVersion(short version)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.SetProtocolVersion, 0, version);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.SetProtocolVersion, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = (HtcfsResult)header.Param0;

            return result;
        }

        public CompoundResult OpenFile(out int handle, string path, OpenMode mode)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.OpenFile, path.Length, (long)mode);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.OpenFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                handle = (int)header.Param2;
            }
            else
            {
                handle = InvalidHandle;
            }

            return result;
        }

        public CompoundResult CloseFile(int handle)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.CloseFile, 0, handle);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.CloseFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, 0);

            return result;
        }

        public CompoundResult CreateFile(string path, long fileSize)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.CreateFile, path.Length, fileSize);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.CreateFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult DeleteFile(string path)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.DeleteFile, path.Length);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.DeleteFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult ReadFile(out byte[] buffer, int handle, long offset, long readSize)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.ReadFile, 0, handle, offset, readSize);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.ReadFile, header.PacketType);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                Assert.IsTrue(0 <= header.BodySize);

                if (header.BodySize > int.MaxValue)
                {
                    throw new Exception("2GB 以上の read に対応していません。");
                }

                buffer = channel.Receive((int)header.BodySize);
            }
            else
            {
                Assert.AreEqual(0, header.BodySize);

                buffer = new byte[0];
            }

            return result;
        }

        public CompoundResult WriteFile(int handle, byte[] buffer, long offset, WriteOption option)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.WriteFile, buffer.Length, handle, (long)option, offset);
            channel.Send(requestHeader.GetBytes());
            channel.Send(buffer);

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.WriteFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult FlushFile(int handle)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.FlushFile, 0, handle);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.FlushFile, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult GetFileSize(out long fileSize, int handle)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.GetFileSize, 0, handle);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.GetFileSize, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                fileSize = header.Param1;
            }
            else
            {
                fileSize = InvalidFileSize;
            }

            return result;
        }

        public CompoundResult OpenDirectory(out int handle, string path, OpenDirectoryMode mode)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.OpenDirectory, path.Length, (long)mode);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.OpenDirectory, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                handle = (int)header.Param2;
            }
            else
            {
                handle = InvalidHandle;
            }

            return result;
        }

        public CompoundResult CloseDirectory(int handle)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.CloseDirectory, 0, handle);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.CloseDirectory, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, 0);

            return result;
        }

        public CompoundResult CreateDirectory(string path)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.CreateDirectory, path.Length);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.CreateDirectory, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult DeleteDirectory(string path, bool recursively)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.DeleteDirectory, path.Length, recursively ? 1 : 0);
            channel.Send(requestHeader.GetBytes());
            channel.Send(Encoding.UTF8.GetBytes(path));

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.DeleteDirectory, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            return result;
        }

        public CompoundResult ReadDirectory(out DirectoryEntry[] entries, int handle, long count)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.ReadDirectory, 0, handle, count);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.ReadDirectory, header.PacketType);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                var entryCount = header.Param2;

                entries = new DirectoryEntry[entryCount];

                var directoryEntrySize = HtcfsNativeLibrary.Directory.GetDirectoryEntrySize();
                Assert.AreEqual(directoryEntrySize * entryCount, header.BodySize);

                for (long i = 0; i < entryCount; i++)
                {
                    var nameBuffer = channel.Receive(772);
                    var entryType = (DirectoryEntryType)channel.ReceiveInt32();
                    var size = channel.ReceiveInt64();

                    var nameSize = Array.IndexOf<byte>(nameBuffer, 0);
                    Assert.IsTrue(0 <= nameSize && nameSize <= 768);

                    var name = Encoding.UTF8.GetString(nameBuffer, 0, nameSize);

                    entries[i] = new DirectoryEntry()
                    {
                        EntryType = entryType,
                        Size = size,
                        Name = name,
                    };
                }
            }
            else
            {
                entries = new DirectoryEntry[0];
                Assert.AreEqual(0, header.BodySize);
            }

            return result;
        }

        public CompoundResult GetEntryCount(out long count, int handle)
        {
            // リクエスト送信
            var requestHeader = factory.MakeRequestHeader(PacketType.GetEntryCount, 0, handle);
            channel.Send(requestHeader.GetBytes());

            // レスポンス受信
            var header = channel.ReceiveHeader();

            Assert.AreEqual(Constants.Protocol, header.Protocol);
            Assert.AreEqual(PacketCategory.Response, header.PacketCategory);
            Assert.AreEqual(PacketType.GetEntryCount, header.PacketType);
            Assert.AreEqual(0, header.BodySize);

            var result = new CompoundResult((HtcfsResult)header.Param0, (int)header.Param1);

            if (result.IsSuccess())
            {
                count = header.Param2;
            }
            else
            {
                count = InvalidCount;
            }

            return result;
        }
    }
}
