﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading;

using System.IO;

namespace Nintendo.HtcTools.HtcToolsTestCommonLibrary
{
    /// <summary>
    /// テストに用いる NetworkStream の代替クラス。
    /// </summary>
    public class NetworkEmulationStream : Stream
    {
        private CancellationTokenSource m_CancelForDispose = new CancellationTokenSource();

        private ManualResetEventSlim m_ReceiveBufferAppended = new ManualResetEventSlim();
        private ManualResetEventSlim m_SendBufferAppended = new ManualResetEventSlim();

        private byte[] m_ReceiveBuffer = new byte[0];
        private byte[] m_SendBuffer = new byte[0];

        private int m_ReceiveBufferHead = 0;
        private int m_SendBufferHead = 0;

        private object m_ReceiveBufferLockGuard = new object();
        private object m_SendBufferLockGuard = new object();

        private bool m_RemoveOldData = false;

        public NetworkEmulationStream(bool removeOldData = false)
        {
            m_RemoveOldData = removeOldData;
        }

        public override int Read(byte[] buffer, int offset, int count)
        {
            if (offset < 0 || count < 0)
            {
                throw new ArgumentOutOfRangeException();
            }

            // 受信バッファを読み取り
            while (true)
            {
                m_ReceiveBufferAppended.Reset();

                lock (m_ReceiveBufferLockGuard)
                {
                    if (m_ReceiveBuffer.Length - m_ReceiveBufferHead >= count)
                    {
                        System.Buffer.BlockCopy(m_ReceiveBuffer, m_ReceiveBufferHead, buffer, offset, count);
                        m_ReceiveBufferHead += count;

                        if (m_RemoveOldData)
                        {
                            var remainedSize = m_ReceiveBuffer.Length - m_ReceiveBufferHead;
                            var remained = new byte[remainedSize];
                            System.Buffer.BlockCopy(m_ReceiveBuffer, m_ReceiveBufferHead, remained, 0, remainedSize);

                            m_ReceiveBuffer = remained;
                            m_ReceiveBufferHead = 0;
                        }

                        return count;
                    }
                }

                try
                {
                    m_ReceiveBufferAppended.Wait(m_CancelForDispose.Token);
                }
                catch (OperationCanceledException e)
                {
                    throw new ObjectDisposedException("NetworkEmulationStream disposed.", e);
                }
            }
        }

        public override void Write(byte[] buffer, int offset, int count)
        {
            // 送信バッファに追加
            lock (m_SendBufferLockGuard)
            {
                var newBuffer = new byte[m_SendBuffer.Length + count];

                System.Buffer.BlockCopy(m_SendBuffer, 0, newBuffer, 0, m_SendBuffer.Length);
                System.Buffer.BlockCopy(buffer, 0, newBuffer, m_SendBuffer.Length, count);

                m_SendBuffer = newBuffer;
            }

            m_SendBufferAppended.Set();
        }

        public byte[] ReadFromTest(int count)
        {
            // 送信バッファを読み取り
            while (true)
            {
                m_SendBufferAppended.Reset();

                lock (m_SendBufferLockGuard)
                {
                    if (m_SendBuffer.Length - m_SendBufferHead >= count)
                    {
                        var buffer = new byte[count];

                        System.Buffer.BlockCopy(m_SendBuffer, m_SendBufferHead, buffer, 0, count);
                        m_SendBufferHead += count;

                        if (m_RemoveOldData)
                        {
                            var remainedSize = m_SendBuffer.Length - m_SendBufferHead;
                            var remained = new byte[remainedSize];
                            System.Buffer.BlockCopy(m_SendBuffer, m_SendBufferHead, remained, 0, remainedSize);

                            m_SendBuffer = remained;
                            m_SendBufferHead = 0;
                        }

                        return buffer;
                    }
                }

                try
                {
                    m_SendBufferAppended.Wait(m_CancelForDispose.Token);
                }
                catch (OperationCanceledException e)
                {
                    throw new ObjectDisposedException("NetworkEmulationStream disposed.", e);
                }
            }
        }

        public void WriteFromTest(byte[] buffer)
        {
            // 受信バッファに追加
            lock (m_ReceiveBufferLockGuard)
            {
                var newBuffer = new byte[m_ReceiveBuffer.Length + buffer.Length];

                System.Buffer.BlockCopy(m_ReceiveBuffer, 0, newBuffer, 0, m_ReceiveBuffer.Length);
                System.Buffer.BlockCopy(buffer, 0, newBuffer, m_ReceiveBuffer.Length, buffer.Length);

                m_ReceiveBuffer = newBuffer;
            }

            m_ReceiveBufferAppended.Set();
        }

        public override bool CanRead
        {
            get
            {
                return true;
            }
        }

        public override bool CanSeek
        {
            get
            {
                return false;
            }
        }

        public override bool CanWrite
        {
            get
            {
                return true;
            }
        }

        public override long Length
        {
            get
            {
                throw new NotSupportedException();
            }
        }

        public override long Position
        {
            get
            {
                throw new NotSupportedException();
            }

            set
            {
                throw new NotSupportedException();
            }
        }

        public override void Flush()
        {
        }

        public override long Seek(long offset, SeekOrigin origin)
        {
            throw new NotSupportedException();
        }

        public override void SetLength(long value)
        {
            throw new NotSupportedException();
        }

        #region Dispose
        private bool m_Disposed = false;

        protected override void Dispose(bool disposing)
        {
            if (m_Disposed)
            {
                return;
            }

            if (disposing)
            {
                m_CancelForDispose.Cancel();

                m_SendBufferAppended.Dispose();
                m_ReceiveBufferAppended.Dispose();
                m_CancelForDispose.Dispose();
            }

            base.Dispose(disposing);

            m_Disposed = true;
        }
        #endregion
    }
}
