﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

namespace Nintendo.Hid
{
    using System;
    using System.IO;

    public abstract class BaseSubModule
    {
        internal BaseSubModule(byte id)
        {
            this.ModuleId = id;
            this.IsMonitored = false;
        }

        internal abstract byte ModuleTypeValue { get; }

        internal abstract byte SubModuleTypeValue { get; }

        internal byte ModuleId { get; private set; }

        internal bool IsMonitored { get; set; }

        protected abstract byte StateCountMax { get; }

        protected abstract bool IsAutoInputChanged { get; }

        protected abstract bool IsAutoInputSet { get; }

        internal abstract void Reset();

        internal abstract void Update();

        internal abstract void AcceptInputState(NetworkBinaryReader reader);

        internal byte[] GenerateAutoInputCommand()
        {
            if (!this.IsAutoInputChanged)
            {
                return new byte[0];
            }

            if (this.IsAutoInputSet)
            {
                return this.GenerateSetAutoInputCommand();
            }
            else
            {
                return this.GenerateResetAutoInputCommand();
            }
        }

        internal byte[] GenerateRequestCommand()
        {
            if (!this.IsMonitored)
            {
                return new byte[0];
            }

            var header = this.GenerateModuleStateHeader();
            var body = new byte[] { 0, 0, 0, this.StateCountMax };
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                writer.Write((ushort)(4 + header.Length + body.Length));
                writer.Write((byte)0);
                writer.Write((byte)CommandType.RequestInputState);
                writer.Write((byte[])header);
                writer.Write((byte[])body);
                return stream.ToArray();
            }
        }

        protected abstract byte[] GenerateModuleState();

        private byte[] GenerateSetAutoInputCommand()
        {
            var header = this.GenerateModuleStateHeader();
            var state = this.GenerateModuleState();
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                writer.Write((ushort)(4 + header.Length + state.Length));
                writer.Write((byte)0);
                writer.Write((byte)CommandType.SetAutoInput);
                writer.Write((byte[])header);
                writer.Write((byte[])state);
                return stream.ToArray();
            }
        }

        private byte[] GenerateResetAutoInputCommand()
        {
            var header = this.GenerateModuleStateHeader();
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                writer.Write((ushort)(4 + header.Length));
                writer.Write((byte)0);
                writer.Write((byte)CommandType.ResetAutoInput);
                writer.Write((byte[])header);
                return stream.ToArray();
            }
        }

        private byte[] GenerateModuleStateHeader()
        {
            using (var stream = new MemoryStream())
            using (var writer = new NetworkBinaryWriter(stream))
            {
                writer.Write((byte)this.ModuleTypeValue);
                writer.Write((byte)this.ModuleId);
                writer.Write((byte)this.SubModuleTypeValue);
                writer.Write((byte)0);
                return stream.ToArray();
            }
        }
    }
}
