﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogChecker;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest.FsAccessLogCheckerTest
{
    [TestClass]
    public class CommandLineParserTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestHelpAndDefaultValue()
        {
            string[] args = new string[] { "--help" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsFalse(parser.Parse(args));
            Assert.IsNull(parser.Settings.LogFile);
            Assert.IsNull(parser.Settings.Output);
            Assert.IsNull(parser.Settings.TimeRange);
            Assert.IsFalse(parser.Settings.Verbose);
            Assert.IsFalse(parser.Settings.Quiet);
            Assert.IsFalse(parser.Settings.Force);
            Assert.IsFalse(parser.Settings.AlsoCheckHost);
            Assert.IsFalse(parser.Settings.Version);
            Assert.AreEqual(100, parser.Settings.WarningRatio);
            Assert.IsTrue(parser.Settings.Help);
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestNoInputFile()
        {
            string[] args = new string[] { };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        public void TestTimeRange()
        {
            string[] args = new string[] { "input_file", "--time_range", "100" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
            Assert.AreEqual(100, parser.Settings.TimeRange);
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidTimeRangeLess()
        {
            long lessMinutes = FsGuideline.MinStepLogTimeMinutes - 1;
            string[] args = new string[] { "input_file", "--time_range", lessMinutes.ToString() };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidTimeRangeGreater()
        {
            long greaterMinutes = FsGuideline.MaxStepLogTimeMinutes + 1;
            string[] args = new string[] { "input_file", "--time_range", greaterMinutes.ToString() };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidTimeRangeDecimal()
        {
            string[] args = new string[] { "input_file", "--time_range", "100.5" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidTimeRangeNoArg()
        {
            string[] args = new string[] { "input_file", "--time_range" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        public void TestInvalidWarningRatio()
        {
            string[] args = new string[] { "input_file", "--warning_ratio", "0" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
            Assert.AreEqual(0, parser.Settings.WarningRatio);
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidWarningRatioLess()
        {
            string[] args = new string[] { "input_file", "--warning_ratio", "-1" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidWarningRatioGreater()
        {
            string[] args = new string[] { "input_file", "--warning_ratio", "101" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidWarningRatioDecimal()
        {
            string[] args = new string[] { "input_file", "--warning_ratio", "100.5" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }

        [TestMethod]
        [ExpectedException(typeof(CommandLineParseErrorException))]
        public void TestInvalidWarningRatioNoArg()
        {
            string[] args = new string[] { "input_file", "--warning_ratio" };
            CommandLineOption parser = new CommandLineOption();
            Assert.IsTrue(parser.Parse(args));
        }
    }
}
