﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    [TestClass]
    public class FsAccessLogUnitTest
    {
        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestGetTotalMilliseconds()
        {
            List<FsAccessLog> list = new List<FsAccessLog>();
            // リストが空の場合は 0
            Assert.AreEqual(0, FsAccessLogAnalyzer.GetTotalMilliseconds(list));

            // start:0,end:0 の2つのログの経過時間は 0
            FsAccessLog a1 = new FsAccessLog();
            FsAccessLog a2 = new FsAccessLog();
            list.Add(a1);
            list.Add(a2);
            Assert.AreEqual(0, FsAccessLogAnalyzer.GetTotalMilliseconds(list));
        }

        [TestMethod]
        public void TestGetElapsedMilliseconds()
        {
            // start:0,end:0 の2つのログの経過時間は 0
            FsAccessLog a1 = new FsAccessLog();
            FsAccessLog a2 = new FsAccessLog();
            Assert.AreEqual(0, FsAccessLogAnalyzer.GetStartPointElapsedMilliseconds(a1, a2));
        }

        [TestMethod]
        public void TestUserId()
        {
            UserId invalidUserId = UserId.InvalidUserId();
            Assert.AreEqual(new UserId(0, 0), invalidUserId);

            UserId userId = (UserId)("0xFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF");
            Assert.AreEqual(new UserId(0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF), userId);
            Assert.AreEqual("FFFFFFFFFFFFFFFFFFFFFFFFFFFFFFFF", userId.ToString());

            UserId userId2 = (UserId)("ABCDEF0123456789ABCDEF0123456789");
            Assert.AreEqual(new UserId(0xABCDEF0123456789, 0xABCDEF0123456789), userId2);
            Assert.AreEqual("ABCDEF0123456789ABCDEF0123456789", userId2.ToString());
        }
    }

    [TestClass]
    public class FsAccessLogListTest
    {
        private FsAccessLogList CreateList(long offset)
        {
            FsAccessLogList list = new FsAccessLogList();
            for (long i = 0; i < 3; ++i)
            {
                for (long j = 0; j < 16; ++j)
                {
                    FsAccessLog log = new FsAccessLog();
                    log.Start = i * 60 * 1000 + offset;
                    log.End = i * 60 * 1000 + offset;
                    list.Add(log);
                }
            }
            return list;
        }

        [TestInitialize]
        public void TestInitialize()
        {
            FsAccessLogCheckerTestUtility.Initialize();
        }

        [TestMethod]
        public void TestGetTotalMilliseconds()
        {
            {
                FsAccessLogList list = new FsAccessLogList();
                Assert.AreEqual(0, list.GetTotalMilliseconds());
            }
            {
                var list = CreateList(1);
                Assert.AreEqual(2 * 60 * 1000, list.GetTotalMilliseconds());

                var log = new FsAccessLog();
                log.Start = 0;
                log.End = 3 * 60 * 1000;
                list.Add(log);
                Assert.AreEqual(3 * 60 * 1000, list.GetTotalMilliseconds());

                list.Clear();
                Assert.AreEqual(0, list.GetTotalMilliseconds());

                list.Add(log);
                Assert.AreEqual(3 * 60 * 1000, list.GetTotalMilliseconds());

                list.Add(log);
                Assert.AreEqual(3 * 60 * 1000, list.GetTotalMilliseconds());
            }
        }

        [TestMethod]
        public void TestGetDistanceMilliseconds()
        {
            var log = new FsAccessLog();
            log.Start = 3 * 60 * 1000;
            log.End = 3 * 60 * 1000;

            {
                FsAccessLogList list = new FsAccessLogList();
                Assert.AreEqual(0, list.GetDistanceMilliseconds(log));
            }
            {
                var list = CreateList(0);
                Assert.AreEqual(1 * 60 * 1000, list.GetDistanceMilliseconds(log));

                list.Clear();
                Assert.AreEqual(0, list.GetDistanceMilliseconds(log));
            }
        }

        [TestMethod]
        public void TestSubList()
        {
            // 0 を指定した場合、同一リストが得られる
            {
                var list = CreateList(0);
                list.SubList(0);
                Assert.IsTrue(list.Any());
                Assert.AreEqual(3 * 16, list.Count);
            }
            // 負数を指定した場合、同一リストが得られる
            {
                var list = CreateList(0);
                list.SubList(-1);
                Assert.IsTrue(list.Any());
                Assert.AreEqual(3 * 16, list.Count);
            }
            // 大きな値を指定した場合、空
            {
                var list = CreateList(0);
                list.SubList(10 * 60 * 1000);
                Assert.IsFalse(list.Any());
                Assert.AreEqual(0, list.Count);
            }
            // 末尾を指定した場合
            {
                var list = CreateList(0);
                list.SubList(2 * 60 * 1000);
                Assert.IsTrue(list.Any());
                Assert.AreEqual(16, list.Count);
            }
            // 中間
            {
                var list = CreateList(0);
                list.SubList(1 * 60 * 1000);
                Assert.IsTrue(list.Any());
                Assert.AreEqual(2 * 16, list.Count);
            }
        }
    }
}
