﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using Nintendo.FsAccessLogAnalysis;

namespace FsAccessLogCheckerTest
{
    internal class TestStream : MemoryStream
    {
        public TestStream()
        {
            Writer = new StreamWriter(this);
            ProcessId = string.Empty;
        }

        public TestStream(string processId)
        {
            Writer = new StreamWriter(this);
            ProcessId = "(" + processId + ") ";
        }

        public string ToAccessLogString(FsAccessLog log)
        {
            string logString = ProcessId;
            logString += "FS_ACCESS: {";
            logString += " start: " + log.Start;
            logString += ", end: " + log.End;
            logString += ", result: " + log.Result;
            logString += ", handle: " + (log.Handle ?? 0);
            logString += ", priority: " + (log.Priority ?? "Normal");
            logString += ", function: \"" + log.Function + "\"";
            logString += ", offset: " + log.Offset;
            logString += ", size: " + log.Size;
            if (log.ClassName != null)
            {
                logString += ", class_name: \"" + log.ClassName + "\"";
            }
            if (log.NameSpace != null)
            {
                logString += ", name_space: \"" + log.NameSpace + "\"";
            }
            if (log.Name != null)
            {
                logString += ", name: \"" + log.Name + "\"";
            }
            if (log.Path != null)
            {
                logString += ", path: \"" + log.Path + "\"";
            }
            if (log.NewPath != null)
            {
                logString += ", new_path: \"" + log.NewPath + "\"";
            }
            if (log.UserId != null)
            {
                logString += ", userid: " + log.UserId;
            }
            if (log.DataId != null)
            {
                logString += ", dataid: " + log.DataId;
            }
            if (log.Index != null)
            {
                logString += ", index: " + log.Index;
            }
            if (log.WriteOption.HasFlag(FsFunction.WriteOptionFlag.Flush))
            {
                logString += ", write_option: Flush";
            }
            if (log.ApplicationId != null)
            {
                logString += ", applicationid: " + log.ApplicationId.Value;
            }
            if (log.BisPartitionId != null)
            {
                logString += ", bispartitionid: " + log.BisPartitionId;
            }
            if (log.SaveDataId != null)
            {
                logString += ", savedataid: " + log.SaveDataId;
            }
            if (log.SaveDataSpaceId != null)
            {
                logString += ", savedataspaceid: " + log.SaveDataSpaceId;
            }
            if (log.ContentStorageId != null)
            {
                logString += ", contentstorageid: " + log.ContentStorageId;
            }
            if (log.ImageDirectoryId != null)
            {
                logString += ", imagedirectoryid: " + log.ImageDirectoryId;
            }
            if (log.ImageDataSize.HasValue)
            {
                logString += ", image_data_size: " + log.ImageDataSize.Value;
            }
            if (log.Width.HasValue)
            {
                logString += ", width: " + log.Width.Value;
            }
            if (log.Height.HasValue)
            {
                logString += ", height: " + log.Height.Value;
            }
            if (log.AlbumReportOption != null)
            {
                logString += ", album_report_option: " + log.AlbumReportOption;
            }
            if (log.SaveDataSize.HasValue)
            {
                logString += ", save_data_size: " + log.SaveDataSize.Value;
            }
            if (log.SaveDataJournalSize.HasValue)
            {
                logString += ", save_data_journal_size: " + log.SaveDataJournalSize.Value;
            }
            logString += "}";
            return logString;
        }

        public void Write(string log)
        {
            Writer.WriteLine(log);
        }

        public void WriteLog(FsAccessLog log)
        {
            string logString = ToAccessLogString(log);
            Writer.WriteLine(logString);
        }

        public void StartLog()
        {
            StartLog("0.16.17", false);
        }

        public void StartLogForSystem()
        {
            StartLog("0.16.17", true);
        }

        public void StartLog(string sdk_version)
        {
            StartLog(sdk_version, false);
        }

        public void StartLog(string sdk_version, bool forSystem)
        {
            if (forSystem)
            {
                Write(ProcessId + "FS_ACCESS: { sdk_version: " + sdk_version + ", spec: NX, for_system: true }");
            }
            else
            {
                Write(ProcessId + "FS_ACCESS: { sdk_version: " + sdk_version + ", spec: NX }");
            }
        }

        public void CreateCacheStorage(long start, long end, int index, long saveDataSize, long saveDataJournalSize)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "CreateCacheStorage";
            log.Result = 0;
            log.SaveDataSize = saveDataSize;
            log.SaveDataJournalSize = saveDataJournalSize;
            log.Index = index;
            WriteLog(log);
        }

        public void DeleteCacheStorage(long start, long end, int index)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "DeleteCacheStorage";
            log.Result = 0;
            log.Index = index;
            WriteLog(log);
        }

        public void MountRom(long start, long end, string name, uint result)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountRom";
            log.Name = name;
            log.Result = result;
            WriteLog(log);
        }
        public void MountRom(long start, long end, string name)
        {
            MountRom(start, end, name, 0);
        }
        public void MountSaveData(long start, long end, string name, ulong userId)
        {
            MountSaveData(start, end, name, userId, null);
        }
        public void MountSaveData(long start, long end, string name, ulong userId, ulong? applicationId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountSaveData";
            log.Name = name;
            log.Result = 0;
            log.UserId = new UserId(0, userId);
            if (applicationId.HasValue)
            {
                log.ApplicationId = applicationId;
            }
            WriteLog(log);
        }
        public void MountSaveDataReadOnly(long start, long end, string name, ulong userId, ulong applicationId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountSaveDataReadOnly";
            log.Name = name;
            log.Result = 0;
            log.UserId = new UserId(0, userId);
            log.ApplicationId = applicationId;
            WriteLog(log);
        }
        public void MountSystemSaveData(long start, long end, string name, FsFunction.SaveDataSpaceId spaceId, ulong savedataId, ulong userId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountSystemSaveData";
            log.Name = name;
            log.Result = 0;
            log.SaveDataSpaceId = spaceId;
            log.SaveDataId = savedataId;
            log.UserId = new UserId(0, userId);
            WriteLog(log);
        }
        public void MountDeviceSaveData(long start, long end, string name, ulong applicationId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountDeviceSaveData";
            log.Name = name;
            log.Result = 0;
            log.ApplicationId = applicationId;
            WriteLog(log);
        }
        public void MountAddOnContent(long start, long end, string name, int index)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountAddOnContent";
            log.Name = name;
            log.Result = 0;
            log.Index = index;
            WriteLog(log);
        }
        public void MountBcatSaveData(long start, long end, string name, ulong id)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountBcatSaveData";
            log.Name = name;
            log.Result = 0;
            log.ApplicationId = id;
            WriteLog(log);
        }
        public void MountBis(long start, long end, string name, FsFunction.BisPartitionId id)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountBis";
            log.Name = name;
            log.Result = 0;
            log.BisPartitionId = id;
            WriteLog(log);
        }
        public void MountContentStorage(long start, long end, string name, FsFunction.ContentStorageId id)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountContentStorage";
            log.Name = name;
            log.Result = 0;
            log.ContentStorageId = id;
            WriteLog(log);
        }
        public void MountImageDirectory(uint result, long start, long end, string name, FsFunction.ImageDirectoryId id)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountImageDirectory";
            log.Name = name;
            log.Result = result;
            log.ImageDirectoryId = id;
            WriteLog(log);
        }
        public void MountCacheStorage(long start, long end, string name)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountCacheStorage";
            log.Name = name;
            log.Result = 0;
            WriteLog(log);
        }

        public void MountCacheStorage(long start, long end, string name, int index)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountCacheStorage";
            log.Name = name;
            log.Result = 0;
            log.Index = index;
            WriteLog(log);
        }
        public void MountCacheStorage(long start, long end, string name, ulong applicationId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountCacheStorage";
            log.Name = name;
            log.Result = 0;
            log.ApplicationId = applicationId;
            WriteLog(log);
        }

        public void MountDeliveryCacheStorage(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "MountDeliveryCacheStorage";
            log.Result = 0;
            WriteLog(log);
        }
        public void UnmountDeliveryCacheStorage(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "UnmountDeliveryCacheStorage";
            log.Result = 0;
            WriteLog(log);
        }

        public void Unmount(long start, long end, string name)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Unmount";
            log.Name = name;
            log.Result = 0;
            WriteLog(log);
        }

        public void Function(long start, long end, string function)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = function;
            log.Result = 0;
            WriteLog(log);
        }

        public void CreateFile(long start, long end, string path, ulong size)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "CreateFile";
            log.Path = path;
            log.Size = size;
            log.Result = 0;
            WriteLog(log);
        }

        public void OpenFile(long start, long end, ulong handle, string path)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Handle = handle;
            log.Function = "OpenFile";
            log.Path = path;
            log.Result = 0;
            WriteLog(log);
        }

        public void WriteFile(long start, long end, ulong handle, ulong offset, ulong size, uint result)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "WriteFile";
            log.Handle = handle;
            log.Offset = offset;
            log.Size = size;
            log.Result = result;
            WriteLog(log);
        }

        public void WriteFile(long start, long end, ulong handle, ulong offset, ulong size)
        {
            WriteFile(start, end, handle, offset, size, 0);
        }

        public void ReadFile(long start, long end, ulong handle, ulong offset, ulong size, uint result)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "ReadFile";
            log.Handle = handle;
            log.Offset = offset;
            log.Size = size;
            log.Result = result;
            WriteLog(log);
        }

        public void ReadFile(long start, long end, ulong handle, ulong offset, ulong size)
        {
            ReadFile(start, end, handle, offset, size, 0);
        }

        public void SetFileSize(long start, long end, ulong handle, ulong size)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "SetFileSize";
            log.Handle = handle;
            log.Size = size;
            log.Result = 0;
            WriteLog(log);
        }

        public void GetSaveDataSize(long start, long end, ulong userId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "GetSaveDataSize";
            log.Result = 0;
            log.UserId = new UserId(0, userId);
            WriteLog(log);
        }

        public void ExtendSaveData(long start, long end, long saveDataSize, long saveDataJournalSize, ulong userId)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "ExtendSaveData";
            log.Result = 0;
            log.SaveDataSize = saveDataSize;
            log.SaveDataJournalSize = saveDataJournalSize;
            log.UserId = new UserId(0, userId);
            WriteLog(log);
        }

        public void GetCacheStorageSize(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "GetCacheStorageSize";
            log.Result = 0;
            WriteLog(log);
        }

        public void GetCacheStorageSize(long start, long end, int index)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "GetCacheStorageSize";
            log.Result = 0;
            log.Index = index;
            WriteLog(log);
        }

        public void Commit(long start, long end, string name)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Commit";
            log.Result = 0;
            log.Name = name;
            WriteLog(log);
        }

        public void CommitSaveData(long start, long end, string name)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "CommitSaveData";
            log.Result = 0;
            log.Name = name;
            WriteLog(log);
        }

        public void QueryApplicationPlayStatistics(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "QueryApplicationPlayStatistics";
            log.Result = 0;
            WriteLog(log);
        }

        public void QueryApplicationPlayStatisticsForSystem(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "QueryApplicationPlayStatisticsForSystem";
            log.Result = 0;
            WriteLog(log);
        }

        public void SaveScreenshot(long start, long end, ulong dataSize, uint reportOption)
        {
            SaveScreenshot(start, end, dataSize, reportOption, 1280, 720);
        }
        public void SaveScreenshot(long start, long end, ulong dataSize, uint reportOption, int width, int height)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "SaveScreenshot";
            log.Result = 0;
            log.ImageDataSize = dataSize;
            log.AlbumReportOption = reportOption;
            log.Width = width;
            log.Height = height;
            WriteLog(log);
        }

        public void SaveAndShareScreenshot(long start, long end, ulong dataSize, uint reportOption, ulong userId)
        {
            SaveAndShareScreenshot(start, end, dataSize, reportOption, userId, 1280, 720);
        }
        public void SaveAndShareScreenshot(long start, long end, ulong dataSize, uint reportOption, ulong userId, int width, int height)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "SaveAndShareScreenshot";
            log.Result = 0;
            log.ImageDataSize = dataSize;
            log.AlbumReportOption = reportOption;
            log.Width = width;
            log.Height = height;
            log.UserId = new UserId(0, userId);
            WriteLog(log);
        }

        public void EnumerateDeliveryCacheDirectory(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "EnumerateDeliveryCacheDirectory";
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheDirectoryOpen(long start, long end, ulong handle, string path)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Open";
            log.Path = path;
            log.ClassName = "DeliveryCacheDirectory";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheDirectoryClose(long start, long end, ulong handle)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Close";
            log.ClassName = "DeliveryCacheDirectory";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheDirectoryDestructor(long start, long end, ulong handle)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "~DeliveryCacheDirectory";
            log.ClassName = "DeliveryCacheDirectory";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheDirectoryRead(long start, long end, ulong handle)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Read";
            log.ClassName = "DeliveryCacheDirectory";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheFileOpen(long start, long end, ulong handle, string path)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Open";
            log.Path = path;
            log.ClassName = "DeliveryCacheFile";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheFileClose(long start, long end, ulong handle)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Close";
            log.ClassName = "DeliveryCacheFile";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheFileDestructor(long start, long end, ulong handle)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "~DeliveryCacheFile";
            log.ClassName = "DeliveryCacheFile";
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void DeliveryCacheFileRead(long start, long end, ulong handle, ulong offset, ulong size)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Function = "Read";
            log.ClassName = "DeliveryCacheFile";
            log.Offset = offset;
            log.Size = size;
            log.Handle = handle;
            log.Result = 0;
            WriteLog(log);
        }

        public void NfpMount(long start, long end, int model_type)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Result = 0;
            log.Function = "Mount";
            log.NameSpace = "nfp";
            log.ModelType = model_type;
            WriteLog(log);
        }

        public void NfpCreateApplicationArea(long start, long end, long access_id, int data_size)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Result = 0;
            log.Function = "CreateApplicationArea";
            log.NameSpace = "nfp";
            log.AccessId = access_id;
            log.DataSize = data_size;
            WriteLog(log);
        }

        public void NfpFlush(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Result = 0;
            log.Function = "Flush";
            log.NameSpace = "nfp";
            WriteLog(log);
        }

        public void NfpRestore(long start, long end)
        {
            FsAccessLog log = new FsAccessLog();
            log.Start = start;
            log.End = end;
            log.Result = 0;
            log.Function = "Restore";
            log.NameSpace = "nfp";
            WriteLog(log);
        }

        public void DumpFile(string path)
        {
            using (var stream = new FileStream(path, FileMode.OpenOrCreate, FileAccess.Write))
            {
                var array = this.ToArray();
                stream.Write(array, 0, array.Length);
            }
        }

        public void SetUp()
        {
            Writer.Flush();
            this.Seek(0, SeekOrigin.Begin);
        }

        public ulong GetUniqueHandle()
        {
            return UniqueHandle++;
        }
        private ulong UniqueHandle { get; set; }
        private StreamWriter Writer { get; set; }
        private string ProcessId { get; set; }
    }
}
