﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------
using System;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using ErrorMessageDatabaseConverter;
using System.Collections.Generic;
using System.IO;
using System.Linq;

namespace ErrorMessageDatabaseConverterTest
{
    using static ErrorMessageDatabaseConverter.InputXmlDataModel;
    using static ErrorMessageDatabaseConverter.SystemDataModel;

    [TestClass]
    public class FunctionTest
    {
        public TestContext TestContext { get; set; }

        public string TestDataDirectory
        {
            get
            {
                var testPath = new TestUtility.TestPath(this.TestContext);
                return testPath.GetSigloRoot() + "\\Tests\\Tools\\Sources\\Tests\\ErrorMessageDatabaseConverterTest\\TestData";
            }
        }

        [TestMethod]
        public void TestErrorCode()
        {
            var testdata = new Dictionary<ErrorCode, ErrorCode>()
            {
                { new ErrorCode("2001-0001"), new ErrorCode(2001, 1) },
                { new ErrorCode("2002-0002"), new ErrorCode(2002, 2) },
                { new ErrorCode("2003-0004"), new ErrorCode(2003, 4) },
            };
            foreach (var data in testdata)
            {
                Console.WriteLine(data.Key.Text);
                Console.WriteLine(data.Value.Text);
                Assert.AreEqual(data.Key.Text, data.Value.Text);
                Assert.AreEqual(data.Key.Category, data.Value.Category);
                Assert.AreEqual(data.Key.Number, data.Value.Number);
            }
        }

        [TestMethod]
        public void TestIsReferenceOnly()
        {
            var testdata = new Dictionary<string, bool>()
            {
                { "001-0000", false },
                { "001-comm", true },
                { "xxxx-0000", true },
            };
            foreach (var data in testdata)
            {
                var ee = new ErrorElement();
                ee.ErrorCodeText = data.Key;
                Assert.AreEqual(data.Value, ee.IsReferenceOnly);
            }
        }

        [TestMethod]
        public void TestIsReference()
        {
            var testdata = new Dictionary<string, KeyValuePair<bool, string>>()
            {
                { "001-0000",      new KeyValuePair<bool, string>(false, String.Empty) },
                { "1234-comm",      new KeyValuePair<bool, string>(false, String.Empty) },
                { "$(001-0000)",   new KeyValuePair<bool, string>(true, "001-0000") },
                { "$(1234-comm)",   new KeyValuePair<bool, string>(true, "1234-comm") },
                { "$(xxx-xxx)",     new KeyValuePair<bool, string>(true, "xxx-xxx") },
            };

            foreach(var data in testdata)
            {
                string referenceErrorCode;
                bool isReferencce = Converter.IsReferenceMessage(out referenceErrorCode, data.Key);
                Assert.AreEqual(data.Value.Key, isReferencce);
                Assert.AreEqual(data.Value.Value, referenceErrorCode);
            }
        }

        [TestMethod]
        public void TestReplaceUrlTags()
        {
            var urlListFile = Path.Combine(TestDataDirectory, "TestUrlList.xml");
            var urlList = Util.DeserializeXml<UrlListElement>(urlListFile);

            // 変換前のメッセージ、言語、変換後のメッセージ
            var testdata = new Dictionary<KeyValuePair<string, Language>, string>()
            {
                { new KeyValuePair<string, Language>("$(URL_FIRST) XXX", Language.JPja), "JPja-1 XXX" },
                { new KeyValuePair<string, Language>("$(URL_FIRST) XXX", Language.USen), "USen-1 XXX" }, // 言語差
                { new KeyValuePair<string, Language>("URL_FIRST XXX", Language.JPja), "URL_FIRST XXX" }, // $() がない
                { new KeyValuePair<string, Language>("$(URL_FIRST) $(URL_SECOND) $(URL_THIRD)", Language.JPja), "JPja-1 JPja-2 JPja-3" }, // 複数
            };

            foreach(var data in testdata)
            {
                Assert.AreEqual(data.Value, Converter.ReplaceUrlTags(data.Key.Key, data.Key.Value, urlList.UrlElements));
            }
        }

        private void AreBinaryArrayEqual(byte[] expected, byte[] actual)
        {
            int expectedLength = expected.Length;
            int actualLength = actual.Length;
            Assert.AreEqual(expectedLength, actualLength);
            for(int i = 0; i < expectedLength; i++)
            {
                Assert.AreEqual(expected[i], actual[i]);
            }

        }

        [TestMethod]
        public void TestReplaceMessageStudioTags()
        {
            // タグ単体
            AreBinaryArrayEqual(new byte[] { 0x0E, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00 },  Converter.ReplaceMessageStudioTags("<nobr>"));
            AreBinaryArrayEqual(new byte[] { 0x0F, 0x00, 0x01, 0x00, 0x04, 0x00 }, Converter.ReplaceMessageStudioTags("</nobr>"));
            AreBinaryArrayEqual(new byte[] { 0x0E, 0x00, 0x01, 0x00, 0x06, 0x00, 0x00, 0x00 }, Converter.ReplaceMessageStudioTags("<indent>"));
            AreBinaryArrayEqual(new byte[] { 0x0F, 0x00, 0x01, 0x00, 0x06, 0x00 }, Converter.ReplaceMessageStudioTags("</indent>"));
            AreBinaryArrayEqual(new byte[] { 0x0E, 0x00, 0x01, 0x00, 0x05, 0x00, 0x00, 0x00 }, Converter.ReplaceMessageStudioTags("<italic>"));
            AreBinaryArrayEqual(new byte[] { 0x0F, 0x00, 0x01, 0x00, 0x05, 0x00 }, Converter.ReplaceMessageStudioTags("</italic>"));
            AreBinaryArrayEqual(new byte[] { 0x0E, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00, 0x9A, 0x99, 0x19, 0x3F }, Converter.ReplaceMessageStudioTags("<paragraph/>"));

            // メッセージ中にタグが複数
            AreBinaryArrayEqual(new byte[] { 0x0E, 0x00, 0x01, 0x00, 0x04, 0x00, 0x00, 0x00,
                                             (byte)'A',  0x00, (byte)'B',  0x00, (byte)'C', 0x00,
                                             0x0F, 0x00, 0x01, 0x00, 0x04, 0x00 },
                                             Converter.ReplaceMessageStudioTags("<nobr>ABC</nobr>"));

        }

        private void TestMessage(string expected, byte[] actual)
        {
            var expectedAsBytes = Converter.OutputEncoding.GetBytes(expected);
            Assert.AreEqual(expectedAsBytes.Length, actual.Length);
            for(int i = 0; i < expectedAsBytes.Length; i++)
            {
                Assert.AreEqual(expectedAsBytes[i], actual[i]);
            }
        }

        [TestMethod]
        public void TestConverter()
        {
            var testXml = Path.Combine(TestDataDirectory, "TestErrorList.xml");

            var errorListElement = Util.DeserializeXml<ErrorListElement>(testXml);
            SystemData systemData;

            Converter converter = new Converter();
            converter.Convert(errorListElement.ErrorElements, null, null, null, out systemData, false);

            // 普通のエラー
            var ee = new ErrorCode("001-0000");
            var commonData0 = systemData.ErrorMessageData[ee].CommonData;

            Assert.AreEqual(3, commonData0.DialogViewButtonNum);
            Assert.AreEqual(0, commonData0.DialogViewButtonAction[0]);
            Assert.AreEqual(1, commonData0.DialogViewButtonAction[1]);
            Assert.AreEqual(2, commonData0.DialogViewButtonAction[2]);
            Assert.AreEqual(3, commonData0.FullScreenViewButtonNum);
            Assert.AreEqual(3, commonData0.FullScreenViewButtonAction[0]);
            Assert.AreEqual(4, commonData0.FullScreenViewButtonAction[1]);
            Assert.AreEqual(5, commonData0.FullScreenViewButtonAction[2]);
            Assert.AreEqual(1u, commonData0.Attribute);

            var landData0 = systemData.ErrorMessageData["001-0000"].LangData;

            TestMessage("001-0000:JPja:DlgMsg", landData0.Message[LanguageElement.MessageElementKey.DlgMsg][Language.JPja]);
            TestMessage("001-0000:JPja:DlgBtn0\n001-0000:JPja:DlgBtn1\n001-0000:JPja:DlgBtn2", landData0.Message[LanguageElement.MessageElementKey.DlgBtn][Language.JPja]);
            TestMessage("001-0000:JPja:FlvMsg", landData0.Message[LanguageElement.MessageElementKey.FlvMsg][Language.JPja]);
            TestMessage("001-0000:JPja:FlvBtn0\n001-0000:JPja:FlvBtn1\n001-0000:JPja:FlvBtn2", landData0.Message[LanguageElement.MessageElementKey.FlvBtn][Language.JPja]);

            TestMessage("001-0000:TWzh:DlgMsg", landData0.Message[LanguageElement.MessageElementKey.DlgMsg][Language.TWzh]);
            TestMessage("001-0000:TWzh:DlgBtn0\n001-0000:TWzh:DlgBtn1\n001-0000:TWzh:DlgBtn2", landData0.Message[LanguageElement.MessageElementKey.DlgBtn][Language.TWzh]);
            TestMessage("001-0000:TWzh:FlvMsg", landData0.Message[LanguageElement.MessageElementKey.FlvMsg][Language.TWzh]);
            TestMessage("001-0000:TWzh:FlvBtn0\n001-0000:TWzh:FlvBtn1\n001-0000:TWzh:FlvBtn2", landData0.Message[LanguageElement.MessageElementKey.FlvBtn][Language.TWzh]);

            // 参照を持ち、初めて参照専用エラーを参照しているエラー
            // ダイアログメッセージは XXX-XXXX を参照している。
            // それ以外は 001-0001 を参照。
            var commonData12 = systemData.ErrorMessageData["001-0012"].CommonData;
            Assert.AreEqual<UInt32>(0, commonData12.DialogViewMessageReferenceCategory);
            var landData12 = systemData.ErrorMessageData["001-0012"].LangData;
            TestMessage("XXX-XXXX:JPja:DlgMsg", landData12.Message[LanguageElement.MessageElementKey.DlgMsg][Language.JPja]);

            var ec_1_1 = new ErrorCode("001-0001");
            Assert.AreEqual<UInt32>(ec_1_1.Category, commonData12.DialogViewButtonMessageReferenceCategory);
            Assert.AreEqual<UInt32>(ec_1_1.Number, commonData12.DialogViewButtonMessageReferenceNumber);
            Assert.AreEqual<UInt32>(ec_1_1.Category, commonData12.FullScreenViewMessageReferenceCategory);
            Assert.AreEqual<UInt32>(ec_1_1.Number, commonData12.FullScreenViewMessageReferenceNumber);
            Assert.AreEqual<UInt32>(ec_1_1.Category, commonData12.FullScreenViewButtonMessageReferenceCategory);
            Assert.AreEqual<UInt32>(ec_1_1.Number, commonData12.FullScreenViewButtonMessageReferenceNumber);
            Assert.AreEqual(0u, commonData12.Attribute);

            // 参照専用エラーを2回目に参照したエラー。先に参照した 001-0012 を参照する
            var commonData13 = systemData.ErrorMessageData["001-0013"].CommonData;
            Assert.AreEqual<UInt32>(new ErrorCode("001-0012").Category, commonData13.DialogViewMessageReferenceCategory);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0012").Number, commonData13.DialogViewMessageReferenceNumber);

            // 二重参照を持つエラー。001-0011 は 001-0010 を参照し、001-0010 は他のエラーを参照する。
            var commonData11 = systemData.ErrorMessageData["001-0011"].CommonData;
            Assert.AreEqual<UInt32>(new ErrorCode("001-0001").Category, commonData11.DialogViewMessageReferenceCategory);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0001").Number, commonData11.DialogViewMessageReferenceNumber);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0002").Category, commonData11.DialogViewButtonMessageReferenceCategory);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0002").Number, commonData11.DialogViewButtonMessageReferenceNumber);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0003").Category, commonData11.FullScreenViewMessageReferenceCategory);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0003").Number, commonData11.FullScreenViewMessageReferenceNumber);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0004").Category, commonData11.FullScreenViewButtonMessageReferenceCategory);
            Assert.AreEqual<UInt32>(new ErrorCode("001-0004").Number, commonData11.FullScreenViewButtonMessageReferenceNumber);
        }
    }
}
