﻿// --------------------------------------------------------------------------------
// <copyright>
// Copyright (C)Nintendo. All rights reserved.
//
// These coded instructions, statements, and computer programs contain proprietary
// information of Nintendo and/or its licensed developers and are protected by
// national and international copyright laws. They may not be disclosed to third
// parties or copied or duplicated in any form, in whole or in part, without the
// prior written consent of Nintendo.
//
// The content herein is highly confidential and should be handled accordingly.
// </copyright>
// --------------------------------------------------------------------------------

using System;
using System.Collections;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.Collections.Specialized;
using System.Diagnostics;
using System.Linq;
using System.Text;
using EffectMaker.Foundation.Extensions;
using EffectMaker.Foundation.Primitives;
using EffectMaker.UIControls.BaseControls;
using EffectMaker.UIControls.Extensions;
using EffectMaker.UIControls.Layout;
using EffectMaker.UIControls.Specifics.Sliders;
using EffectMaker.UILogic.ViewModels;
using Microsoft.VisualStudio.TestTools.UnitTesting;
using Test.EffectMaker.UIControls.TestTypes;

namespace Test.EffectMaker.UIControls.UnitTests
{
    /// <summary>
    /// Class containing unit test related to the UIComboBox control.
    /// </summary>
    [TestClass]
    public class UIComboBoxUnitTests
    {
        /// <summary>
        /// Test the Checked property through data binding.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestMethod()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            combobox.SelectedIndex = 0;
            Assert.AreEqual(dataSource.SelectedItem, ComboBoxTestViewModel.Items.A, "Failed");
            combobox.SelectedIndex = 1;
            Assert.AreEqual(dataSource.SelectedItem, ComboBoxTestViewModel.Items.B, "Failed");
            combobox.SelectedIndex = 2;
            Assert.AreEqual(dataSource.SelectedItem, ComboBoxTestViewModel.Items.C, "Failed");

            dataSource.SelectedItem = ComboBoxTestViewModel.Items.A;
            Assert.AreEqual(combobox.SelectedIndex, 0, "Failed");
            dataSource.SelectedItem = ComboBoxTestViewModel.Items.B;
            Assert.AreEqual(combobox.SelectedIndex, 1, "Failed");
            dataSource.SelectedItem = ComboBoxTestViewModel.Items.C;
            Assert.AreEqual(combobox.SelectedIndex, 2, "Failed");
        }

        /// <summary>
        /// Add test.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestAdd()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            dataSource.AvailableItems.Add(
                new KeyValuePair<string, object>("Combo Box Item D", ComboBoxTestViewModel.Items.D));

            // Add a new available item in the end.
            Assert.AreEqual(combobox.Items.Count, 4, "Failed");

            var item = combobox.Items[3] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.D, "Failed");

            // Insert a new available item at index 1.
            dataSource.AvailableItems.Insert(
                1,
                new KeyValuePair<string, object>("Combo Box Item E", ComboBoxTestViewModel.Items.E));

            Assert.AreEqual(combobox.Items.Count, 5, "Failed");

            item = combobox.Items[1] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.E, "Failed");
        }

        /// <summary>
        /// Replace test.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestReplace()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            // Replace available item B with E.
            dataSource.AvailableItems[1] =
                new KeyValuePair<string, object>("Combo Box Item E", ComboBoxTestViewModel.Items.E);

            // Add a new available item in the end.
            Assert.AreEqual(combobox.Items.Count, 3, "Failed");

            var item = combobox.Items[1] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.E, "Failed");
        }

        /// <summary>
        /// Clear test.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestClear()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            // Clear the available items.
            dataSource.AvailableItems.Clear();

            // Add a new available item in the end.
            Assert.AreEqual(combobox.Items.Count, 0, "Failed");
        }

        /// <summary>
        /// Remove test.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestRemove()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            // Clear the available items.
            dataSource.AvailableItems.RemoveAt(0);

            // Add a new available item in the end.
            Assert.AreEqual(combobox.Items.Count, 2, "Failed");

            var item = combobox.Items[0] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.B, "Failed");
        }

        /// <summary>
        /// Move test.
        /// </summary>
        [TestMethod]
        public void UIComboBoxUnitTestMove()
        {
            var combobox = new UIComboBox();
            var dataSource = new ComboBoxTestViewModel();

            combobox.AddBinding("SelectedItem", "SelectedItem");
            combobox.AddBinding("AvailableItems", "AvailableItems");

            combobox.DataContext = dataSource;

            // Clear the available items.
            dataSource.AvailableItems.Move(0, 1);

            // Add a new available item in the end.
            Assert.AreEqual(combobox.Items.Count, 3, "Failed");

            var item = combobox.Items[1] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.A, "Failed");

            item = combobox.Items[0] as UIComboBox.ComboBoxItem;
            Assert.AreEqual(item.Value, ComboBoxTestViewModel.Items.B, "Failed");
        }
    }

    /// <summary>
    /// コンボボックステスト用ビューモデル
    /// </summary>
    public class ComboBoxTestViewModel : DummyViewModel
    {
        /// <summary>
        /// 選択中の値
        /// </summary>
        private object selectedItem;

        /// <summary>
        /// アイテム
        /// </summary>
        private ObservableCollection<KeyValuePair<string, object>> availableItems =
            new ObservableCollection<KeyValuePair<string, object>>();

        /// <summary>
        /// Constructor.
        /// </summary>
        public ComboBoxTestViewModel()
        {
            this.availableItems.Add(new KeyValuePair<string, object>("Combo Box Item A", Items.A));
            this.availableItems.Add(new KeyValuePair<string, object>("Combo Box Item B", Items.B));
            this.availableItems.Add(new KeyValuePair<string, object>("Combo Box Item C", Items.C));
        }

        /// <summary>
        /// The items for the combo box.
        /// </summary>
        public enum Items
        {
            /// <summary>Item A</summary>
            A,

            /// <summary>Item B</summary>
            B,

            /// <summary>Item C</summary>
            C,

            /// <summary>Item D</summary>
            D,

            /// <summary>Item E</summary>
            E,
        }

        /// <summary>
        /// The selected item.
        /// </summary>
        public object SelectedItem
        {
            get { return this.selectedItem; }
            set { this.SetValue(ref this.selectedItem, value); }
        }

        /// <summary>
        /// アイテム
        /// </summary>
        public ObservableCollection<KeyValuePair<string, object>> AvailableItems
        {
            get { return this.availableItems; }
        }
    }
}
