﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "effectdemo.h"

#include <nw/dev.h>
//#include <viewer/eftvw_ViewerSystem.h>
//#include <ut/eftut_PrimitiveRendererNw.h>
//#include <ut/eftut_TextWriter.h>

#include <eftdemo_System.h>
#include <eftdemo_ModelUtility.h>
#include <eftdemo_PerfomanceAnalyzer.h>

#include <nw/eft/eft2_System.h>
#include <nw/eft/eftvw2_ViewerSystem.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <G3dModelPreview.h>
#if EFT_IS_WIN
#include <fstream>
#endif

#include <nw/eft/eftut2_Heap.h>
#include <cafe/gx2/gx2Enum.h>


#include "CustomShaderSample.h"
#include "CustomActionSample.h"
#include "EffectBinaryPlayer.h"
#include "BackGroundTexture.h"



// エフェクトビューアデバッグ描画用プリミティブレンダラ
//nw::eftut::PrimitiveRendererNw* gPrimitiveRendererNw = NULL;

// デバッグプリミティブ表示オンオフ
//nw::eftdemo::DebugMenuItemOnOff gDebugPrimitiveOnOff;

// シンプルコリジョン共有平面オンオフ
//nw::eftdemo::DebugMenuItemOnOff gFileldCollisionCmnPlaneOnOff;


nw::eftut2::NwHeap              g_StaticHeap;                           // エフェクトヒープ
nw::eftut2::NwHeap              g_DynamicHeap;                          // エフェクトヒープ
nw::eftut2::NwHeap              g_TempHeap;                             // エフェクトヒープ
nw::eftut2::NwHeap              g_ConnectionHeap;                       // EffectMakerとの通信用ヒープ
void*                           g_ConnectionHeapBuffer      = NULL;     // EffectMakerとの通信用ヒープ先メモリ
nw::eft2::System*               g_EftSystem                 = NULL;     // エフェクトシステム
nw::eftvw2::ViewerSystem*       g_EffectViewer              = NULL;     // エフェクトビューア
EffectBinaryPlayer*             g_EffectBinPlayer           = NULL;     // エフェクトバイナリプレイヤー
BackGroundTexture*              g_BackGroundTexture         = NULL;

//------------------------------------------------------------------------------
//      ビューアデータの処理
//------------------------------------------------------------------------------
static void HandleViewerData( nw::eft2::Heap* eftHeap, void* pThis, void* pBin )
{
    EffectDemo* eftDemo = (EffectDemo*)pThis;

    nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( pBin );
    nw::eft2::BinaryData* pTextureBinary = pNodeBinary->GetNextData();

    if ( pNodeBinary->GetBinaryTag() == EFT_MAKE_TAG( 'V', 'I', 'W', 'D' ) )
    {
        nw::eft2::BinaryData* pNodeBinary = static_cast<nw::eft2::BinaryData*>( pBin );
        nw::eftvw2::ResCamera* camera     = reinterpret_cast<nw::eftvw2::ResCamera*>( pNodeBinary->GetBinaryData() );

        camera->FlipEndian();

        // 基本設定
        eftDemo->SetPlaySpeed( camera->frameRate );
        eftDemo->SetGridType( camera->gridType );
        eftDemo->SetLinearMode( camera->isLinearMode != 0 );

        // カメラ設定
        eftDemo->SetCameraSettings(
            camera->position.x, camera->position.y, camera->position.z,
            camera->target.x,   camera->target.y,   camera->target.z,
            camera->rotation,
            camera->projectionType,
            camera->aspectRatio,
            camera->fieldOfView,
            camera->nearClip,
            camera->farClip );

        // 背景カラー
        eftDemo->SetBackgroundColor( camera->bgColor );

        // リアルタイム編集だったらここで終了
        if (camera->isRealtimeEditing == 1)
        {
            return;
        }

        //-----------------------------------------
        // 背景テクスチャ
        //-----------------------------------------
        nw::eft2::ResTexture* resTexture = ((nw::eft2::ResTexture*)pTextureBinary->GetBinaryData());
        resTexture->FlipEndian();

        void*   texBuffer          = ((char*)pTextureBinary->GetBinaryData()) + sizeof( nw::eft2::ResTexture );
        u32     texBufferSize      = pTextureBinary->GetBinarySize() - sizeof( nw::eft2::ResTexture );

        g_BackGroundTexture->Setup( pTextureBinary->guid, resTexture, texBuffer, texBufferSize );
        eftDemo->SetBackgroundImageRes( reinterpret_cast<const void*>( g_BackGroundTexture->GetTextureResource() ) );


        //-----------------------------------------
        // エフェクトバイナリ再生機能
        //-----------------------------------------
        nw::eft2::BinaryData* pSubBinary = pNodeBinary->GetSubData();
        if (pSubBinary != NULL)
        {
            // バイナリ転送対応
            if ( pSubBinary->GetBinaryTag() == EFT_MAKE_TAG( 'P', 'T', 'C', 'L' ) )
            {
                nw::eftvw2::ResPtclBinary* resPtcl = static_cast<nw::eftvw2::ResPtclBinary*>( pSubBinary->GetBinaryData() );
                resPtcl->FlipEndian();

                // ファイルパスを取得
                resPtcl->filePath = &resPtcl->_filePath;

                // 強制リロードのOn/Offを取得
                static int currentReloadCtrl = 0;
                bool forceReload = resPtcl->reloadCtrl != currentReloadCtrl;
                currentReloadCtrl = resPtcl->reloadCtrl;

                // エフェクトバイナリプレイヤークラスにパスを登録
                bool resSetup = g_EffectBinPlayer->SetupEffectBinary( reinterpret_cast<const char*>( resPtcl->filePath), forceReload );

                // 再生するエミッタセットのインデックス送り/戻し処理
                if ( !resSetup )
                {
                    static int currentIndexCtrl = 0;
                    int subIndex = resPtcl->esetIndexCtrl - currentIndexCtrl;
                    currentIndexCtrl = resPtcl->esetIndexCtrl;

                    if (subIndex > 0)
                    {
                        // インデックス送り
                        g_EffectBinPlayer->Send();
                    }
                    else if (subIndex < 0)
                    {
                        // インデックス戻し
                        g_EffectBinPlayer->Prev();
                    }
                }
            }
        }
    }
}

//------------------------------------------------------------------------------
//      ビューアメッセージの処理
//------------------------------------------------------------------------------
static void HandleViewerMessage( nw::eft2::Heap* eftHeap, void* pThis, void* pBin )
{
    EffectDemo* eftDemo = (EffectDemo*)pThis;

    struct Config
    {
        s32             frameRate;          //!< フレームレート
        s32             resolution;         //!< 解像度
        nw::math::VEC4  clearColor;         //!< 背景カラー
        f32             worldScaleRange;    //!< ワールドスケール範囲
        s32             worldScaleTime;     //!< ワールドスケール時間
        f32             gridScale;          //!< グリッドスケール
        s32             isLinearEditMode;   //!< リニア編集モード

        // エンディアン反転
        void FlipEndian()
        {
            nw::eft2::EndianUtil::Flip( &frameRate );
            nw::eft2::EndianUtil::Flip( &resolution );
            nw::eft2::EndianUtil::Flip( &clearColor );
            nw::eft2::EndianUtil::Flip( &worldScaleRange );
            nw::eft2::EndianUtil::Flip( &worldScaleTime );
            nw::eft2::EndianUtil::Flip( &gridScale );
            nw::eft2::EndianUtil::Flip( &isLinearEditMode );
        }
    };

    Config* config   = (Config *)pBin;
    config->FlipEndian();

    eftDemo->SetFrameRate( (nw::eftdemo::INIT_SETTINGS_FRAME_RATE)config->frameRate );

    u32 resolution = 0;
    switch ( config->resolution )
    {
        case 0: resolution = 2; break;
        case 1: resolution = 1; break;
        case 2: resolution = 0; break;
        default:                break;
    }

    // フレームレート / 解像度 / 背景色 をファイルとして出力
    nw::eftdemo::System::SaveConfigFile(
        (nw::eftdemo::INIT_SETTINGS_FRAME_RATE)config->frameRate,
        (nw::eftdemo::INIT_SETTINGS_RESOLUTION)resolution,
        0.0f, 0.0f, 0.0f, 1.0f );
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 変数設定
//------------------------------------------------------------------------------
void EffectDemo::InitDemoValue()
{
    // デバッグメニューの表示位置
    mDebugMenuPosX  = 16;
    mDebugMenuPosY  = 16;
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 初期化処理
//------------------------------------------------------------------------------
void EffectDemo::Initialize( nw::ut::MemoryAllocator* allocator, nw::eftdemo::DebugMenu* debugMenu )
{
    mNwAllocator      = allocator;
    mDebugMenu        = debugMenu;

    mDisplayWindow    = true;

    // エフェクトヒープを初期化します。
    //mEftHeap.SetNwAllocator( allocator );
    //mEftDynamicHeap.SetNwAllocator( allocator );

    // デバッグ描画用プリミティブレンダラ生成
    //gPrimitiveRendererNw = new nw::eftut::PrimitiveRendererNw( allocator );
    //gPrimitiveRendererNw->SetRenderer( nw::dev::PrimitiveRenderer::GetInstance() );
    //gPrimitiveRendererNw->SetGraphics( nw::gfnd::Graphics::GetInstance() );

    //// デバッグメニュー
    //mDebugMenuPage.SetPageName( "EftVwrDemo" );

    //// デバッグプリミティブ
    //gDebugPrimitiveOnOff.SetItemName( "DbgPrim" );
    //gDebugPrimitiveOnOff.SetValue( false );

    //// シンプルコリジョン　共有平面OnOff
    //gFileldCollisionCmnPlaneOnOff.SetItemName( "ColCmnPlane" );
    //gFileldCollisionCmnPlaneOnOff.SetValue( true );

    //mDebugMenuPage.AddMenuItem( &gDebugPrimitiveOnOff );
    //mDebugMenuPage.AddMenuItem( &gFileldCollisionCmnPlaneOnOff );

    //// ページを追加。
    //mDebugMenu->AddMenuPage( &mDebugMenuPage );

    //----------------------------------------------------------------
    //
    //----------------------------------------------------------------

    // Heap
    g_StaticHeap.SetNwAllocator( allocator );
    g_DynamicHeap.SetNwAllocator( allocator );
    g_TempHeap.SetNwAllocator( allocator );

    // System 初期化
    nw::eft2::Config  config;
    config.SetEffectHeap( &g_StaticHeap );
    config.SetEffectDynamicHeap( &g_DynamicHeap );
    config.SetEmitterNum( 1024 );
    config.SetStripeNum( 128 );
    config.SetEmitterSetNum( 256 );
    config.SetEmitterNum( 1024 );
    config.SetDoubleBufferSize( 1024 * 32 );
    config.SetStripeNum( 128 );
    config.SetUniformBlockCpuChacheFlush( true );
    config.SetUniformBlockGpuChacheFlush( true );
    g_EftSystem = new nw::eft2::System( config );

    this->mEftSystem = g_EftSystem;

    // エフェクト インゲーム ビューア 初期化
    // 通信用ヒープとして32M切り出す。
    g_ConnectionHeapBuffer = g_TempHeap.Alloc( 1024 * 1024 * 32 );
    g_ConnectionHeap.Initialize( g_ConnectionHeapBuffer, 1024 * 1024 * 32 );

    // Viewer 初期化
    g_EffectViewer = new nw::eftvw2::ViewerSystem( &g_TempHeap, &g_ConnectionHeap, g_EftSystem );
    mEftVwrSystem  = g_EffectViewer;

    // ViewerData用のコールバックを登録します。
    g_EffectViewer->SetViewerDataCallBack( this, HandleViewerData );

    // ViewerMessage用のコールバックを登録します。
    g_EffectViewer->SetViewerMessageCallBack( this, HandleViewerMessage );

    // モデル初期化
    nw::eftdemo::InitializeG3dShaderResource( allocator );
    g_EffectViewer->SetCreateModelPreviewCallback(  G3dModelPreview::CreateModelPreview  );
    g_EffectViewer->SetDestroyModelPreviewCallback( G3dModelPreview::DestroyModelPreview );

    // 通信OK
    g_EffectViewer->SetAppIsInitialized();

    // カスタムシェーダテスト
    CustomShaderSample::Initialize( g_EftSystem, allocator );

    // カスタムアクションテスト
    CustomActionTest::Initialize( g_EftSystem );

    // エフェクトバイナリプレイヤー
    g_EffectBinPlayer = new EffectBinaryPlayer( g_EftSystem, &g_TempHeap, 0, 0 );

    // 背景テクスチャ
    g_BackGroundTexture = new BackGroundTexture( &g_TempHeap );
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 終了処理
//------------------------------------------------------------------------------
void EffectDemo::Finalize()
{
    delete g_BackGroundTexture;
    delete g_EffectBinPlayer;
    delete g_EffectViewer;
    delete g_EftSystem;

    g_StaticHeap.Finalize();
    g_DynamicHeap.Finalize();

    g_ConnectionHeap.Finalize();
    g_TempHeap.Free( g_ConnectionHeapBuffer );
    g_TempHeap.Finalize();
}

//------------------------------------------------------------------------------
//      EffectViewerDemo 計算処理
//------------------------------------------------------------------------------
void EffectDemo::ProcCalc( nw::dev::Pad* pad, nw::eftdemo::DrawParam& drawParam  )
{
    // フレーム開始処理
    g_EftSystem->BeginFrame();

    // エフェクト計算処理
    g_EftSystem->Calc( 0, 1.0f, true );

    // ビューア計算処理を行い返り値をみて、ビューアグループのエフェクト計算処理を分岐
    if( !g_EffectViewer->ProcCalc( GetPlaySpeed(), drawParam.mViewMtx ) )
    {
        g_EftSystem->Calc( nw::eft2::EFT_VIEWER_GROUP, GetPlaySpeed(), true );  // 再生中
    }
    else
    {
        g_EftSystem->Calc( nw::eft2::EFT_VIEWER_GROUP, 0.0f, false );           // 一時停止
    }
}

//------------------------------------------------------------------------------
//      通常パスのエフェクト描画処理
//------------------------------------------------------------------------------
void EffectDemo::ProcDraw( nw::eftdemo::DrawParam& drawParam, u32 drawPathFlag )
{
#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

    nw::eft2::ViewParam view;
    view.Set( &drawParam.mViewMtx, &drawParam.mProjMtx, &drawParam.mCameraPos, drawParam.mNearZ, drawParam.mFarZ );

    g_EftSystem->SwapBuffer();
    g_EftSystem->SetViewParam( &view );
    g_EftSystem->SetEmitterDrawProfilerCallback( nw::eftdemo::PerfAnalyzerCallBack );

    // EffectMakerから指定された描画パス順での描画に変更予定
    g_EftSystem->AddSortBuffer( 0, drawPathFlag );
    g_EftSystem->AddSortBuffer( nw::eft2::EFT_VIEWER_GROUP, drawPathFlag );
    g_EftSystem->DrawSortBuffer( true, NULL );


#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif
}


//------------------------------------------------------------------------------
//  モデルプレビュー計算処理
//------------------------------------------------------------------------------
void CalcModelPreviewBlock( const nw::math::Matrix44*           projMatrix,
                           const nw::math::Matrix34*           viewMatrix,
                           const nw::math::Vector4*            lightPos,
                           nw::eftdemo::SimpleShader::ViewId   id )
{
    // モデルのマトリクスブロック更新処理
    {
        nw::eftvw2::Preview* preview = g_EffectViewer->GetPreviewHead();
        while( preview )
        {
            if (preview->GetPreviewType() == nw::eftvw2::Preview::EFT_VWR_PRV_MODEL )
            {
                G3dModelPreview* modelPreview = reinterpret_cast<G3dModelPreview*>(preview);
                modelPreview->CalcBlock(
                    reinterpret_cast<const nw::g3d::math::Mtx34*>(viewMatrix),
                    reinterpret_cast<const nw::g3d::math::Mtx44*>(projMatrix),
                    reinterpret_cast<const nw::g3d::math::Vec4*>(lightPos), id );
            }
            preview = preview->GetNextPreview();
        }

        // 以下の関数は この関数の外で呼び出す必要があります。
        nw::g3d::CPUCache::Sync(); // CPU キャッシュ操作の完了を待ちます。

        // GPU のリードキャッシュはここで一括して Invalidate します。
        nw::g3d::GPUCache::InvalidateAll();
    }
}

//------------------------------------------------------------------------------
//  モデルプレビュー描画処理
//------------------------------------------------------------------------------
void DrawModelPreview( nw::eftdemo::SimpleShader::ViewId id )
{
    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_BLOCK);

    nw::eftvw2::Preview* preview = g_EffectViewer->GetPreviewHead();
    while( preview )
    {
        if (preview->GetPreviewType() == nw::eftvw2::Preview::EFT_VWR_PRV_MODEL && preview->IsVisible() )
        {
            G3dModelPreview* modelPreview =  reinterpret_cast<G3dModelPreview*>(preview);
            modelPreview->Draw( id );
        }
        preview = preview->GetNextPreview();
    }

    nw::g3d::SetShaderMode(GX2_SHADER_MODE_UNIFORM_REGISTER);
}

//---------------------------------------------------------------------------
//      モデルの描画処理
//---------------------------------------------------------------------------
void EffectDemo::ProcModelDraw( nw::eftdemo::DrawParam& drawParam, nw::eftdemo::SimpleShader::ViewId viewId )
{
#if defined(NW_PLATFORM_WIN32)
    static const GLenum renderTargetList[] = { GL_COLOR_ATTACHMENT0_EXT, GL_COLOR_ATTACHMENT1_EXT };
    glDrawBuffers( sizeof(renderTargetList) / sizeof(renderTargetList[0]), renderTargetList );
#endif

    // ライトパラメータを設定
    if ( viewId == nw::eftdemo::SimpleShader::VIEW_ID_LIGHT )
    {
        nw::math::Matrix44 lightViewProjection;
        lightViewProjection.SetMult( drawParam.mProjMtx, nw::math::Matrix44( drawParam.mViewMtx ) );
        CustomShaderSample::SetLightViewProjectionMatrix( lightViewProjection );
    }

    // モデルプレビューの描画
    nw::math::VEC4 lightPos;
    lightPos.x = drawParam.mCameraPos.x;
    lightPos.y = drawParam.mCameraPos.y;
    lightPos.z = drawParam.mCameraPos.z;

    CalcModelPreviewBlock( &drawParam.mProjMtx, &drawParam.mViewMtx, &lightPos, viewId );
    DrawModelPreview( viewId );

#if defined(NW_PLATFORM_WIN32)
    glDrawBuffer( GL_COLOR_ATTACHMENT0_EXT );
#endif
}


// eset数取得
inline s32 _getNumEset( nw::eft2::System* ptclSys , s32 resId )
{
    nw::eft2::Resource *res = ptclSys->GetResource( resId );
    if( !res )
    {
        return 0;
    }
    return res->GetEmitterSetNum();
}

// esetの名前取得
inline const char *_getEsetName( nw::eft2::System* ptclSys , s32 resId , s32 setIx )
{
    nw::eft2::Resource *res = ptclSys->GetResource( resId );
    return res->GetEmitterSetName( setIx );
}

//---------------------------------------------------------------------------
//      エフェクトデモのデバッグ文字描画
//---------------------------------------------------------------------------
void EffectDemo::DebugDraw( nw::dev::DevTextWriter* writer )
{
    writer->Printf( "\n" );
    writer->Printf( " EmitterSet(%4d/%4d)\n",       g_EftSystem->GetProcessingEmitterSetCount(), g_EftSystem->GetEmitterSetMaxNum() );
    writer->Printf( " Emitter   (%4d/%4d/%4d)\n",   g_EftSystem->GetProcessingEmitterCount(), g_EftSystem->GetEmitterMaxNum(), g_EftSystem->GetFreeEmitterActualNum() );
    writer->Printf( " EmitterAnm(%4d)\n",           g_EftSystem->GetProcessingEmitterAnimNum() );
    writer->Printf( " CpuPtcl   (%4d)\n",           g_EftSystem->GetProcessingCpuParticleCount() );
    writer->Printf( " GpuPtcl   (%4d)\n",           g_EftSystem->GetProcessingGpuParticleCount() );
    writer->Printf( " GpuSoPtcl (%4d)\n",           g_EftSystem->GetProcessingGpuSoParticleCount() );
    writer->Printf( " D-HeapSize(%8d)\n",           g_EftSystem->GetAllocedFromDynamicHeapSize() );
    writer->Printf( " TempBuffer(%8d)\n",           g_EftSystem->GetUsedTemporaryBufferSize() );
    writer->Printf( " TempBufMax(%8d)\n",           g_EftSystem->GetMaxUsedTemporaryBufferSize( nw::eft2::EFT_CPU_CORE_1 ) );
    writer->Printf( " Time      ( %f/%f )\n",       g_EffectViewer->GetTime(), g_EffectViewer->GetEndTime() );
    writer->Printf( " Loop      ( %d )\n",          g_EffectViewer->IsLoop() );
}

//---------------------------------------------------------------------------
//      カメラを設定します。
//---------------------------------------------------------------------------
void EffectDemo::SetCameraSettings(
    f32 posX, f32 posY, f32 posZ,
    f32 lookAtX, f32 lookAtY, f32 lookAtZ,
    f32 rotation,
    int projType,
    f32 aspect,
    f32 fov,
    f32 znear, f32 zfar )
{
    if ( mVwrCamera == NULL || mProjectionMatrix == NULL )
    {
        return;
    }

    nw::math::VEC3 zaxis( lookAtX - posX, lookAtY - posY, lookAtZ - posZ );
    if ( zaxis.LengthSquare() == 0.0f )
    {
        return;
    }

    mVwrCamera->SetPos( posX, posY, posZ );
    mVwrCamera->SetLookAtPos( lookAtX, lookAtY, lookAtZ );

    // アップベクトルを計算
    //mVwrCamera->SetTwist( rotation / nw::math::F_PI * 180.0f );
    {
        nw::math::VEC3 lookUp;

        if ( zaxis.x * zaxis.x + zaxis.z * zaxis.z != 0.0f)
        {
            lookUp = nw::math::VEC3( 0.0f, 1.0f, 0.0f );
        }
        else
        {
            lookUp = nw::math::VEC3( 0.0f, 0.0f, 1.0f );
        }

        nw::math::MTX34 rotMtx;
        rotMtx.SetRotate( zaxis, -rotation );

        nw::math::VEC3 rotLookUp;
        VEC3Transform( &rotLookUp, &rotMtx, &lookUp );

        mVwrCamera->GetMatrix();
        mVwrCamera->SetUpVec( rotLookUp );
    }

    if ( znear <= zfar * 0.000001f )
    {
        znear = zfar * 0.000001f;
    }

    if ( projType == 0 )
    {
        // パースペクティブ
        mProjectionMatrix->SetPerspective( fov, aspect, znear, zfar * 100.f );
    }
    else
    {
        // オルソ
        f32 h = fov / 2.0f;
        f32 w = h * aspect;

        mProjectionMatrix->SetOrtho( -w, w, -h, h, znear, zfar );
    }
}
