﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once


#include <nw/dev.h>
#include <nw/eft2.h>
#include <nw/eft/eftvw2_FileSystem.h>
#include <nw/eft/eftvw2_String.h>

//---------------------------------------------------
//  エフェクトバイナリプレイヤー
//---------------------------------------------------
class EffectBinaryPlayer
{
public:
    //------------------------------------------------------------------------------
    //! @brief  コンストラクタ
    //------------------------------------------------------------------------------
    EffectBinaryPlayer( nw::eft2::System* system, nw::eft2::Heap* heap, u32 resourceID, u8 groupID )
    {
        m_System            = system;
        m_Heap              = heap;
        m_ResourceID        = resourceID;
        m_GroupID           = groupID;
        m_EmitterSetID      = 0;
        m_EffectBinary      = NULL;
        m_EffectBinaryPath  = nw::eftvw2::VwrString();
    }

    //------------------------------------------------------------------------------
    //! @brief  デストラクタ
    //------------------------------------------------------------------------------
    ~EffectBinaryPlayer()
    {
        UnloadEffectBinary();
    }

    //------------------------------------------------------------------------------
    //! @brief  エフェクトバイナリをセットアップする
    //------------------------------------------------------------------------------
    bool SetupEffectBinary( const char* effectBinaryPath, bool forceReload = false )
    {
        u32   readBufferSize    = 0;

        // パスを比較して異なる場合はセットアップを行う
        if ( forceReload == false && strcmp( m_EffectBinaryPath.Cstr(), effectBinaryPath ) == 0 )
        {
            return false;
        }

        // 既存のバイナリを破棄
        UnloadEffectBinary();

        // 再生するエミッタセットIDをリセット
        m_EmitterSetID = 0;

        // バイナリをロードしてシステムにエントリ
        bool ret = nw::eftvw2::FileSystem::Load( m_Heap, effectBinaryPath, &m_EffectBinary, &readBufferSize );
        if ( ret )
        {
            m_System->EntryResource( m_Heap, m_EffectBinary, m_ResourceID, true );
            m_System->GetResource( m_ResourceID )->OutputResourceInfo();
            m_System->CreateEmitterSetID( &m_Handle, m_EmitterSetID, m_ResourceID, m_GroupID );
//            m_System->CreateManualEmitEmitterSetID( &m_Handle, m_EmitterSetID, m_ResourceID, m_GroupID, 256 );
            m_EffectBinaryPath.Copy( effectBinaryPath );
            return  true;
        }
        else
        {
            m_EffectBinaryPath.Copy( effectBinaryPath );
            return false;
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  エフェクトバイナリを破棄する
    //------------------------------------------------------------------------------
    bool UnloadEffectBinary()
    {
        // リソースに紐づいたエミッタセットを破棄
        m_System->KillEmitterSetGroup( m_GroupID );

        // バイナリを破棄
        if ( m_EffectBinary )
        {
            m_Heap->Free( m_EffectBinary );
            m_EffectBinary = NULL;
            return  true;
        }

        return false;
    }

    //------------------------------------------------------------------------------
    //! @brief  指定IDのエミッタセットを再生
    //------------------------------------------------------------------------------
    void Play( s32 index )
    {
        if ( !m_EffectBinary )
        {
            return;
        }

        m_EmitterSetID = index;

        // エミッタセット数を取得する
        u32 emitterSetCount = m_System->GetResource( m_ResourceID )->GetEmitterSetCount();

        if ( m_EmitterSetID < emitterSetCount )
        {
            m_System->KillEmitterSetGroup( m_GroupID );
            m_System->CreateEmitterSetID( &m_Handle, m_EmitterSetID, m_ResourceID, m_GroupID );
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  次のエミッタセットを再生
    //------------------------------------------------------------------------------
    void Send()
    {
        if ( !m_EffectBinary )
        {
            return;
        }

        // エミッタセット数を取得する
        u32 emitterSetCount = m_System->GetResource( m_ResourceID )->GetEmitterSetCount();

        if ( m_EmitterSetID + 1 < emitterSetCount )
        {
            m_EmitterSetID++;
            m_System->KillEmitterSetGroup( m_GroupID );
            m_System->CreateEmitterSetID( &m_Handle, m_EmitterSetID, m_ResourceID, m_GroupID );
            nw::eft2::OutputLog( "EmitterSet Name : %s\n", m_Handle.GetEmitterSet()->GetEmitterSetName() );
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  前のエミッタセットを再生
    //------------------------------------------------------------------------------
    void Prev()
    {
        if ( !m_EffectBinary )
        {
            return;
        }

        if ( m_EmitterSetID > 0 )
        {
            m_EmitterSetID--;
            m_System->KillEmitterSetGroup( m_GroupID );
            m_System->CreateEmitterSetID( &m_Handle, m_EmitterSetID, m_ResourceID, m_GroupID );
        }
    }

    //------------------------------------------------------------------------------
    //! @brief  マニュアル放出
    //------------------------------------------------------------------------------
    void Emit()
    {
        if ( !m_EffectBinary )
        {
            return;
        }

        if ( m_Handle.IsValid() )
        {
            m_Handle.GetEmitterSet()->EmitParticle( nw::math::VEC3( 0.0f, 0.0f, 0.0f ) );
        }
    }


private:
    nw::eft2::System*       m_System;
    nw::eft2::Heap*         m_Heap;
    u32                     m_ResourceID;
    u8                      m_GroupID;
    u32                     m_EmitterSetID;
    nw::eft2::Handle        m_Handle;
    void*                   m_EffectBinary;
    nw::eftvw2::VwrString   m_EffectBinaryPath;
};
