﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <effectdemo.h>

#include <nw/eft/eftvw2_FileSystem.h>
#include <eftdemo_System.h>
#include <eftdemo_FrameBuffer.h>
#include <eftdemo_Camera.h>
#include <eftdemo_Light.h>
#include <eftdemo_Layer.h>
#include <eftdemo_ScreenDrawer.h>
#include <eftdemo_GlareEffect.h>
#include <eftdemo_CubeMap.h>
#include <eftdemo_Menu.h>
#include <eftdemo_DisplayList.h>
#include <nw/eft/eft2_System.h>
#include <eftdemo_PerfomanceAnalyzer.h>
#include <nw/eft/eftvw2_EffectPreview.h>

#include <Demo.h>


//#define NW_USE_GPU_PROFILER
#if defined(NW_PLATFORM_WIN32)
#include <windows.h>
#include <tsu_SavecurrentBufferAsBmp.h>
#endif

#include <nw/dw.h>
#include <nw/dw/dw_Controls.h>
#include <nw/dw/control/dw_FixedStackPanel.h>

// Allocator
nw::ut::MemoryAllocator         gNwAllocator;

// System
nw::eftdemo::System*            gSystem = NULL;

// デフォルトフレームバッファ
nw::eftdemo::FrameBuffer        gFrameBuffer64;
nw::eftdemo::FrameBuffer        gFrameBuffer32;
nw::eftdemo::FrameBuffer        gReductionFrameBuffer;
nw::eftdemo::FrameBuffer        gNormalFrameBuffer;

// デフォルトフレームバッファコピーテクスチャ
nw::eftdemo::FrameBufferTexture gFrameBufferTextureCopy;
nw::eftdemo::DepthBufferTexture gDepthBufferTextureCopy;

// ライトビュー用から生成するシャドウマップテクスチャ
nw::eftdemo::DepthBufferTexture gShadowMapTextureCopy;

// グレアエフェクト
nw::eftdemo::GlareEffect        gGlareEffect;

// キューブマップ
nw::eftdemo::CubeMap            gCubeMap;

// PrimitiveRenderer
nw::dev::PrimitiveRenderer*     gPrimitiveRenderer = NULL;

nw::math::VEC3                  gCameraPosition;            // カメラ位置
nw::math::VEC3                  gCameraLookAt;              // カメラ視点
nw::eftdemo::Camera             gDemoCamera;                // マウス操作カメラクラス
nw::math::MTX44                 gProjctionMatrix;           // プロジェクション

nw::eftdemo::DrawParam          gLightParam;                // ライトパラメータ
nw::eftdemo::FrameBuffer        gLightFrameBuffer;          // ライトビュー用のフレームバッファ

nw::eftdemo::DefaultBackground  gBackground;                // 背景描画クラス
nw::eftdemo::DefaultGrid        gGrid;                      // グリッド描画クラス
nw::eftdemo::HighGrid           gHighGrid;                  // 高詳細グリッド描画クラス

// CPUコスト平均
enum
{
    CPU_COST_TICK_MAX = 600
};

u32                                 gCpuCostTickCounter;
nw::ut::Tick                        gCpuCostTick[CPU_COST_TICK_MAX];
nw::ut::Tick                        gCpuCostAverage;
s64                                 gCpuCostAverageNanoSec;

// 描画パス
#define DRAW_PATH_OPAQU_BUFFER              ( nw::eft2::EFT_DRAW_PATH_FLAG_3 )
#define DRAW_PATH_PRE_COPY_COLOR_BUFFER     ( nw::eft2::EFT_DRAW_PATH_FLAG_2 )
#define DRAW_PATH_DEFAULT                   ( nw::eft2::EFT_DRAW_PATH_FLAG_0 )
#define DRAW_PATH_REDUCTION_BUFFER          ( nw::eft2::EFT_DRAW_PATH_FLAG_1 )

// ディスプレイリスト
u32                             gUseDisplayListSide = 0;
nw::eftdemo::DisplayList        gDisplayList[2];

// demo common コードを共有している為の回避
extern nw::eft2::System*        g_EftSystem;

// デバッグ表示
nw::internal::dw::WindowManager     s_DWMgr;
nw::internal::dw::NwUIRenderer      s_UIRenderer;
nw::internal::dw::NwTextRenderer    s_TextRenderer;

//----------------------------------------
// グリッドタイプ
//----------------------------------------
enum GridType
{
    GRID_TYPE_GRID      = 0,
    GRID_TYPE_HIGH_GRID = 1,
    GRID_TYPE_BLOCK     = 2,
    GRID_TYPE_NONE      = 3,
};

//------------------------------------------------------------------------------
//      デモシステム 初期化処理
//------------------------------------------------------------------------------
void Initialize()
{
    // ファイルシステムの初期化
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    nw::dev::FileDeviceManager::CreateArg fileDeviceArg;
    fileDeviceArg.allocator = &gNwAllocator;
    fileSystem->Initialize( fileDeviceArg );

    // デモシステムの作成
    nw::eftdemo::System::CreateArg    arg;

    arg.allocator   = &gNwAllocator;
    arg.drawMeter   = true;
    arg.fontPath    = "common/fonts/nintendo_NTLG-DB_002.bffnt";
    arg.clearColor  = 0x333333FF;

#if defined(NW_PLATFORM_WIN32)
    arg.width       = 960;
    arg.height      = 540;
    arg.waitVBlank  = 0;
#endif

#if defined(NW_PLATFORM_CAFE)
    arg.width          = 1280;
    arg.height         = 720;
    arg.waitVBlank     = 1;
    arg.fontShaderPath = "common/shaders/font_BuildinShader.gsh";
#endif

#if defined(NW_PLATFORM_WIN32)
    nw::eftdemo::System::LoadConfigFile(&gNwAllocator, &arg);
#endif

    gSystem = new nw::eftdemo::System( arg );

    // デモシステムの初期化
    gSystem->Initialize();

    // 入力インターフェースの初期化
    gSystem->InitializeInputInterface();

    // グラフィックシステムの初期化
    gSystem->InitializeGraphicsSystem();

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        // PrimitiveRenderer の初期化
        gPrimitiveRenderer = nw::dev::PrimitiveRenderer::GetInstance();

#if defined(NW_PLATFORM_WIN32)
        gPrimitiveRenderer->Initialize( &gNwAllocator );
#else
        // PrimitiveRenderer で用いるシェーダーバイナリへのパスを指定する。
        nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
        nw::dev::FileDevice::LoadArg loadArg;
        loadArg.path = "common/shaders/dev_PrimitiveRenderer.gsh";
        loadArg.allocator = &gNwAllocator;
        u8* binary = fileSystem->Load( loadArg );
        gPrimitiveRenderer->InitializeFromBinary( &gNwAllocator, binary, loadArg.readSize );
        fileSystem->Unload( loadArg, binary );
#endif
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

#if defined(NW_PLATFORM_WIN32)
    // GL では座標を Cafe と同じ中心に設定する。
//    nw::ut::DynamicCast<nw::dev::WPadWin*>( gSystem->GetWPad() )->SetPointerCenterOrigin( true );
//    nw::ut::DynamicCast<nw::dev::MouseWin*>( gSystem->GetMouse() )->SetPointerCenterOrigin( true );
#endif
}


//------------------------------------------------------------------------------
//      デモシステム 終了処理
//------------------------------------------------------------------------------
void Finalize()
{
    // PrimitiveRenderer終了処理
    gPrimitiveRenderer->Finalize( &gNwAllocator );

    // デモシステム終了処理
    gSystem->FinalizeGraphicsSystem();
    gSystem->FinalizeInputInterface();
    gSystem->Finalize();

    // ファイルシステム終了処理
    nw::dev::FileDeviceManager* fileSystem = nw::dev::FileDeviceManager::GetInstance();
    fileSystem->Finalize();
}



//------------------------------------------------------------------------------
//　描画パス コールバック
//  縮小バッファへの描画時にコールされる
//------------------------------------------------------------------------------
static void DrawPathRenderStateSetCallback( nw::eft2::RenderStateSetArg& arg )
{
}


//------------------------------------------------------------------------------
//      キューブマップテクスチャ 生成処理
//------------------------------------------------------------------------------
void DrawCubeMap( EffectDemo* effectDemo, nw::eftdemo::DrawParam& drawParam, nw::eftdemo::SimpleShader::ViewId id )
{
        effectDemo->ProcModelDraw( drawParam, id );
}


//------------------------------------------------------------------------------
//　描画コマンドを発行する
//------------------------------------------------------------------------------
void MakeDrawCommand( EffectDemo*                effectDemo,
                      nw::eftdemo::DrawParam*    drawParam,
                      nw::eftdemo::DrawParam*    lightParam,
                      nw::eftdemo::ScreenDrawer* drawer,
                      nw::eftdemo::FrameBuffer*  currentFrameBuffer,
                      u32                        gridType,
                      bool                       makeFrameBufferCopy,
                      bool                       makeShadowMap,
                      u32                        drawGlareNum,
                      bool                       drawNormalBuffer,
                      bool                       drawCubeMapTexture )
{
    NW_GL_ASSERT();
    gSystem->GetMeterDraw().BeginMeasure();

    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();

    bool isRenderingReductionBuffer = false;
    bool isRenderingShadowMapBuffer = false;
    bool isRenderingPreColorCopyBuffer = true;
/*
    // 縮小バッファに描画するエフェクトが存在するか?
    isRenderingReductionBuffer      = effectDemo->IsHasRenderingEmitter( nw::eftvw2::EFT_VWR_USE_GROUP_ID, DRAW_PATH_REDUCTION_BUFFER );

    // シャドウマップに描画する(不透明)エフェクトが存在するか?
    isRenderingShadowMapBuffer      = effectDemo->IsHasRenderingEmitter( nw::eftvw2::EFT_VWR_USE_GROUP_ID, DRAW_PATH_OPAQU_BUFFER );

    // フレームバッファコピー前(屈折に移る)エフェクトが存在するか?
    isRenderingPreColorCopyBuffer   = effectDemo->IsHasRenderingEmitter( nw::eftvw2::EFT_VWR_USE_GROUP_ID, DRAW_PATH_PRE_COPY_COLOR_BUFFER );
*/
    //-----------------------------------------------------------
    // 各バッファのクリアカラー指定
    //-----------------------------------------------------------
    // 背景カラーを反映
            NW_GL_ASSERT();
    currentFrameBuffer->SetClearColor( effectDemo->GetBackgroundColor() );
        NW_GL_ASSERT();
    // 縮小バッファは、黒背景+α1.0fでクリアする
    gReductionFrameBuffer.SetClearColor( nw::math::VEC4( 0.0f, 0.0f, 0.0f, 1.0f ) );
        NW_GL_ASSERT();
/*
    //-----------------------------------------------------------
    // シャドウマップ生成
    //-----------------------------------------------------------
    if ( makeShadowMap )
    {
        // ライト用フレームバッファへ切り替え
        gLightFrameBuffer.Bind();

        // ライト用フレームバッファのクリア
        gLightFrameBuffer.Clear( false, true );

        // カラーは描かない
        graphics->SetColorMask( false, false, false, false );

        // ライトビューを描画
        effectDemo->ProcModelDraw( *lightParam, nw::eftdemo::SimpleShader::VIEW_ID_LIGHT );

        //-----------------------------------------------------------
        // 不透明　描画パス　描画処理
        //-----------------------------------------------------------
        if ( isRenderingShadowMapBuffer )
        {
            effectDemo->ProcDraw( *lightParam, DRAW_PATH_OPAQU_BUFFER );
        }

        graphics->SetColorMask( true, true, true, true );

        // 標準フレームバッファに戻す。
        currentFrameBuffer->Bind();

        // デプスをシャドウマップテクスチャとして設定する。
#if defined(NW_PLATFORM_WIN32)
        drawParam->mShadowMap = gLightFrameBuffer.GetDepthBufferTexture();
#endif
#if defined(NW_PLATFORM_CAFE)
        gLightFrameBuffer.CopyDepthBufferTexture( &gShadowMapTextureCopy );
        drawParam->mShadowMap = gShadowMapTextureCopy.GetGX2Texture();
#endif
    }
/*

/*
    //-----------------------------------------------------------
    // キューブマップ生成
    //-----------------------------------------------------------
    {
        gCubeMap.SetPosition( nw::math::VEC3( 0.0f, 8.0f, 0.0f ) );
        //gCubeMap.SetColor( effectDemo->GetBackgroundColor() );
        gCubeMap.SetColor( nw::math::VEC4( 0.0f, 0.0f, 0.0f, 0.0f ) );

        gCubeMap.Draw( effectDemo, *drawParam );
        currentFrameBuffer->Bind();

        drawParam->mCubeMap = gCubeMap.GetCmTexture();
    }
*/

    //-----------------------------------------------------------
    // ベースとなるフレームバッファに切り替えてクリア
    //-----------------------------------------------------------
    currentFrameBuffer->Bind();
    currentFrameBuffer->Clear( true, true );


    //-----------------------------------------------------------
    // 背景の描画
    //-----------------------------------------------------------
    {
#if defined(NW_PLATFORM_WIN32)
        glBindTexture( GL_TEXTURE_2D, 0 );
#endif

        // 背景画像の描画
        if ( effectDemo->GetBackgroundTexture().IsInitialized() || effectDemo->GetBackgroundTextureRes() != NULL )
        {
            nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
            graphics->SetBlendEnable( false );
            graphics->SetDepthEnable( false, false );
            graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );
#if defined (NW_PLATFORM_WIN32)
            drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY,  reinterpret_cast<const nw::eft2::TextureResource*>( effectDemo->GetBackgroundTextureRes() ) , true );
#endif
#if defined (NW_PLATFORM_CAFE)
            drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY,  reinterpret_cast<const nw::eft2::TextureResource*>( effectDemo->GetBackgroundTextureRes() ), false );
#endif
            graphics->SetBlendEnable( true );
            graphics->SetDepthEnable( true, true );
        }

        // 背景モデルの描画
        if ( gridType == GRID_TYPE_BLOCK )
        { gBackground.ProcDraw( *drawParam ); }

        // グリッドの描画
        if ( gridType == GRID_TYPE_GRID )
        { gGrid.ProcDraw( *drawParam ); }
        if ( gridType == GRID_TYPE_HIGH_GRID )
        { gHighGrid.ProcDraw( *drawParam ); }

        // 法線バッファをクリア
        gNormalFrameBuffer.Bind();
        gNormalFrameBuffer.SetClearColor(nw::math::VEC4(0.0f, 0.0f, 0.0f, 0.0f));
        gNormalFrameBuffer.Clear( true, false );

        currentFrameBuffer->Bind();

#if defined(NW_PLATFORM_WIN32)
        // レンダーターゲットを設定
        glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, gNormalFrameBuffer.GetColorBufferTexture(), 0 );

        // モデル描画
        effectDemo->ProcModelDraw( *drawParam );

        // レンダーターゲットを戻す
        glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT1_EXT, GL_TEXTURE_2D, 0, 0 );

        NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)
        // レンダーターゲットを設定
        GX2SetColorBuffer( gNormalFrameBuffer.GetGX2ColorBuffer(), GX2_RENDER_TARGET_1 );

        // モデル描画
        effectDemo->ProcModelDraw( *drawParam );
#endif
    }

    //-----------------------------------------------------------
    // カラーコピー前 + 不透明　描画パス　描画処理
    //-----------------------------------------------------------
    if ( isRenderingPreColorCopyBuffer || isRenderingShadowMapBuffer )
    {
        effectDemo->ProcDraw( *drawParam, static_cast<nw::eft2::DrawPathFlag>( DRAW_PATH_PRE_COPY_COLOR_BUFFER | DRAW_PATH_OPAQU_BUFFER ) );
    }

#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_BLOCK );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

    Demo_ProcDraw();

#if EFT_IS_CAFE
    GX2SetShaderMode( GX2_SHADER_MODE_UNIFORM_REGISTER );
    GX2Invalidate( static_cast<GX2InvalidateType>( GX2_INVALIDATE_SHADER ), NULL, 0xFFFFFFFF );
#endif

    //-----------------------------------------------------------
    // 「コピー前」描画パス以前で描画されたバッファをコピーして
    // エフェクトランタイムに設定する
    //-----------------------------------------------------------
    if ( makeFrameBufferCopy )
    {
        currentFrameBuffer->CopyFrameBufferTexture( &gFrameBufferTextureCopy );
        currentFrameBuffer->CopyDepthBufferTexture( &gDepthBufferTextureCopy );
        currentFrameBuffer->Bind();

#if defined(NW_PLATFORM_WIN32)
        g_EftSystem->SetFrameBufferTexture( gFrameBufferTextureCopy.GetGLTextureID() );
        g_EftSystem->SetDepthBufferTexture( gDepthBufferTextureCopy.GetGLTextureID() );
#endif
#if defined(NW_PLATFORM_CAFE)
        g_EftSystem->SetFrameBufferTexture( gFrameBufferTextureCopy.GetGX2Texture() );
        g_EftSystem->SetDepthBufferTexture( gDepthBufferTextureCopy.GetGX2Texture() );
#endif
    }

    //-----------------------------------------------------------
    // ここまでの描画結果を縮小バッファにコピー(Zのみ)
    //-----------------------------------------------------------
    if ( isRenderingReductionBuffer )
    {
        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );

        // 縮小バッファへデプスバッファをコピーする
        gReductionFrameBuffer.Bind();

        graphics->SetDepthEnable( true, true );
        graphics->SetColorMask( true, true, true, true );
        gReductionFrameBuffer.Clear( true, true );

        graphics->SetColorMask( false, false, false, false );
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_DEPTH_COPY, &gDepthBufferTextureCopy );

        graphics->SetColorMask( true, true, true, true );
        graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_BACK );
    }

    //-----------------------------------------------------------
    // デフォルト描画パス 描画
    //-----------------------------------------------------------
    currentFrameBuffer->Bind();
    effectDemo->ProcDraw( *drawParam, DRAW_PATH_DEFAULT );

    //----------------------------------------
    // 「縮小バッファ」描画パス 描画処理
    //----------------------------------------
    if ( isRenderingReductionBuffer )
    {
        // 縮小バッファに切り替えて
        gReductionFrameBuffer.Bind();

        // シェーダタイプを縮小シェーダに切り替える
        effectDemo->SetShaderType( nw::eft2::EFT_SHADER_TYPE_PATH_DEF1 );

        // 縮小バッファ 描画パスを描画
        effectDemo->ProcDraw( *drawParam, DRAW_PATH_REDUCTION_BUFFER );

        // シェーダタイプを縮小シェーダに切り替える
        effectDemo->SetShaderType( nw::eft2::EFT_SHADER_TYPE_NORMAL );

        // 標準フレームバッファに戻す。
        currentFrameBuffer->Bind();

        // 縮小バッファの内容を標準フレームバッファに書き戻す
        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->SetBlendEnable( true );
        graphics->SetDepthEnable( false, false );
        graphics->SetBlendEquation( nw::gfnd::Graphics::BLEND_EQUATION_ADD );
        graphics->SetBlendFactorSeparate(   nw::gfnd::Graphics::BLEND_FACTOR_ONE,
            nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA,
            nw::gfnd::Graphics::BLEND_FACTOR_ZERO,
            nw::gfnd::Graphics::BLEND_FACTOR_ONE );
        graphics->SetAlphaTestEnable( false );
        drawer->DrawScreen( nw::eftdemo::ScreenDrawer::SHADER_TYPE_COPY, &gReductionFrameBuffer );
        graphics->SetAlphaTestEnable( true );
    }

    //----------------------------------------
    // 「法線バッファ」描画処理
    //----------------------------------------
    if ( drawNormalBuffer )
    {
        graphics->SetDepthTestEnable( false );
        graphics->SetAlphaTestEnable( false );
        graphics->SetBlendEnable( false );

        drawer->DrawWipe( &gNormalFrameBuffer, 0.4f, 0.4f, 0.9f, 0.9f );

        graphics->SetDepthTestEnable( true );
        graphics->SetAlphaTestEnable( true );
        graphics->SetBlendEnable( true );
    }

    //----------------------------------------
    // 「キューブマップテクスチャ」描画処理
    //----------------------------------------
    if ( drawCubeMapTexture )
    {
        graphics->SetDepthTestEnable( false );
        graphics->SetAlphaTestEnable( false );
        graphics->SetBlendEnable( false );

        drawer->DrawCubeMapTexture( gCubeMap.GetCmTextureArray(), 0.0f, -1.0f, 1.0f, 0.0f );

        graphics->SetDepthTestEnable( true );
        graphics->SetAlphaTestEnable( true );
        graphics->SetBlendEnable( true );
    }

    //----------------------------------------
    // グレア処理
    //----------------------------------------
    if ( drawGlareNum != 0 )
    {
        gGlareEffect.Draw( drawer, currentFrameBuffer, drawGlareNum );
    }

    // レンダリングステートを戻す
    graphics->SetCullingMode( nw::gfnd::Graphics::CULLING_MODE_NONE );
    graphics->SetBlendFactorSeparate( nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA,
        nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA,
        nw::gfnd::Graphics::BLEND_FACTOR_SRC_ALPHA,
        nw::gfnd::Graphics::BLEND_FACTOR_INV_SRC_ALPHA );

    gSystem->GetMeterDraw().EndMeasure();
} // NOLINT(readability/fn_size)


//------------------------------------------------------------------------------
//　計算処理を行う
//------------------------------------------------------------------------------
f32 CalcProcess( EffectDemo* effectDemo, nw::eftdemo::DrawParam* drawParam )
{
    //----------------------------------------
    // エフェクトデモ計算処理
    //----------------------------------------
    NW_GL_ASSERT();
    gSystem->GetMeterCalc().BeginMeasure();
    {
        effectDemo->ProcCalc( gSystem->GetPad(), *drawParam );

        Demo_ProcCalc();
        NW_GL_ASSERT();
    }
    gSystem->GetMeterCalc().EndMeasure();
    NW_GL_ASSERT();

    //----------------------------------------
    // エフェクトデモ計算処理　コスト平均値を求める
    //----------------------------------------
    gCpuCostTick[gCpuCostTickCounter] = gSystem->GetMeterCalc().GetLastTotalSpan();
    gCpuCostAverage = nw::ut::Tick(0);
    for ( u32 i = 0; i < CPU_COST_TICK_MAX; ++i )
    {
        gCpuCostAverage += gCpuCostTick[i];
    }
    gCpuCostAverageNanoSec = gCpuCostAverage.ToTimeSpan().GetNanoSeconds() / CPU_COST_TICK_MAX;
    gCpuCostTickCounter++;
    if ( gCpuCostTickCounter == CPU_COST_TICK_MAX )
    {
        gCpuCostTickCounter = 0;
    }
    NW_GL_ASSERT();
    return (f32)gCpuCostAverageNanoSec / 1000000.f;
}



//------------------------------------------------------------------------------
//      Main 関数
//------------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
int PCSDKMain(int argc, char **argv)
#elif defined(NW_PLATFORM_CAFE)
int main(int argc, char **argv)
#endif
{
    NW_UNUSED_VARIABLE( argc );
    NW_UNUSED_VARIABLE( argv );

#if ( EFT_IS_WIN && _DEBUG )
    _CrtSetDbgFlag(_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF);
#endif

    EffectDemo eftDemo;

    // アロケーターの初期化
    enum
    {
        HEAP_SIZE = ( 512 * 1024 * 1024 )
    };
#if defined(NW_PLATFORM_WIN32)
    void* addr = malloc( HEAP_SIZE );
#else
    void* addr = MEMAllocFromDefaultHeap( HEAP_SIZE );
#endif
    gNwAllocator.Initialize( addr, HEAP_SIZE );

#if defined(NW_PLATFORM_CAFE)
    // ビューアデモ用の通信初期化
    HIOInit();

#if defined( NW_USE_CPU_PROFILER )
    // CPUプロファイラの初期化
    void *buffer = MEMAllocFromDefaultHeap( PROFILER_MINBUFFERSIZE );
    BOOL profInitResult = PROFILERInitialize( buffer, PROFILER_MINBUFFERSIZE );
    PROFILERHeartbeat( PROFILEREventHeartbeatName_Main, PROFILEREventHeartbeatType_Beat );
#endif

#if defined( NW_USE_GPU_PROFILER )
    // GPUプロファイラの初期化
    GX2DebugCaptureInit( NULL );
#endif

    // TODO : 実機版の場合は、MCSを利用してマウスをPCマウスと連動させる

#endif

    // エフェクトビューアのファイルシステムにHeapを設定する。
#if defined( NW_PLATFORM_CAFE )
    nw::eftvw::FileSystem::SetAllocator( &gNwAllocator );
#endif

    //----------------------------------------
    // フレームワーク初期化処理
    //----------------------------------------
    Initialize();

    //----------------------------------------
    // フレームバッファの初期化
    //----------------------------------------
    f32 width  = static_cast<f32>( gSystem->GetWidth() );
    f32 height = static_cast<f32>( gSystem->GetHeight() );

#if defined(NW_PLATFORM_WIN32)
    // スクリーンキャプチャ用のフォルダを削除
    RemoveDirectory( L"ScreenCap" );

    // 起動フォルダパスを取得
    TCHAR baseDirectory[_MAX_PATH];
    GetCurrentDirectory( _MAX_PATH, baseDirectory );
#endif

    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
#if defined( NW_PLATFORM_CAFE )
        nw::ut::MemoryAllocator* allocator = gSystem->GetMem1Allocator();
#else
        nw::ut::MemoryAllocator* allocator = &gNwAllocator;
#endif
        // 標準フレームバッファ Float
        gFrameBuffer64.Initialize( allocator, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        // 標準フレームバッファ RGB10A2
        gFrameBuffer32.Initialize( allocator, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGB10A2 );

        // 法線フレームバッファ
        gNormalFrameBuffer.Initialize( allocator, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGB10A2 );

        // 標準のフレームバッファからコピー用テクスチャ
        gFrameBufferTextureCopy.Initialize(  allocator, &gFrameBuffer64, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGB16 );
        gDepthBufferTextureCopy.Initialize( allocator, &gFrameBuffer64 );

        // デプスシャドウ用のバッファ
        gLightFrameBuffer.Initialize( &gNwAllocator, width, height, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_RGB16 );

        // 縮小フレームバッファ
        gReductionFrameBuffer.Initialize( &gNwAllocator, width / 2.f, height / 2.f, nw::eftdemo::FrameBuffer::FRAMEBUFFER_TYPE_FLOAT16 );

        // シャドウマップ用テクスチャ
        gShadowMapTextureCopy.Initialize( &gNwAllocator, &gLightFrameBuffer );

        // グレア用のバッファ
        gGlareEffect.Initialize( &gNwAllocator, width, height );

        // キューブマップ用のバッファ
        gCubeMap.Initialize( &gNwAllocator, 64, 64, DrawCubeMap );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    //----------------------------------------
    // カメラを初期化
    //----------------------------------------
    gCameraPosition.Set( 0.0f, 50.0f, 150.0f );
    gCameraLookAt.Set( 0.0f, 00.0f, 0.0f );
    gDemoCamera.SetPos( gCameraPosition );
    gDemoCamera.SetLookAtPos( gCameraLookAt );
    gDemoCamera.Preset();

    //----------------------------------------
    // プロジェクションの初期化
    //----------------------------------------
    nw::math::MTX44Perspective( &gProjctionMatrix, nw::math::F_PI / 3, width / (f32)height, 0.1f, 100000.f);

    //----------------------------------------
    // 簡易デバッグメニュー
    //----------------------------------------
    nw::eftdemo::DebugMenu debugMenu;
    debugMenu.SetMenuName( "EFT DEBUG MENU");

    //------------------------------------------------------------
    nw::eftdemo::DebugMenuPage viewerPage;
    viewerPage.SetPageName( "SCENE" );

    nw::eftdemo::DebugMenuItemOnOff displayListMode;
    displayListMode.SetItemName( "DisplayList" );
    displayListMode.SetValue( 0 );

    if ( eftDemo.EnabledDisplayList() )
    {
        viewerPage.AddMenuItem( &displayListMode );
    }

    nw::eftdemo::DebugMenuItemOnOff cacheFlush;
    cacheFlush.SetItemName( "CacheInvalidate" );
    viewerPage.AddMenuItem( &cacheFlush );

    nw::eftdemo::DebugMenuItemFloatParam cpuCost;
    cpuCost.SetItemName( "CpuCost" );
    viewerPage.AddMenuItem( &cpuCost );

    //------------------------------------------------------------
    nw::eftdemo::DebugMenuPage envPage;
    envPage.SetPageName( "ENV" );

    nw::eftdemo::DebugMenuItemFloatParam camPosX;
    nw::eftdemo::DebugMenuItemFloatParam camPosY;
    nw::eftdemo::DebugMenuItemFloatParam camPosZ;
    nw::eftdemo::DebugMenuItemFloatParam camTarX;
    nw::eftdemo::DebugMenuItemFloatParam camTarY;
    nw::eftdemo::DebugMenuItemFloatParam camTarZ;

    nw::eftdemo::DebugMenuItemFloatParam lightPosX;
    nw::eftdemo::DebugMenuItemFloatParam lightPosY;
    nw::eftdemo::DebugMenuItemFloatParam lightPosZ;

    camPosX.SetItemName( "CamPosX" );
    envPage.AddMenuItem( &camPosX );
    camPosY.SetItemName( "CamPosY" );
    envPage.AddMenuItem( &camPosY );
    camPosZ.SetItemName( "CamPosZ" );
    envPage.AddMenuItem( &camPosZ );

    camTarX.SetItemName( "CamTarX" );
    envPage.AddMenuItem( &camTarX );
    camTarY.SetItemName( "CamTarY" );
    envPage.AddMenuItem( &camTarY );
    camTarZ.SetItemName( "CamTarZ" );
    envPage.AddMenuItem( &camTarZ );

    lightPosX.SetItemName( "LhtPosX" );
    envPage.AddMenuItem( &lightPosX );
    lightPosY.SetItemName( "LhtPosY" );
    envPage.AddMenuItem( &lightPosY );
    lightPosZ.SetItemName( "LhtPosZ" );
    envPage.AddMenuItem( &lightPosZ );


    //------------------------------------------------------------
    nw::eftdemo::DebugMenuPage bufferPage;
    bufferPage.SetPageName( "BUFFER" );

    nw::eftdemo::DebugMenuItem frameBufferSetting;
    frameBufferSetting.SetItemName( "FrameBufferSetting" );
    frameBufferSetting.AddItemParam( "FLOAT16",    0 );
    frameBufferSetting.AddItemParam( "RGB10 A2",   1 );
    frameBufferSetting.SetValue( 0 );
    bufferPage.AddMenuItem( &frameBufferSetting );

    nw::eftdemo::DebugMenuItemOnOff makeShadowMap;
    makeShadowMap.SetItemName( "Make Shadow Map" );
    makeShadowMap.SetValue( 1 );
    bufferPage.AddMenuItem( &makeShadowMap );

    nw::eftdemo::DebugMenuItemOnOff makeFrameBufferCopy;
    makeFrameBufferCopy.SetItemName( "Make Frame Buffer Copy" );
    makeFrameBufferCopy.SetValue( true );
    bufferPage.AddMenuItem( &makeFrameBufferCopy );

    nw::eftdemo::DebugMenuItem drawGlareBuffer;
    drawGlareBuffer.SetItemName( "Draw Glare Buffer" );
    drawGlareBuffer.AddItemParam( "OFF", 0 );
    drawGlareBuffer.AddItemParam( "1",   1 );
    drawGlareBuffer.AddItemParam( "2",   2 );
    drawGlareBuffer.AddItemParam( "3",   3 );
    drawGlareBuffer.AddItemParam( "4",   4 );
    drawGlareBuffer.AddItemParam( "5",   5 );
    drawGlareBuffer.SetValue( 0 );

    bufferPage.AddMenuItem( &drawGlareBuffer );

    nw::eftdemo::DebugMenuItemOnOff drawNormalBuffer;
    drawNormalBuffer.SetItemName( "Draw Normal Buffer" );
    drawNormalBuffer.SetValue( false );
    bufferPage.AddMenuItem( &drawNormalBuffer );

    nw::eftdemo::DebugMenuItemOnOff drawCubeMapTexture;
    drawCubeMapTexture.SetItemName( "Draw CubeMap Texture" );
    drawCubeMapTexture.SetValue( false );
    bufferPage.AddMenuItem( &drawCubeMapTexture );

    //TODO:デモの初期設定を引き継ぐ

    //------------------------------------------------------------
    // ページを追加
    debugMenu.AddMenuPage( &viewerPage );
    debugMenu.AddMenuPage( &bufferPage );
    debugMenu.AddMenuPage( &envPage );


    //------------------------------------------------------------
    // Cpuコストカウンタ
    //------------------------------------------------------------
    gCpuCostTickCounter = 0;

    //------------------------------------------------------------
    // リニアモード切替
    //------------------------------------------------------------
    gSystem->SetScanBuffer( eftDemo.GetLinearMode() );

    //----------------------------------------
    // 簡易プリミティブ描画クラス
    //----------------------------------------
    nw::eftdemo::ScreenDrawer drawer;
    drawer.Initialize( &gNwAllocator );

    //----------------------------------------
    // ディスプレイリスト初期化
    //----------------------------------------
    gDisplayList[0].Initialize( &gNwAllocator, 0x400000 );
    gDisplayList[1].Initialize( &gNwAllocator, 0x400000 );

    //----------------------------------------
    // エフェクトデモの初期化
    //----------------------------------------
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        eftDemo.Initialize( &gNwAllocator, &debugMenu );
        eftDemo.SetCamera( &gDemoCamera );
        eftDemo.SetProjectionMatrix( &gProjctionMatrix );

        Demo_Initialize( &gNwAllocator, eftDemo.GetEffectSystem(), eftDemo.GetEffectViewerSystem() );
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

#if defined(NW_PLATFORM_WIN32)
    // FPS設定
    eftDemo.SetFrameRate( gSystem->GetCreateArg().fps );
#endif

    //----------------------------------------
    // デバッグUI初期化処理
    //----------------------------------------
    if( eftDemo.DisplayWindow() )
    {
        s_DWMgr.Initialize( width, height, &gNwAllocator );
        s_TextRenderer.Initialize(  gNwAllocator,
            gSystem->GetCreateArg().fontBinary,
            gSystem->GetCreateArg().fontBinarySize,
#if defined( NW_PLATFORM_CAFE )
            gSystem->GetCreateArg().fontShaderBinary,
            gSystem->GetCreateArg().fontShaderBinarySize );
#else
            NULL,
            0 );
#endif
        s_UIRenderer.SetTextRenderer( &s_TextRenderer );
        s_UIRenderer.SetPrimitiveRenderer( gPrimitiveRenderer );

        nw::math::Matrix44 projectionMatrix;
        projectionMatrix.SetOrtho(0.f, width, height, 0.f, 0.0f, 1.f);

        s_UIRenderer.SetProjectionMatrix( projectionMatrix );
        s_UIRenderer.SetViewMatrix( nw::math::Matrix34::Identity() );

           // パフォーマンス解析初期化
        nw::eftdemo::InitializePerfAnalyzer( &gNwAllocator, &s_DWMgr, width, height );

        // コストメータの表示を半分に。
        gSystem->GetMeterDrawer().SetWidth( 400.f );
    }

    //----------------------------------------
    // メインループ
    //----------------------------------------

    // ライトパラメータ
    nw::eftdemo::DrawParam lightParam;
    lightParam.mCameraPos.x = 0.0f;
    lightParam.mCameraPos.y = 100.f;
    lightParam.mCameraPos.z = 100.f;

    while ( !gSystem->IsExiting() )
    {
#if defined(NW_PLATFORM_CAFE)
        if ( cacheFlush.GetValue() )
        {
            DCFlushRange( addr, HEAP_SIZE );
        }
#endif

        // dw Calc
        s_DWMgr.Update( s_UIRenderer );

        // demo Pad では駄目。。。
        s_DWMgr.UpdateInputs( nw::internal::dw::Inputs( gSystem->GetPad(), NULL ) );

        // システム フレームの開始処理
        // コストメーターのフレーム開始処理。
        gSystem->BeginFrame();

        // システム デモパッドの更新
        gSystem->UpdateInputInterface();

        // カメラの更新
#if defined(NW_PLATFORM_CAFE)
        gDemoCamera.UpdateCamera( gSystem->GetPad(), NULL, true, true );
#endif
#if defined(NW_PLATFORM_WIN32)
        gDemoCamera.UpdateCamera( gSystem->GetPad(), gSystem->GetMouse(), gSystem->GetMouse()->IsPointerOn(), true );
#endif

        // デバッグ描画のオンオフl
        bool trigL = gSystem->GetPad()->IsRelease( nw::demo::Pad::MASK_L );
        bool holdL = gSystem->GetPad()->IsHold( nw::demo::Pad::MASK_L );
        bool holdR = gSystem->GetPad()->IsHold( nw::demo::Pad::MASK_R );

        //----------------------------------------
        // 描画パラメータ更新
        //----------------------------------------
        // カメラ位置更新
        gDemoCamera.GetPos( &gCameraPosition );
        gDemoCamera.GetLookAtPos( &gCameraLookAt );

        // メニューにカメラ位置を反映
        camPosX.SetFloatParam( gCameraPosition.x );
        camPosY.SetFloatParam( gCameraPosition.y );
        camPosZ.SetFloatParam( gCameraPosition.z );

        camTarX.SetFloatParam( gCameraLookAt.x );
        camTarY.SetFloatParam( gCameraLookAt.x );
        camTarZ.SetFloatParam( gCameraLookAt.x );

        nw::eftdemo::DrawParam drawParam;
        {
            drawParam.mProjMtx           = gProjctionMatrix;
            drawParam.mViewMtx           = gDemoCamera.GetMatrix();
            drawParam.mCameraPos         = gCameraPosition;
            drawParam.mNearZ             = 0.1f;
            drawParam.mFarZ              = 10000.f;
            drawParam.mPrimitiveRenderer = gPrimitiveRenderer;
            drawParam.mShadowMap         = 0;
//          drawParam.mGpuProfiler       = nw::eftdemo::PerfAnalyzerCallBack;
        }


        {
            lightParam.mProjMtx           = gProjctionMatrix;
            lightParam.mNearZ             = 0.1f;
            lightParam.mFarZ              = 10000.f;
            lightParam.mPrimitiveRenderer = gPrimitiveRenderer;
            lightParam.mShadowMap         = 0;
//            lightParam.mGpuProfiler       = NULL;
        }

        // ライト更新
        if ( trigL )
        {
            lightParam.mCameraPos.x = gCameraPosition.x;
            lightParam.mCameraPos.y = gCameraPosition.y;
            lightParam.mCameraPos.z = gCameraPosition.z;

            lightPosX.SetFloatParam( lightParam.mCameraPos.x );
            lightPosY.SetFloatParam( lightParam.mCameraPos.y );
            lightPosZ.SetFloatParam( lightParam.mCameraPos.z );

            lightParam.mViewMtx     = gDemoCamera.GetMatrix();
        }

        // 書き込み先フレームバッファの選定
        nw::eftdemo::FrameBuffer* currentFrameBuffer = NULL;
        if ( frameBufferSetting.GetValue() == 0 )
        {
            currentFrameBuffer = &gFrameBuffer64;
        }
        else
        {
            currentFrameBuffer = &gFrameBuffer32;
        }

        // リニア <-> ノンリニア　切り替え
        {
            static bool linearModeFlag = true;
            if( eftDemo.GetLinearMode() != linearModeFlag )
            {
                gSystem->SetScanBuffer( eftDemo.GetLinearMode() );
                linearModeFlag = eftDemo.GetLinearMode();
            }
        }

        nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();
        graphics->LockDrawContext();

        // GPUコスト計測開始
        gSystem->GetMeterGPU().BeginMeasure();

        // Gpuカウンタリセット
        nw::eftdemo::ResetPerfAnalyzer();

        if ( displayListMode.IsEnable() )
        {
            // DL実行
            gDisplayList[gUseDisplayListSide].CallDisplayList();
            gUseDisplayListSide = 1 - gUseDisplayListSide;

            // 計算処理
            f32 cost = CalcProcess( &eftDemo, &drawParam );
            cpuCost.SetFloatParam( cost );

            // DL作成開始
            gDisplayList[gUseDisplayListSide].BindDisplayList( false );

            // 描画コマンドを発行する
            MakeDrawCommand( &eftDemo, &drawParam, &lightParam, &drawer, currentFrameBuffer,
                eftDemo.GetGridType(), makeFrameBufferCopy.IsEnable(), makeShadowMap.IsEnable(), drawGlareBuffer.GetValue(),
                drawNormalBuffer.IsEnable(), drawCubeMapTexture.IsEnable() );

            // DL作成終了
            gDisplayList[gUseDisplayListSide].UnbindDisplayList();
        }
        else
        {
            // 計算処理
            f32 cost = CalcProcess( &eftDemo, &drawParam );
                        NW_GL_ASSERT();
            cpuCost.SetFloatParam( cost );
                        NW_GL_ASSERT();

            // 描画コマンドを発行する
            MakeDrawCommand( &eftDemo, &drawParam, &lightParam, &drawer, currentFrameBuffer,
                eftDemo.GetGridType(), makeFrameBufferCopy.IsEnable(), makeShadowMap.IsEnable(), drawGlareBuffer.GetValue(),
                drawNormalBuffer.IsEnable(), drawCubeMapTexture.IsEnable() );
        }

        // GPUコスト計測終了
        gSystem->GetMeterGPU().EndMeasure();


        // アルファテストをデフォルトの状態に戻しておく
#if defined(NW_PLATFORM_WIN32)
        glEnable(  GL_ALPHA_TEST );
        glAlphaFunc( GL_GREATER, 0.0 );
#endif
#if defined(NW_PLATFORM_CAFE)
        GX2SetAlphaTest( GX2_ENABLE, GX2_COMPARE_GREATER , 0.0 );
#endif

#if defined(NW_PLATFORM_WIN32)
        // Rボタン(キー)でスクリーンキャプチャを行う
        static u32 s_CaptureIndex = 0;

        u32 uWidth  = static_cast<u32>( width );
        u32 uHeight = static_cast<u32>( height );

        if ( holdR )
        {
            WIN32_FIND_DATA fd = {0};

            SetCurrentDirectory( baseDirectory );
            if ( FindFirstFile( L"ScreenCap", &fd ) == INVALID_HANDLE_VALUE )
            {
               CreateDirectory( L"ScreenCap", NULL );
            }
            SetCurrentDirectory( L"ScreenCap" );

            if ( s_CaptureIndex == 0 )
            {
                // *.bmpファイルを削除する
                {
                    wchar_t serchBuf[MAX_PATH];
                    HANDLE hFind = INVALID_HANDLE_VALUE;

                    wsprintf( serchBuf, L"%s", L"*.bmp" );
                    hFind = ( FindFirstFile( serchBuf, &fd ) );
                    if ( hFind != INVALID_HANDLE_VALUE )
                    {
                        do
                        {
                            if ( !(fd.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY ) )
                            {
                                if ( wcsstr( fd.cFileName, L".bmp" ) != NULL )
                                {
                                    DeleteFile( fd.cFileName );
                                }
                            }
                        } while( FindNextFile( hFind, &fd ) );
                    }
                    FindClose( hFind );
                }
            }
            if ( s_CaptureIndex % 4 == 0 || s_CaptureIndex == 0 )
            {
                char fileName[_MAX_PATH];
                sprintf_s( fileName, _MAX_PATH, "ScreenCap_%8d.bmp", s_CaptureIndex );
                SaveCurrentBufferAsBmpFile( fileName, uWidth, uHeight, eftDemo.GetLinearMode() );
            }
            s_CaptureIndex++;

            // baseDirectoryに戻します。
            SetCurrentDirectory( baseDirectory );
        }
        else
        {
            s_CaptureIndex = 0;
        }
#endif

        //----------------------------------------
        // 簡易デバッグメニュー
        //----------------------------------------
        if ( !holdL )
        {
            // 負荷計測メーターを描画します。
            gSystem->DrawLoadMeters();

            debugMenu.Behavior( gSystem->GetPad() );
            f32 posX     = eftDemo.GetDebugMenuPosX();
            f32 posY     = eftDemo.GetDebugMenuPosY();
            debugMenu.WritePage( gSystem->GetTextWriter(), posX, posY );
        }

        //----------------------------------------
        // デモデバッグ文字描画処理
        //----------------------------------------
        if ( !holdL )
        {
#if defined(NW_PLATFORM_WIN32)
            glEnable(  GL_BLEND );
            glBlendFunc( GL_SRC_ALPHA , GL_ONE );
            glBlendEquationEXT(GL_FUNC_ADD_EXT);
            glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );
#endif
            eftDemo.DebugDraw( gSystem->GetTextWriter() );

            gSystem->GetTextWriter()->Flush();
        }

        //----------------------------------------
        // デバッグUI描画処理
        //----------------------------------------
        if( eftDemo.DisplayWindow() )
        {
            s_UIRenderer.ClearBuffer();
            s_DWMgr.Draw(s_UIRenderer);        // バッファのスワップ
        }

        graphics->UnlockDrawContext();

        // バッファのスワップ
        gSystem->SwapBuffer( currentFrameBuffer );

        // フレームの終了処理
        // テキストライターやコストメーターのフレーム終了処理。
        gSystem->EndFrame();

        //----------------------------------------
        // フレームレート切り替え
        //----------------------------------------
        {
            f32 fps;
            eftDemo.GetFrameRate( &fps );

#if defined(NW_PLATFORM_WIN32)
            gSystem->SetFps(fps);
#else
            u32 vBlank = 1;

            if ( fps == 60.f )
            {
                vBlank = 1;
            }
            if ( fps == 30.f )
            {
                vBlank = 2;
            }
            if ( fps == 15.f )
            {
                vBlank = 3;
            }

            gSystem->SetWaitVBlankInterval( vBlank );
            GX2SetSwapInterval( vBlank );
#endif
        }

        if( eftDemo.DisplayWindow() )
        {

            nw::eft2::EmitterSet* set2 = g_EftSystem->GetEmitterSetHead( 63 );
            u32 numLines = nw::eftdemo::ProcessPerfAnalyzer( set2 );
        }

        //  VBlank 待ち
        gSystem->WaitForFlip();
    }


    //----------------------------------------
    // エフェクトデモ終了処理
    //----------------------------------------
    nw::gfnd::Graphics::GetInstance()->LockDrawContext();
    {
        Demo_Finalize();
        eftDemo.Finalize();

        // ライトビュー用のフレームバッファを破棄
        gLightFrameBuffer.Finalize( &gNwAllocator );

        drawer.Finalize();
    }
    nw::gfnd::Graphics::GetInstance()->UnlockDrawContext();

    // デバックUI終了処理
    if( eftDemo.DisplayWindow() )
    {
        s_DWMgr.Finalize();

        s_TextRenderer.Finalize();
    }

    // フレームワークの終了処理
    Finalize();

    // アロケータ終了処理
    gNwAllocator.Finalize();
#if defined(NW_PLATFORM_WIN32)
    free( addr );
#else
    MEMFreeToDefaultHeap( addr );
#endif

    return 0;
} // NOLINT(readability/fn_size)
