﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nw/types.h>
#include <nw/eft2.h>

namespace nw      {
namespace eftdemo {


//---------------------------------------------------------------------------
//! @brief      テクスチャクラス
//---------------------------------------------------------------------------
class Texture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    Texture()
    {
        mInitialized = false;
        mAllocator   = NULL;
#if defined(NW_PLATFORM_CAFE)
        mGx2Texture.surface.imagePtr = NULL;
#endif
    }

    //---------------------------------------------------------------------------
    //! @brief        デストラクタです。
    //---------------------------------------------------------------------------
    ~Texture()
    {
        Finalize( mAllocator );
    }

    //---------------------------------------------------------------------------
    //! @brief        画像を破棄します。
    //---------------------------------------------------------------------------
    void Finalize( nw::ut::IAllocator* allocator )
    {
        if ( !allocator )
        {
            return;
        }

#if defined(NW_PLATFORM_WIN32)
        // 作成済みのテクスチャを破棄
        if ( mTextutreID != 0 )
        {
            glDeleteTextures(1, &mTextutreID );
        }
#endif
#if defined(NW_PLATFORM_CAFE)
        // 作成済みのテクスチャを破棄
        if ( mGx2Texture.surface.imagePtr != NULL )
        {
            allocator->Free( mGx2Texture.surface.imagePtr );
            mGx2Texture.surface.imagePtr = NULL;
        }
#endif
        mInitialized = false;
    }

    //---------------------------------------------------------------------------
    //! @brief        画像が初期化済みかどうか。
    //---------------------------------------------------------------------------
    bool IsInitialized()
    {
        return mInitialized;
    }

    //---------------------------------------------------------------------------
    //! @brief        テクスチャサイズを取得します。
    //---------------------------------------------------------------------------
    nw::math::VEC2 GetSize() const
    {
        return mSize;
    }

    //---------------------------------------------------------------------------
    //! @brief        テクスチャを取得します。
    //---------------------------------------------------------------------------
#if defined(NW_PLATFORM_CAFE)
    GX2Texture* GetGX2Texture()
    {
        return &mGx2Texture;
    }
#endif
#if defined(NW_PLATFORM_WIN32)
    GLuint GetGLTextureID()
    {
        return mTextutreID;
    }
#endif

    //---------------------------------------------------------------------------
    //! @brief        nw::eft::Texture を取得します。
    //---------------------------------------------------------------------------
//    nw::eft2::TextureResource GetTexture() const
//    {
//#if defined(NW_PLATFORM_CAFE)
//        return &mGx2Texture;
//#endif
//#if defined(NW_PLATFORM_WIN32)
//        return mTextutreID;
//#endif
//    }

protected:
#if defined(NW_PLATFORM_WIN32)
    GLuint                          mTextutreID;
#endif
#if defined(NW_PLATFORM_CAFE)
    GX2Texture                      mGx2Texture;
#endif
    bool                            mInitialized;           //!< 初期化済みかどうか。
    nw::ut::IAllocator*             mAllocator;             //!< 初期化時に渡されたアロケータ
    nw::math::VEC2                  mSize;                  //!< フレームバッファサイズです。
};


//---------------------------------------------------------------------------
//! @brief      背景画像用テクスチャクラス
//---------------------------------------------------------------------------
class BackgroundTexture : public Texture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        引数で指定されるイメージからテクスチャを作成します。
    //---------------------------------------------------------------------------
    void SetImage( nw::ut::MemoryAllocator* allocator, u32 width, u32 height, const u8* pixels )
    {
        // 背景画像はリロードがかかる為、一度破棄。
        Finalize( mAllocator );

        if (width == 0 || height == 0)
        {
            return;
        }

        mSize.x = static_cast<f32>( width );
        mSize.y = static_cast<f32>( height );

#if defined(NW_PLATFORM_WIN32)
        glGenTextures( 1, &mTextutreID );
        glBindTexture( GL_TEXTURE_2D, mTextutreID );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );
#endif
#if defined(NW_PLATFORM_CAFE)
        memset( &mGx2Texture, 0, sizeof(GX2Texture) );
        GX2InitTexture( &mGx2Texture, width, height, 1, 0, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_SURFACE_DIM_2D );
        u32* texturePtrs = (u32*)allocator->Alloc( mGx2Texture.surface.imageSize, mGx2Texture.surface.alignment );
        GX2InitTexturePtrs( &mGx2Texture, texturePtrs, 0 );
        GX2TileTexture( &mGx2Texture, (void*)pixels );
#endif
        mInitialized = true;
        mAllocator   = allocator;
    }
};


//---------------------------------------------------------------------------
//! @brief      ライトマップ用テクスチャクラス
//---------------------------------------------------------------------------
class LightMapTexture : public Texture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        引数で指定されるイメージからライトマップイメージを作成します。
    //---------------------------------------------------------------------------
    void CreateImage( nw::ut::IAllocator* allocator,
                      u32 width, u32 height,
                      nw::math::VEC3 color0, f32 color0In,
                      nw::math::VEC3 color1, f32 color1In,
                      nw::math::VEC3 color2, f32 color2In )
    {
        Finalize( mAllocator );

        if (width == 0 || height == 0)
        {
            return;
        }

        mSize.x = static_cast<f32>( width );
        mSize.y = static_cast<f32>( height );

        // イメージ配列を作成
        u32* pixels = static_cast<u32*>( allocator->Alloc( width * height * 4 ) );
        if ( !pixels )
        {
            return;
        }
        memset( pixels, 0, width * height * 4 );

        nw::math::VEC3 delta10;
        nw::math::VEC3 delta21;

        f32 dt10 = ( color1In - color0In ) * height;
        f32 dt21 = ( color2In - color1In ) * height;

        delta10.Set(( color1.x - color0.x ) / ( dt10 ),
                    ( color1.y - color0.y ) / ( dt10 ),
                    ( color1.z - color0.z ) / ( dt10 ) );
        delta21.Set(( color2.x - color1.x ) / ( dt21 ),
                    ( color2.y - color1.y ) / ( dt21 ),
                    ( color2.z - color1.z ) / ( dt21 ) );

        nw::ut::Color4u8 colotU8;

        for ( u32 i = 0; i < height; i++ )
        {
            f32 rate = (f32)i / height;

            if ( rate < color0In )
            {
                colotU8.r = static_cast<u8>( color0.x * 255.f );
                colotU8.g = static_cast<u8>( color0.y * 255.f );
                colotU8.b = static_cast<u8>( color0.z * 255.f );
                colotU8.a = 255;
            }else
            if ( color0In <= rate && rate < color1In )
            {
                colotU8.r = static_cast<u8>( ( color0.x + delta10.x * ( i - color0In * height ) ) * 255.f );
                colotU8.g = static_cast<u8>( ( color0.y + delta10.y * ( i - color0In * height ) ) * 255.f );
                colotU8.b = static_cast<u8>( ( color0.z + delta10.z * ( i - color0In * height ) ) * 255.f );
                colotU8.a = 255;
            }else
            if ( color1In <= rate && rate < color2In )
            {
                colotU8.r = static_cast<u8>( ( color1.x + delta21.x * ( i - color1In * height ) ) * 255.f );
                colotU8.g = static_cast<u8>( ( color1.y + delta21.y * ( i - color1In * height ) ) * 255.f );
                colotU8.b = static_cast<u8>( ( color1.z + delta21.z * ( i - color1In * height ) ) * 255.f );
                colotU8.a = 255;
            }
            else
            {
                colotU8.r = static_cast<u8>( color2.x * 255.f );
                colotU8.g = static_cast<u8>( color2.y * 255.f );
                colotU8.b = static_cast<u8>( color2.z * 255.f );
                colotU8.a = 255;
            }

            for ( u32 j = 0; j < width; j++ )
            {
#if defined(NW_PLATFORM_WIN32)
                pixels[j + i * height] = colotU8.ToU32LE();
#else
                pixels[j + i * height] = colotU8.ToU32();
#endif
            }
        }

#if defined(NW_PLATFORM_WIN32)
        glGenTextures( 1, &mTextutreID );
        glBindTexture( GL_TEXTURE_2D, mTextutreID );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, width, height, 0, GL_RGBA, GL_UNSIGNED_BYTE, pixels );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );

        allocator->Free( pixels );
#endif
#if defined(NW_PLATFORM_CAFE)
        memset( &mGx2Texture, 0, sizeof(GX2Texture) );
        GX2InitTexture( &mGx2Texture, width, height, 1, 0, GX2_SURFACE_FORMAT_TCS_R8_G8_B8_A8_UNORM, GX2_SURFACE_DIM_2D );
        u32* texturePtrs = (u32*)allocator->Alloc( mGx2Texture.surface.imageSize, mGx2Texture.surface.alignment );
        GX2InitTexturePtrs( &mGx2Texture, texturePtrs, 0 );
        GX2TileTexture( &mGx2Texture, (void*)pixels );
#endif
        mInitialized = true;
        mAllocator   = allocator;
    }
};


//---------------------------------------------------------------------------
//! @brief      フレームバッファテクスチャクラス
//---------------------------------------------------------------------------
class FrameBufferTexture : public Texture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    FrameBufferTexture()
    {
        mInitialized = false;
        mFrameBuffer = NULL;
    }

    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャを作成します。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator, FrameBuffer* frameBuffer, FrameBuffer::FrameBufferType type = FrameBuffer::FRAMEBUFFER_TYPE_NONE )
    {
#if defined(NW_PLATFORM_WIN32)
        NW_UNUSED_VARIABLE( allocator );
        NW_UNUSED_VARIABLE( type );

        glGenTextures( 1, &mTextutreID );
        glBindTexture( GL_TEXTURE_2D, mTextutreID );
        glTexImage2D(   GL_TEXTURE_2D,
                        0,
                        GL_RGBA,
                        static_cast<GLsizei>( frameBuffer->GetSize().x ),
                        static_cast<GLsizei>( frameBuffer->GetSize().y ),
                        0,
                        GL_RGBA,
                        GL_UNSIGNED_BYTE,
                        NULL );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );

        NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)
        mFrameBuffer = frameBuffer;

        GX2ColorBuffer* colorBuffer = mFrameBuffer->GetGX2ColorBuffer();

        GX2SurfaceFormat format;
        if ( type == FrameBuffer::FRAMEBUFFER_TYPE_NONE )
        {
            format = colorBuffer->surface.format;
        }
        else
        {
            format = static_cast<GX2SurfaceFormat>( type );
        }

        // カラーバッファテクスチャ
        GX2InitTexture( &mGx2Texture,
            colorBuffer->surface.width,
            colorBuffer->surface.height,
            colorBuffer->surface.depth,
            colorBuffer->surface.numMips,
            format,
            colorBuffer->surface.dim );

        mGx2Texture.surface.tileMode = colorBuffer->surface.tileMode;
        GX2CalcSurfaceSizeAndAlignment( &mGx2Texture.surface );
        GX2InitTextureRegs( &mGx2Texture );
        GX2InitTexturePtrs( &mGx2Texture, allocator->Alloc( mGx2Texture.surface.imageSize, mGx2Texture.surface.alignment ), 0 );
#endif
        mSize.x      = frameBuffer->GetSize().x;
        mSize.y      = frameBuffer->GetSize().y;
        mInitialized = true;
        mAllocator   = allocator;
    }


    //---------------------------------------------------------------------------
    //! @brief        フレームバッファテクスチャを作成します。
    //---------------------------------------------------------------------------
    void Initialize(    nw::ut::MemoryAllocator* allocator,
                        FrameBuffer* frameBuffer,
                        f32 width, f32 height,
                        FrameBuffer::FrameBufferType type = FrameBuffer::FRAMEBUFFER_TYPE_NONE )
    {
#if defined(NW_PLATFORM_WIN32)
        NW_UNUSED_VARIABLE( allocator );
        NW_UNUSED_VARIABLE( frameBuffer );
        NW_UNUSED_VARIABLE( type );

        glGenTextures( 1, &mTextutreID );
        glBindTexture( GL_TEXTURE_2D, mTextutreID );
        glTexImage2D(   GL_TEXTURE_2D,
            0,
            GL_RGBA,
            static_cast<GLsizei>( width ),
            static_cast<GLsizei>( height ),
            0,
            GL_RGBA,
            GL_UNSIGNED_BYTE,
            NULL );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );
        NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)
        mFrameBuffer = frameBuffer;

        GX2ColorBuffer* colorBuffer = mFrameBuffer->GetGX2ColorBuffer();

        GX2SurfaceFormat format;
        if ( type == FrameBuffer::FRAMEBUFFER_TYPE_NONE )
        {
            format = colorBuffer->surface.format;
        }
        else
        {
            format = static_cast<GX2SurfaceFormat>( type );
        }

        // カラーバッファテクスチャ
        GX2InitTexture( &mGx2Texture,
            width,
            height,
            colorBuffer->surface.depth,
            colorBuffer->surface.numMips,
            format,
            colorBuffer->surface.dim );

        mGx2Texture.surface.tileMode = colorBuffer->surface.tileMode;
        GX2CalcSurfaceSizeAndAlignment( &mGx2Texture.surface );
        GX2InitTextureRegs( &mGx2Texture );
        GX2InitTexturePtrs( &mGx2Texture, allocator->Alloc( mGx2Texture.surface.imageSize, mGx2Texture.surface.alignment ), 0 );
#endif
        mSize.x      = width;
        mSize.y      = height;
        mInitialized = true;
        mAllocator   = allocator;
    }

private:
    FrameBuffer*        mFrameBuffer;
};


//---------------------------------------------------------------------------
//! @brief      デプスバッファテクスチャクラス
//---------------------------------------------------------------------------
class DepthBufferTexture : public Texture
{
public:
    //---------------------------------------------------------------------------
    //! @brief        コンストラクタです。
    //---------------------------------------------------------------------------
    DepthBufferTexture()
    {
        mInitialized = false;
    }


    //---------------------------------------------------------------------------
    //! @brief        デプスバッファテクスチャを作成します。
    //---------------------------------------------------------------------------
    void Initialize( nw::ut::MemoryAllocator* allocator, FrameBuffer* frameBuffer )
    {
#if defined(NW_PLATFORM_WIN32)
        NW_UNUSED_VARIABLE( allocator );

        glGenTextures( 1, &mTextutreID );
        glBindTexture( GL_TEXTURE_2D, mTextutreID );
        glTexImage2D(   GL_TEXTURE_2D,
            0,
            GL_DEPTH_COMPONENT32,
            static_cast<GLsizei>( frameBuffer->GetSize().x ),
            static_cast<GLsizei>( frameBuffer->GetSize().y ),
            0,
            GL_DEPTH_COMPONENT,
            GL_UNSIGNED_BYTE,
            NULL );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP );
        NW_GL_ASSERT();
#endif
#if defined(NW_PLATFORM_CAFE)
        GX2DepthBuffer* depthBuffer = frameBuffer->GetGX2DepthBuffer();

        GX2InitTexture( &mGx2Texture,
            depthBuffer->surface.width,
            depthBuffer->surface.height,
            1,
            0,
            GX2_SURFACE_FORMAT_TCD_R32_FLOAT,
            depthBuffer->surface.dim );

        mGx2Texture.surface.use        = GX2_SURFACE_USE_TEXTURE;
        mGx2Texture.compSel            = GX2_COMP_SEL_XXXX;

        GX2InitTextureRegs( &mGx2Texture );
        GX2InitTexturePtrs( &mGx2Texture, allocator->Alloc( mGx2Texture.surface.imageSize, mGx2Texture.surface.alignment ), 0 );
#endif

        mSize.x = frameBuffer->GetSize().x;
        mSize.y = frameBuffer->GetSize().y;
        mInitialized = true;
    }
};


} // namespace eftdemo
} // namespace nw
