﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <eftdemo_CubeMap.h>
#include <eftdemo_Texture.h>

namespace nw      {
namespace eftdemo {

//! キューブマップ生成に使う look at, look up ベクトル
#if defined(NW_PLATFORM_WIN32)
static const nw::math::VEC3 VIEW_VEC[6][2] =
{
    { nw::math::VEC3(  1.0f,  0.0f,  0.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // +X
    { nw::math::VEC3( -1.0f,  0.0f,  0.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // -X
    { nw::math::VEC3(  0.0f,  1.0f,  0.0f ), nw::math::VEC3( 0.0f,  0.0f,  1.0f ) },  // +Y
    { nw::math::VEC3(  0.0f, -1.0f,  0.0f ), nw::math::VEC3( 0.0f,  0.0f, -1.0f ) },  // -Y
    { nw::math::VEC3(  0.0f,  0.0f,  1.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // +Z
    { nw::math::VEC3(  0.0f,  0.0f, -1.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) }   // -Z
};
#endif
#if defined(NW_PLATFORM_CAFE)
static const nw::math::VEC3 VIEW_VEC[6][2] =
{
    { nw::math::VEC3(  1.0f,  0.0f,  0.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // +X
    { nw::math::VEC3( -1.0f,  0.0f,  0.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // -X
    { nw::math::VEC3(  0.0f, -1.0f,  0.0f ), nw::math::VEC3( 0.0f,  0.0f, -1.0f ) },  // -Y
    { nw::math::VEC3(  0.0f,  1.0f,  0.0f ), nw::math::VEC3( 0.0f,  0.0f,  1.0f ) },  // +Y
    { nw::math::VEC3(  0.0f,  0.0f,  1.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) },  // +Z
    { nw::math::VEC3(  0.0f,  0.0f, -1.0f ), nw::math::VEC3( 0.0f, -1.0f,  0.0f ) }   // -Z
};
#endif

//! テクスチャスライス毎のスウィズル
static const u32 CM_SWIZZLE[6] = { 0, 2, 4, 6, 0, 2 };
static const u32 CM_SWIZZLE_8x8[6] = { 0, 2, 0, 0, 0, 2 };

#if 0
//! キューブマップ生成時の背景色
static const f32 CLEAR_COLOR[6][4] =
{
    { 0.0f, 0.0f, 0.0f, 1.0f },
    { 0.0f, 1.0f, 0.0f, 1.0f },
    { 0.0f, 0.0f, 1.0f, 1.0f },
    { 1.0f, 0.0f, 1.0f, 1.0f },
    { 1.0f, 0.0f, 0.0f, 1.0f },
    { 1.0f, 1.0f, 1.0f, 1.0f }
};
#endif


//---------------------------------------------------------------------------
//! @brief        コンストラクタです。
//---------------------------------------------------------------------------
CubeMap::CubeMap()
{
}


//---------------------------------------------------------------------------
//! @brief        初期化処理です。
//!
//! @param[in] allocator フレームバッファに使用するアロケータ。
//! @param[in] width     キューブマップテクスチャの横幅。
//! @param[in] height    キューブマップテクスチャの縦幅。
//! @param[in] drawFunc  キューブマップテクスチャの描画関数
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
void CubeMap::Initialize( nw::ut::MemoryAllocator* allocator, f32 width, f32 height, void (*drawFunc)(EffectDemo*, DrawParam&, SimpleShader::ViewId) )
{
    mWidth     = width;
    mHeight    = height;
    m_drawFunc = drawFunc;

    // カラーバッファを作成
    {
        glGenTextures( 1, &mCmColor );
        glBindTexture( GL_TEXTURE_CUBE_MAP, mCmColor );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_CUBE_MAP, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE );

        for ( u32 i = 0; i < 6; ++i )
        {
            glTexImage2D( GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, 0, GL_RGBA, (GLsizei)mWidth, (GLsizei)mHeight, 0, GL_BGRA, GL_UNSIGNED_BYTE, 0 );
        }

        NW_GL_ASSERT();
    }

    // デプスバッファを作成
    {
        glGenRenderbuffersEXT( 1, &mCmDepth );
        glBindRenderbufferEXT( GL_RENDERBUFFER_EXT, mCmDepth );
        glRenderbufferStorageEXT( GL_RENDERBUFFER_EXT, GL_DEPTH_COMPONENT, (GLsizei)mWidth, (GLsizei)mHeight );

        glBindRenderbufferEXT( GL_RENDERBUFFER_EXT, 0 );

        NW_GL_ASSERT();
    }

    // フレームバッファオブジェクトを作成
    {
        glGenFramebuffersEXT( 1, &mCmFbo );
        glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, mCmFbo );

        //glFramebufferRenderbufferEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, mCmColor, 0 );
        //glFramebufferRenderbufferEXT( GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, mCmDepth );

        glFramebufferTexture2DEXT(GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_CUBE_MAP_POSITIVE_X, mCmColor, 0 );
        glFramebufferRenderbufferEXT( GL_FRAMEBUFFER_EXT, GL_DEPTH_ATTACHMENT_EXT, GL_RENDERBUFFER_EXT, mCmDepth );

        NW_GL_ASSERT();
    }

    // 確認用のカラーバッファを作成
    for ( u32 i = 0; i < 6; ++i )
    {
        glGenTextures( 1, &mCmColorArray[i] );
        glBindTexture( GL_TEXTURE_2D, mCmColorArray[i] );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE );
        glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_R, GL_CLAMP_TO_EDGE );
        glTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, (GLsizei)mWidth, (GLsizei)mHeight, 0, GL_BGRA, GL_UNSIGNED_BYTE, 0 );

        NW_GL_ASSERT();
    }
}
#endif

#if defined(NW_PLATFORM_CAFE)
void CubeMap::Initialize( nw::ut::MemoryAllocator* allocator, f32 width, f32 height, void (*drawFunc)(EffectDemo*, DrawParam&, SimpleShader::ViewId) )
{
    mWidth     = width;
    mHeight    = height;
    m_drawFunc = drawFunc;

    const u32* swizzle;
    if ( width == 8.0f && height == 8.0f )
    {
        swizzle = CM_SWIZZLE_8x8;
    }
    else
    {
        swizzle = CM_SWIZZLE;
    }

    // テクスチャフォーマット
    GX2SurfaceFormat format  = GX2_SURFACE_FORMAT_TC_R8_G8_B8_A8_SNORM ;
    GX2AAMode aamode = GX2_AA_MODE_1X;

    // カラーバッファを作成
    {
        DEMOGfxInitColorBufferEx( &mCmColorCube, GX2_SURFACE_DIM_CUBE, mWidth, mHeight, 6, format, aamode );
        GX2InitColorBufferPtr( &mCmColorCube, allocator->Alloc( mCmColorCube.surface.imageSize, mCmColorCube.surface.alignment ) );

        GX2Invalidate( GX2_INVALIDATE_CPU, mCmColorCube.surface.imagePtr, mCmColorCube.surface.imageSize );
    }

    // カラーバッファをスライス毎に分割
    for ( u32 i = 0; i < 6; ++i )
    {
        GX2InitColorBuffer( &mCmColorArray[i], mWidth, mHeight, format, aamode );
        GX2SetSurfaceSwizzle( &mCmColorArray[i].surface, swizzle[i] );
        void* ptr = (u8*)mCmColorCube.surface.imagePtr + mCmColorCube.surface.imageSize / 6 * i;
        GX2InitColorBufferPtr( &mCmColorArray[i], ptr );
    }

    // テクスチャを作成
    GX2InitTexture( &mCmTexture, mWidth, mHeight, 6, 0, format, GX2_SURFACE_DIM_CUBE );
    GX2InitTexturePtrs( &mCmTexture, mCmColorCube.surface.imagePtr, 0 );

    for ( u32 i = 0; i < 6; ++i )
    {
        GX2InitTexture( &mCmTextureArray[i], mWidth, mHeight, 1, 0, format, GX2_SURFACE_DIM_2D );
        GX2SetSurfaceSwizzle( &mCmTextureArray[i].surface, swizzle[i] );
        GX2InitTexturePtrs( &mCmTextureArray[i], mCmColorArray[i].surface.imagePtr, 0 );
    }

    // デプスバッファを作成
    {
        GX2InitDepthBuffer( &mCmDepth, mWidth, mHeight, GX2_SURFACE_FORMAT_TCD_R32_FLOAT, aamode );
        GX2InitDepthBufferPtr( &mCmDepth, allocator->Alloc( mCmDepth.surface.imageSize, mCmDepth.surface.alignment ) );

        GX2Invalidate( GX2_INVALIDATE_CPU, mCmDepth.surface.imagePtr, mCmDepth.surface.imageSize );
    }
}
#endif


//---------------------------------------------------------------------------
//! @brief        描画処理です。
//---------------------------------------------------------------------------
#if defined(NW_PLATFORM_WIN32)
void CubeMap::Draw( EffectDemo* effectDemo, const DrawParam& drawParam )
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();

    graphics->SetBlendEnable( true );
    graphics->SetDepthEnable( true, true );
    graphics->SetAlphaTestEnable( false );

    nw::eftdemo::DrawParam copyParam = drawParam;

    // プロジェクション行列を作成
    nw::math::MTX44 projMtx;
    nw::math::MTX44PerspectiveRadNew( &copyParam.mProjMtx, NW_MATH_DEG_TO_RAD(90.0f), 1.0f, 1.0f, 10000.0f );

    glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, mCmFbo );

    for ( u32 i = 0; i < 6; ++i )
    {
        glFramebufferTexture2DEXT( GL_FRAMEBUFFER_EXT, GL_COLOR_ATTACHMENT0_EXT, GL_TEXTURE_CUBE_MAP_POSITIVE_X + i, mCmColor, 0 );

        // ビュー行列を作成
        nw::math::VEC3 target = mViewPos + VIEW_VEC[i][0];
        nw::math::MTX34LookAt( &copyParam.mViewMtx, &mViewPos, &VIEW_VEC[i][1], &target );

        glViewport( 0, 0, (GLsizei)mWidth, (GLsizei)mHeight );
#if 0
        glClearColor( CLEAR_COLOR[i][0], CLEAR_COLOR[i][1], CLEAR_COLOR[i][2], CLEAR_COLOR[i][3] );
#else
        glClearColor( mBgColor.x, mBgColor.y, mBgColor.z, mBgColor.w );
#endif
        glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

        SimpleShader::ViewId id = static_cast<SimpleShader::ViewId>(SimpleShader::VIEW_ID_CUBEMAP_PX + i);
        m_drawFunc( effectDemo, copyParam, id );

        // フレームバッファを確認用テクスチャにコピー
        glBindTexture( GL_TEXTURE_2D, mCmColorArray[i] );
        glCopyTexImage2D( GL_TEXTURE_2D, 0, GL_RGBA, 0, 0, (GLsizei)mWidth, (GLsizei)mHeight, 0);

        NW_GL_ASSERT();
    }

    glBindFramebufferEXT( GL_FRAMEBUFFER_EXT, 0 );
}
#endif

#if defined(NW_PLATFORM_CAFE)
void CubeMap::Draw( EffectDemo* effectDemo, const DrawParam& drawParam )
{
    nw::gfnd::Graphics* graphics = nw::gfnd::Graphics::GetInstance();

    graphics->SetBlendEnable( true );
    graphics->SetDepthEnable( true, true );
    graphics->SetAlphaTestEnable( false );

    nw::eftdemo::DrawParam copyParam = drawParam;

    // プロジェクション行列を作成
    nw::math::MTX44Perspective( &copyParam.mProjMtx, NW_MATH_DEG_TO_RAD(90.0f), 1.0f, 0.1f, 1000.f);

    for ( u32 i = 0; i < 6; ++i )
    {
        GX2SetViewport( 0, 0, mWidth, mHeight, 0.0f, 1.0f );
        GX2SetScissor( 0, 0, mWidth, mHeight );

        GX2SetColorBuffer( &mCmColorArray[i], GX2_RENDER_TARGET_0 );
        GX2SetDepthBuffer( &mCmDepth );

#if 0
        GX2ClearColor( &mCmColorArray[i], CLEAR_COLOR[i][0], CLEAR_COLOR[i][1], CLEAR_COLOR[i][2], CLEAR_COLOR[i][3] );
#else
        GX2ClearColor( &mCmColorArray[i], mBgColor.x, mBgColor.y, mBgColor.z, mBgColor.w );
#endif
        GX2ClearDepthStencil( &mCmDepth, GX2_CLEAR_BOTH );

        GX2SetContextState( graphics->GetGX2ContextState() );

        // ビュー行列を作成
        nw::math::VEC3 target = mViewPos + VIEW_VEC[i][0];
        nw::math::MTX34LookAt( &copyParam.mViewMtx, &mViewPos, &VIEW_VEC[i][1], &target );

        SimpleShader::ViewId id = static_cast<SimpleShader::ViewId>(SimpleShader::VIEW_ID_CUBEMAP_PX + i);
        m_drawFunc( effectDemo, copyParam, id);
    }

    GX2Invalidate( GX2_INVALIDATE_COLOR_BUFFER, mCmColorCube.surface.imagePtr, mCmColorCube.surface.imageSize );
    GX2Invalidate( GX2_INVALIDATE_TEXTURE, mCmTexture.surface.imagePtr, mCmTexture.surface.imageSize );
}
#endif


} // namespace nw::eftdemo
} // namespace nw
